// DecoderFactory.h
// 2008/11/27

#pragma once

#include "VorbisDecoder.h"

namespace QAX {

extern CriticalSection g_cs_codec;

// QVorbisDecoder
class QVorbisDecoder {

	QVorbisDecoderSetup_t* m_setup;

	QVorbisDecoder_t* m_decoder;

	enum {
		DECODER_BUFFER = 0x8000
	};

public:

	QVorbisDecoder(QVorbisDecoderSetup_t* s) :
		m_setup(s),
		m_decoder(0)
	{
	}

	~QVorbisDecoder()
	{
		if (m_decoder != 0) {
			QV_ReleaseDecoder(m_decoder);
		}
	}

	QVorbisDecoderSetup_t* GetSetup()
	{
		return m_setup;
	}

	QVorbisDecoder_t* GetDecoder()
	{
		return m_decoder;
	}

	void Setup()
	{
		m_decoder = QV_CreateDecoder();
		if (m_decoder == 0) {
			DecoderError::Throw("QV_CreateDecoder");
		}

		{
			Lock clock(&g_cs_codec);

			if (!QV_SetupDecoder(
				m_decoder,
				m_setup,
				DECODER_BUFFER)) {
				DecoderError::Throw("QV_SetupDecoder");
			}
		}
	}

}; // QVorbisDecoder

// DecoderFactoryImpl
class DecoderFactoryImpl : public QDecoderFactory {

	class Key {
		BYTE m_Key[20];
	public:
		Key(const BYTE k[20])
		{
			CopyMemory(m_Key, k, sizeof(m_Key));
		}

		friend bool operator==(const Key& lhs, const Key& rhs)
		{
			return memcmp(lhs.m_Key, rhs.m_Key, 20) == 0;
		}

		friend bool operator<(const Key& lhs, const Key& rhs)
		{
			return memcmp(lhs.m_Key, rhs.m_Key, 20) < 0;
		}
	};

	CriticalSection m_cs;

	typedef std::map<Key, QVorbisDecoderSetup_t*> Map;

	Map m_Map;

	DecoderFactoryImpl()
	{
	}

public:

	static QDecoderFactory* Create()
	{
		DecoderFactoryImpl* impl = new DecoderFactoryImpl();

		return impl;
	}

	~DecoderFactoryImpl()
	{
		for (Map::iterator it = m_Map.begin(); it != m_Map.end(); ++it) {
			QVorbisDecoderSetup_t* p = (*it).second;
			QV_ReleaseDecoderSetup(p);
		}
	}

	/* */

	virtual void STDCALL Release()
	{
		delete this;
	}

	virtual QVorbisDecoder* STDCALL CreateVorbisDecoder(
		const BYTE  key[20],
		const VOID* id,
		SIZE_T      idSize,
		const VOID* setup,
		SIZE_T      setupSize)
	{
		Lock lock(&m_cs);

		QVorbisDecoder* p = 0;

		Key k(key);

		Map::iterator it = m_Map.find(k);
		if (it != m_Map.end()) {
			p = new QVorbisDecoder((*it).second);

		} else {
			QVorbisDecoderSetup_t* s = QV_CreateDecoderSetup();
			if (s == 0) {
				DecoderError::Throw("QV_CreateDecoderSetup");
			}

			{
				Lock clock(&g_cs_codec);

				if (!QV_SetupDecoderSetup(
					s,
					id,
					idSize,
					setup,
					setupSize)) {
					DecoderError::Throw("QV_SetupDecoderSetup");
				}
			}

			m_Map.insert(
				Map::value_type(k, s));

			p = new QVorbisDecoder(s);
		}

		try {
			p->Setup();

		} catch (...) {
			delete p;
			throw;
		}

		return p;
	}

}; // DecoderFactoryImpl

} // namespace QAX

