// DXSound.h
// 2008/12/04

#pragma once

namespace QAX {

// DirectSound
class DirectSound {

	ATL::CComPtr<IDirectSound8> m_ds;

public:

	DirectSound()
	{
	}

	~DirectSound()
	{
	}

	HRESULT Create(HWND hwnd)
	{
		HRESULT hRslt = DirectSoundCreate8(
			&DSDEVID_DefaultPlayback,
			&m_ds,
			0);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		hRslt = m_ds->SetCooperativeLevel(
			hwnd,
			DSSCL_PRIORITY);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		return S_OK;
	}

	operator IDirectSound8*()
	{
		return m_ds;
	}

}; // DirectSound

/* */

// DSoundBuffer
class DSoundBuffer {

	ATL::CComPtr<IDirectSoundBuffer8> m_sb;

	INT32 m_BufferSize;

	UINT64 m_iPlay;
	UINT64 m_iWrite;

	UINT32 m_LastPlay;

public:

	DSoundBuffer() :
		m_BufferSize(0),
		m_iPlay(0),
		m_iWrite(0),
		m_LastPlay(0)
	{
	}

	~DSoundBuffer()
	{
	}

	HRESULT Create(
		IDirectSound8*      pDS,
		const WAVEFORMATEX* wfex,
		INT32               BufferSize)
	{
		DSBUFFERDESC dsbd  = { 0 };
		dsbd.dwSize        = sizeof(dsbd);
		dsbd.dwFlags       = DSBCAPS_CTRLPAN | DSBCAPS_CTRLVOLUME |
		                     DSBCAPS_STATIC | DSBCAPS_GLOBALFOCUS;
		dsbd.dwBufferBytes = BufferSize;
		dsbd.lpwfxFormat   = const_cast<WAVEFORMATEX*>(wfex);

		ATL::CComPtr<IDirectSoundBuffer> sb;
		HRESULT hRslt = pDS->CreateSoundBuffer(
			&dsbd,
			&sb,
			0);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		hRslt = sb->QueryInterface(
			IID_IDirectSoundBuffer8,
			(void**)&m_sb);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		m_BufferSize = BufferSize;

		return S_OK;
	}

	void Release()
	{
		m_sb.Release();
	}

	operator IDirectSoundBuffer8*()
	{
		return m_sb;
	}

	INT32 GetBufferSize()
	{
		return m_BufferSize;
	}

	void Reset()
	{
		if (m_sb != 0) {
			m_sb->Stop();
			m_sb->SetCurrentPosition(0);
		}

		m_iPlay  = 0;
		m_iWrite = 0;

		m_LastPlay = 0;
	}

	UINT64 Poll()
	{
		DWORD dwPlay  = 0;
		DWORD dwWrite = 0;
		m_sb->GetCurrentPosition(
			&dwPlay,
			&dwWrite);

		if (dwPlay < m_LastPlay) {
			m_iPlay += m_BufferSize;
		}

		m_LastPlay = dwPlay;

		UINT64 pos = m_iPlay + dwPlay;

		return pos;
	}

	UINT32 GetFeedable(
		UINT64 pos)
	{
		UINT32 feedable = UINT32(m_BufferSize - (m_iWrite - pos));
		return feedable;
	}

	HRESULT Lock(
		VOID**  ppvBuffer,
		UINT32* pcbBuffer,
		UINT32  cbSize)
	{
		*ppvBuffer = 0;
		*pcbBuffer = 0;

		INT32 iWrite  = m_iWrite % m_BufferSize;

		VOID* pvPart1 = 0;
		DWORD cbPart1 = 0;

		VOID* pvPart2 = 0;
		DWORD cbPart2 = 0;

		HRESULT hRslt = m_sb->Lock(
			iWrite,
			cbSize,
			&pvPart1,
			&cbPart1,
			&pvPart2,
			&cbPart2,
			0);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		if (pvPart2 != 0) {
			m_sb->Unlock(
				pvPart1,
				cbPart1,
				pvPart2,
				cbPart2);
			return E_FAIL;
		}

		*ppvBuffer = pvPart1;
		*pcbBuffer = cbPart1;

		return S_OK;
	}

	HRESULT Unlock(
		VOID*  pvBuffer,
		UINT32 cbBuffer)
	{
		HRESULT hRslt = m_sb->Unlock(
			pvBuffer,
			cbBuffer,
			0,
			0);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		m_iWrite += cbBuffer;

		return S_OK;
	}

	/* */

	HRESULT Play()
	{
		HRESULT hRslt = m_sb->Play(
			0,
			0,
			DSBPLAY_LOOPING);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		return S_OK;
	}


	HRESULT Stop()
	{
		HRESULT hRslt = m_sb->Stop();
		if (FAILED(hRslt)) {
			return hRslt;
		}

		return S_OK;
	}

	/* */

}; // DSoundBuffer

} // namespace QAX

