// StreamReader.h
// 2008/12/25

#pragma once

namespace QAX {

/* */

// ReadRange
struct ReadRange {

	const VOID* Buffer;
	SIZE_T      Length;

}; // ReadRange

// StreamReader
class StreamReader {

public:

	virtual void Release() = 0;

	virtual void Seek(
		SIZE_T pos) = 0;

	virtual ReadRange Fetch(
		SIZE_T size) = 0;

	virtual void Retire(
		SIZE_T size) = 0;

}; // StreamReader

/* */

// FileMapping_Reader
class FileMapping_Reader : public StreamReader {

	QFileMapping* m_Mapping;

	const VOID* m_View;

	const BYTE* m_Start;
	const BYTE* m_End;

	const BYTE* m_Current;

private:

	FileMapping_Reader(QFileMapping* p) :
		m_Mapping(p),
		m_View(0),
		m_Start(0),
		m_End(0),
		m_Current(0)
	{
	}

	void Setup(
		UINT64 pos,
		SIZE_T size)
	{
		UINT64 base = (pos / g_OFFSET_ALIGN) * g_OFFSET_ALIGN;
		UINT32 diff = UINT32(pos - base);

		m_View = m_Mapping->MapView(
			base,
			SIZE_T(diff + size));

		m_Start = static_cast<const BYTE*>(m_View) + diff;
		m_End   = m_Start + size;

		m_Current = m_Start;
	}

	void SafeRelease()
	{
		m_Mapping->UnmapView(
			m_View);

		m_View = 0;

		m_Start   = 0;
		m_End     = 0;
		m_Current = 0;
	}

public:

	static StreamReader* Create(
		QFileMapping* mapping,
		UINT64        offset,
		SIZE_T        size)
	{
		FileMapping_Reader* p = new FileMapping_Reader(mapping);

		try {
			p->Setup(offset, size);

		} catch (...) {
			delete p;
			throw;
		}

		return p;
	}

	~FileMapping_Reader()
	{
		SafeRelease();
	}

	virtual void Release()
	{
		delete this;
	}

	virtual void Seek(
		SIZE_T pos)
	{
		const BYTE* p = m_Start + pos;
		if (p < m_Start || p >= m_End) {
			throw FormatError("StreamReader.Seek.OutOfRange");
		}

		m_Current = p;
	}

	virtual ReadRange Fetch(
		SIZE_T size)
	{
		ReadRange r;

		r.Buffer = m_Current;
		r.Length = size;

		SIZE_T rest = m_End - m_Current;
		if (r.Length > rest) {
			r.Length = rest;
		}

		return r;
	}

	virtual void Retire(
		SIZE_T size)
	{
		const BYTE* p = m_Current + size;
		if (p < m_Start || p > m_End) {
			throw FormatError("StreamReader.Retire.OutOfRange");
		}

		m_Current = p;
	}

}; // FileMapping_Reader

/* */

// IStream_Reader
class IStream_Reader : public StreamReader {

	IStream* m_p;

	UINT64 m_Offset;

	BYTE*  m_pbBuffer;
	SIZE_T m_cbBuffer;

	BYTE*  m_Current;
	BYTE*  m_End;

	enum {
		ALIGN_SIZE = 0x100,
		INIT_SIZE  = 0x10000
	};

private:

	IStream_Reader(IStream* p) :
		m_p(p),
		m_Offset(0),
		m_pbBuffer(0),
		m_cbBuffer(0),
		m_Current(0),
		m_End(0)
	{
		m_p->AddRef();
	}

	void Setup(UINT64 offset)
	{
		BYTE* p = static_cast<BYTE*>(
			_aligned_malloc(INIT_SIZE, ALIGN_SIZE));
		if (p == 0) {
			throw DecoderError("OutOfMemory");
		}

		m_pbBuffer = p;
		m_cbBuffer = INIT_SIZE;

		m_Current = p;
		m_End     = p;

		m_Offset = offset;
	}

	void SafeRelease()
	{
		_aligned_free(m_pbBuffer);

		m_pbBuffer = 0;
		m_cbBuffer = 0;

		m_Current = 0;
		m_End     = 0;

		if (m_p != 0) {
			m_p->Release();
			m_p = 0;
		}
	}

public:

	static StreamReader* Create(
		IStream* s,
		UINT64   offset)
	{
		IStream_Reader* p = new IStream_Reader(s);

		try {
			p->Setup(offset);

			p->Seek(0);

		} catch (...) {
			delete p;
			throw;
		}

		return p;
	}

	~IStream_Reader()
	{
		SafeRelease();
	}

	virtual void Release()
	{
		delete this;
	}

	virtual void Seek(
		SIZE_T p)
	{
		LARGE_INTEGER pos;
		ULARGE_INTEGER newPos;

		pos.QuadPart = m_Offset + p;
		HRESULT hRslt = m_p->Seek(
			pos,
			STREAM_SEEK_SET,
			&newPos);
		if (FAILED(hRslt)) {
			COMError::Throw("IStream::Seek", hRslt);
		}

		m_Current = m_pbBuffer;
		m_End     = m_pbBuffer;
	}

	virtual ReadRange Fetch(
		SIZE_T size)
	{
		ReadRange r;

		SIZE_T sz = m_End - m_Current;
		if (size <= sz) {
			r.Buffer = m_Current;
			r.Length = size;

		} else {
			sz = m_pbBuffer + m_cbBuffer - m_Current;
			if (size > sz) {
				SIZE_T b = m_End - m_Current;
				memmove(
					m_pbBuffer,
					m_Current,
					b);

				m_Current = m_pbBuffer;
				m_End     = m_pbBuffer + b;

				if (size > m_cbBuffer) {
					SIZE_T newSize = ((size + INIT_SIZE - 1) / INIT_SIZE) * INIT_SIZE;

					VOID* pv = _aligned_realloc(
						m_pbBuffer,
						newSize,
						ALIGN_SIZE);
					if (pv == 0) {
						throw DecoderError("OutOfMemory");
					}

					m_pbBuffer = static_cast<BYTE*>(pv);
					m_cbBuffer = newSize;

					m_Current = m_pbBuffer;
					m_End     = m_pbBuffer + b;
				}
			}

			sz = size - (m_End - m_Current);

			ULONG cb = 0;
			HRESULT hRslt = m_p->Read(
				m_End,
				ULONG(sz),
				&cb);
			if (FAILED(hRslt)) {
				COMError::Throw("IStream::Read", hRslt);
			}

			m_End += cb;

			r.Buffer = m_Current;
			r.Length = m_End - m_Current;
		}

		return r;
	}

	virtual void Retire(
		SIZE_T size)
	{
		BYTE* p = m_Current + size;
		if (p > m_End) {
			throw FormatError("StreamReader.Retire.OutOfRange");
		}

		m_Current = p;
	}

}; // IStream_Reader

/* */

} // namespace QAX

