#
# $Id: Ipmsg::Net.pm,v 0.17 2004/02/23 Mikio Fukushima Exp Mikio Fukushima $
#

package Ipmsg::Net;
use Ipmsg::Common;
use Ipmsg::Message::Recv;
use Ipmsg::Message::Send;

use IO::Socket;

use strict;

use vars qw( $VERSION $THISNAME
	@ISA @EXPORT @EXPORT_OK
);

require Exporter;
require AutoLoader;

@ISA	= qw( Exporter AutoLoader Ipmsg::Common );
@EXPORT	= qw();

# --------------------
# Set global variables in this module.
#

	$VERSION	= 0.17;
	$THISNAME	= "Ipmsg Net Ver.$VERSION";

	use constant Code			=> 0;

	use constant BindAddr		=> 1;
	use constant LocalPort		=> 2;

	use constant Verbose		=> 3;
	use constant UseLocalPort	=> 4;
	use constant DirectBroadcast=> 5;

	use constant TIMEOUT_RECV	=> 6;

	use constant User			=> 7;
	use constant Host			=> 8;

	use constant PacketN		=> 9;
	use constant PacketSEND		=> 10;
	use constant PacketRECV		=> 11;
	use constant PacketERR		=> 12;
	use constant PacketRJCT		=> 13;

	use constant SendQueue		=> 14;

	use constant Socket			=> 15;

	use constant LocalAddr		=> 16;
	use constant AllowAddr		=> 17;
	use constant DenyAddr		=> 18;

	my %_index = (
		Code		=> Code,

		BindAddr		=> BindAddr,
		LocalPort		=> LocalPort,

		Verbose			=> Verbose,
		UseLocalPort	=> UseLocalPort,
		DirectBroadcast	=> DirectBroadcast,

		TIMEOUT_RECV	=> TIMEOUT_RECV,

		User			=> User,
		Host			=> Host,

		PacketN			=> PacketN,
		PacketSEND		=> PacketSEND,
		PacketRECV		=> PacketRECV,
		PacketERR		=> PacketERR,
		PacketRJCT		=> PacketRJCT,

		SendQueue		=> SendQueue,

		Socket			=> Socket,

		LocalAddr		=> LocalAddr,
		AllowAddr		=> AllowAddr,
		DenyAddr		=> DenyAddr,
	);

# --------------------
# Category: Constructor

# --------------------
# Name: new

sub new
{
	my $class = shift;
	   $class = ref( $class ) || $class;

	my %arg = @_;
	my $self = [];

	# --------------------
	# initialize

	@{ $self }[ values %_index ] = @arg{ keys %_index };

	# --------------------
	# set default

	$self->[ Code ]			= 1;

	return undef if ! defined $self->[ LocalPort ];
	$self->[ BindAddr ]		= inet_ntoa( INADDR_ANY ) if ! defined $arg{ BindAddr };

	$self->[ Verbose ]			= 0 if ! defined $arg{ Verbose };
	$self->[ UseLocalPort ]		= 0 if ! defined $arg{ UseLocalPort };
	$self->[ DirectBroadcast ]	= 1 if ! defined $arg{ DirectBroadcast };

	$self->[ TIMEOUT_RECV ]	= $class->SUPER::timeout_recv	if ! defined $arg{ TIMEOUT_RECV };

	$self->[ User ]			= getlogin					if ! length $arg{ User };
	$self->[ User ]			= $ENV{ 'USERNAME' }		if ! length $arg{ User };
	return undef										if ! length $self->[ User ];

	$self->[ Host ]			= $ENV{ 'HOSTNAME' }		if ! length $arg{ Host };
	$self->[ Host ]			= $ENV{ 'COMPUTERNAME' }	if ! length $arg{ Host };
	return undef										if ! length $self->[ Host ];

	$self->[ PacketN ]		= time() if ! defined $arg{ PacketN };
	$self->[ PacketSEND ]	= 0;
	$self->[ PacketRECV ]	= 0;
	$self->[ PacketERR ]	= 0;
	$self->[ PacketRJCT ]	= 0;

	$self->[ SendQueue ]	= [];

	$self->[ Socket ]		= '';

	$self->[ LocalAddr ]	= {};
	$self->[ AllowAddr ]	= {};
	$self->[ DenyAddr ]		= {};

	# --------------------
	# bless

	bless $self, $class;

	# --------------------
	# init

	$self->[ Socket ]	= $self->_open( BindAddr => $arg{ BindAddr } );
	return undef if ! $self->[ Socket ];

	$self->_lengthcheck;

	$self->add_localaddr( PeerAddr => $arg{ BindAddr }, PeerPort => $self->[ LocalPort ] ) if $arg{ BindAddr } and $arg{ BindAddr } ne inet_ntoa( INADDR_ANY );

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: flush

sub flush
{
	my ( $self ) = shift;

	$self->[ PacketN ]		= time();
	$self->[ PacketSEND ]	= 0;
	$self->[ PacketRECV ]	= 0;
	$self->[ PacketERR ]	= 0;
	$self->[ PacketRJCT ]	= 0;

	$self->[ SendQueue ]	= [];

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: _open

sub _open
{
	my ( $self, %options ) = @_;

	# --------------------
	# initialize

	$self->[ Code ] = undef;

	# --------------------
	# check requirements

	if( ! defined $options{ BindAddr } ){
		$options{ BindAddr } = inet_ntoa( INADDR_ANY );
	}

	# --------------------
	# Open

	my $handle = new IO::Socket::INET(
		LocalAddr	=> $options{ BindAddr },
		LocalPort	=> $self->[ LocalPort ],
		Proto		=> "udp",
		Type		=> SOCK_DGRAM,
		Reuse		=> 0,
		Broadcast	=> 1,
		MultiHomed	=> 1,
	) or return undef;

	$handle->sockopt( SO_SNDBUF,    int( $self->sockbuf ) ) or return undef;
	$handle->sockopt( SO_RCVBUF,    int( $self->sockbuf ) ) or return undef;
	$handle->sockopt( SO_BROADCAST, int( $self->directbroadcast ) ) or return undef;

	select( $handle ); $|=1; select( STDOUT );

	# --------------------
	# return

	return $handle;
}

# --------------------
# Sub1: DESTROY

sub DESTROY
{
	my $self = shift;

	# --------------------
	# close socket

	$self->close;
}

# --------------------
# Category: Access methods
#

# --------------------
# Name: ܾ

# --------------------
# Sub1: addr

sub addr
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ BindAddr ];
}

# --------------------
# Sub1: port

sub port
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ LocalPort ];
}

# --------------------
# Sub1: user

sub user
{
	my ( $self, $options ) = @_;

	# --------------------
	# update hostname

	if( defined $options ){

		$self->[ User ] = $options;
		$self->_lengthcheck;

	}

	# --------------------
	# return

	return $self->[ User ];
}

# --------------------
# Sub1: host

sub host
{
	my ( $self, $options ) = @_;

	# --------------------
	# update hostname

	if( defined $options ){

		$self->[ Host ] = $options;
		$self->_lengthcheck;

	}

	# --------------------
	# return

	return $self->[ Host ];
}

# --------------------
# Sub1: socket

sub socket
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ Socket ];
}

# --------------------
# Sub1: uselocalport

sub uselocalport
{
	my $self = shift;

	# --------------------
	# return

	@_	? $self->[ UseLocalPort ] = shift
		: $self->[ UseLocalPort ];
}

# --------------------
# Sub1: directbroadcast

sub directbroadcast
{
	my ( $self, $option ) = @_;

	# --------------------
	# return

	if( defined $option ){
		$self->[ DirectBroadcast ] = $option;

		if( $self->[ Socket ] ){
			$self->[ Socket ]->sockopt( SO_BROADCAST, $self->[ DirectBroadcast ] ) or return undef;
		}

	}else{
		return $self->[ DirectBroadcast ];
	}

}

# --------------------
# Sub1: timeout_recv

sub timeout_recv
{
	my $self = shift;

	# --------------------
	# return

	@_	? $self->[ TIMEOUT_RECV ] = shift
		: $self->[ TIMEOUT_RECV ];
}

# --------------------
# Sub1: verbose

sub verbose
{
	my $self = shift;

	# --------------------
	# return

	@_	? $self->[ Verbose ] = shift
		: $self->[ Verbose ];
}

# --------------------
# Sub1: _lengthcheck

sub _lengthcheck
{
	my $self = shift;

	# --------------------
	# chop over $MAX_NAMEBUF( 50bytes ) strings.

	$self->[ User ] = substr( $self->[ User ], 0, $self->SUPER::namebuf );
	$self->[ Host ] = substr( $self->[ Host ], 0, $self->SUPER::namebuf );

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: packet

# --------------------
# Sub1: packetn

sub packetn
{
	my ( $self, $plus ) = @_;

	# --------------------
	# add PacketN

	$self->[ PacketN ]++ if $plus;

	# --------------------
	# return

	return $self->[ PacketN ];
}

# --------------------
# Sub1: packetsend

sub packetsend
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ PacketSEND ];
}

# --------------------
# Sub1: packetrecv

sub packetrecv
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ PacketRECV ];
}

# --------------------
# Sub1: packeterr

sub packeterr
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ PacketERR ];
}

# --------------------
# Sub1: packetrjct

sub packetrjct
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ PacketRJCT ];
}

# --------------------
# Name: Local / Allow / DenyAddr

# --------------------
# Sub1: add_localaddr

sub add_localaddr
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! $options{ PeerAddr };
	if( ! $options{ PeerPort } ){ $options{ PeerPort } = $self->port; }

	# --------------------
	# push hash

	$self->[ LocalAddr ]{ "$options{ PeerAddr }:$options{ PeerPort }" } = [ $options{ PeerAddr }, $options{ PeerPort } ];

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: get_localaddr

sub get_localaddr
{
	my ( $self, %options ) = @_;
	my @tmp;

	# --------------------
	# return w/options

	if( $options{ PeerAddr } and $options{ PeerPort } ){

		return $self->[ LocalAddr ]{ "$options{ PeerAddr }:$options{ PeerPort }" };

	}

	# --------------------
	# return

	if( ! wantarray ){

		return scalar values %{ $self->[ LocalAddr ] };

	}else{

		foreach my $value ( values %{ $self->[ LocalAddr ] } ){ push @tmp, [ $value->[ 0 ], $value->[ 1 ] ] };
		return @tmp;

	}

}

# --------------------
# Sub1: del_localaddr

sub del_localaddr
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! $options{ PeerAddr };
	if( ! $options{ PeerPort } ){ $options{ PeerPort } = $self->port; }

	# --------------------
	# push hash

	delete $self->[ LocalAddr ]{ "$options{ PeerAddr }:$options{ PeerPort }" };

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: add_allowaddr

sub add_allowaddr
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! $options{ PeerNet };
	if( ! $options{ NetMask } ){ $options{ NetMask } = '255.255.255.0'; }

	# --------------------
	# push hash

	$self->[ AllowAddr ]{ "$options{ PeerNet }:$options{ NetMask }:$options{ PeerPort }" } =
		[ $options{ PeerNet }, $options{ NetMask }, $options{ PeerPort } ];

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: get_allowaddr

sub get_allowaddr
{
	my ( $self ) = shift;
	my @tmp;

	# --------------------
	# return

	if( ! wantarray ){

		return scalar values %{ $self->[ AllowAddr ] };

	}else{

		foreach my $value ( values %{ $self->[ AllowAddr ] } ){
			push @tmp, [ $value->[ 0 ], $value->[ 1 ], $value->[ 2 ] ];
		}

		return @tmp;

	}

}

# --------------------
# Sub1: del_allowaddr

sub del_allowaddr
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! $options{ PeerNet };
	if( ! $options{ NetMask } ){ $options{ NetMask } = '255.255.255.0'; }

	# --------------------
	# push hash

	delete $self->[ AllowAddr ]{ "$options{ PeerNet }:$options{ NetMask }:$options{ PeerPort }" };

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: add_denyaddr

sub add_denyaddr
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! $options{ PeerNet };
	if( ! $options{ NetMask } ){ $options{ NetMask } = '255.255.255.0'; }

	# --------------------
	# push hash

	$self->[ DenyAddr ]{ "$options{ PeerNet }:$options{ NetMask }:$options{ PeerPort }" } =
		[ $options{ PeerNet }, $options{ NetMask }, $options{ PeerPort } ];

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: get_denyaddr

sub get_denyaddr
{
	my ( $self ) = shift;
	my @tmp;

	# --------------------
	# return

	if( ! wantarray ){

		return scalar values %{ $self->[ DenyAddr ] };

	}else{

		foreach my $value ( values %{ $self->[ DenyAddr ] } ){
			push @tmp, [ $value->[ 0 ], $value->[ 1 ], $value->[ 2 ] ];
		}

		return @tmp;

	}

}

# --------------------
# Sub1: del_denyaddr

sub del_denyaddr
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	return undef if ! $options{ PeerNet };
	if( ! $options{ NetMask } ){ $options{ NetMask } = '255.255.255.0'; }

	# --------------------
	# push hash

	delete $self->[ DenyAddr ]{ "$options{ PeerNet }:$options{ NetMask }:$options{ PeerPort }" };

	# --------------------
	# return

	return $self;
}

# --------------------
# Category: Methods

# --------------------
# Name: close

sub close
{
	my $self = shift;

	$self->socket->close if ref( $self->socket ) eq 'IO::Socket::INET';

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: recv

sub recv
{
	my ( $self, %options ) = @_;
	my ( $rout );

	my $recv = undef;	# Ipmsg::Message::Recv Object

	# --------------------
	# set code

	$self->[ Code ] = undef;

	# --------------------
	# check requirements

	$options{ FROMCHECK }	= 0 if ! defined $options{ FROMCHECK };
	$options{ SELFACPT }	= 0 if ! defined $options{ SELFACPT };

	# --------------------
	# initialize

	my $rbit	= $self->_set_bits( $self->socket );
	my $ret		= select( $rout = $rbit, undef, undef, $self->timeout_recv );

	# --------------------
	# receive

	if ( vec( $rout, fileno( $self->socket ), 1) ){

		if( my $sockaddr = recv( $self->socket, my $message, $self->sockbuf, 0 ) ){

			my ( $PeerPort, $PeerAddr ) = sockaddr_in( $sockaddr );
				 $PeerAddr = inet_ntoa( $PeerAddr );

			# --------------------
			# Count up recv

			$self->[ PacketRECV ]++;

			# --------------------
			# check source address

			if( ! $options{ SELFACPT } ){

				if( $self->get_localaddr( PeerAddr => $PeerAddr, PeerPort => $PeerPort ) ){

					# --------------------
					# set code

					$self->[ Code ] = 0;

					# --------------------
					# return

					return 0;

				}

			} # End of SELFACPT

			if( $options{ FROMCHECK } ){

				if( ! $self->_check_fromaddress( $PeerAddr, $PeerPort ) ){

					$self->[ PacketRJCT ]++;

					# --------------------
					# set code

					$self->[ Code ] = 0;

					# --------------------
					# return

					return 0;

				}

			} # End of FROMCHECK

			# --------------------
			# make Message::Recv object

			$recv = Ipmsg::Message::Recv->new(
				%{ $self->_split_msg( $sockaddr, $message ) },
				NetIF => $self,
			);

			# --------------------
			# Verbose

			$recv->verbose if $recv and $self->verbose;

			# --------------------
			# set code

			$self->[ Code ] = 1;

		} # End of if recv

	} # End of if vec

	# --------------------
	# return

	return $recv;
}

# --------------------
# Sub1: _check_fromaddress

sub _check_fromaddress
{
	my ( $self, $PeerAddr, $PeerPort ) = @_;

	# --------------------
	# check require

	return undef if ! $PeerAddr;
	return undef if ! $PeerPort;

	# --------------------
	# check allow address

	foreach my $i ( $self->get_allowaddr ){

		# --------------------
		# from port specified

		if( $i->[ 2 ] ){
			return 1 if (
							_ipaddr_rangecheck( PeerNet => $i->[ 0 ], NetMask => $i->[ 1 ], PeerAddr => $PeerAddr )
							&&
							$i->[ 2 ] == $PeerPort
						);

		# --------------------
		# from port not specified

		}else{
			return 1 if _ipaddr_rangecheck( PeerNet => $i->[ 0 ], NetMask => $i->[ 1 ], PeerAddr => $PeerAddr );
		}

	}

	# --------------------
	# check deny address

	foreach my $i ( $self->get_denyaddr ){

		# --------------------
		# from port specified

		if( $i->[ 2 ] ){
			return 0 if (
							_ipaddr_rangecheck( PeerNet => $i->[ 0 ], NetMask => $i->[ 1 ], PeerAddr => $PeerAddr )
							&&
							$i->[ 2 ] == $PeerPort
						);

		# --------------------
		# from port not specified

		}else{
			return 0 if _ipaddr_rangecheck( PeerNet => $i->[ 0 ], NetMask => $i->[ 1 ], PeerAddr => $PeerAddr );
		}

	}

	# --------------------
	# return

	return 1;

}

# --------------------
# Sub1: _ipaddr_rangecheck

sub _ipaddr_rangecheck
{
	my ( %options ) = @_;
	my $i;

	my @network	= split( /\./, $options{ PeerNet } );
	my @netmask	= split( /\./, $options{ NetMask } );
	my @target	= split( /\./, $options{ PeerAddr } );
	my @tmp;

	$tmp[ 0 ] = ( 255 ^ int( $netmask[ 0 ] ) ) | int( $network[ 0 ] );
	$tmp[ 1 ] = ( 255 ^ int( $netmask[ 1 ] ) ) | int( $network[ 1 ] );
	$tmp[ 2 ] = ( 255 ^ int( $netmask[ 2 ] ) ) | int( $network[ 2 ] );
	$tmp[ 3 ] = ( 255 ^ int( $netmask[ 3 ] ) ) | int( $network[ 3 ] );

	for( $i = 0; $i <= 3; $i++ ){
		if( ! ( ( $network[ $i ] <= $target[ $i ] ) && ( $target[ $i ] <= $tmp[ $i ] ) ) ){
			return 0;
		}
	}

	return 1;
}

# --------------------
# Sub1: _split_msg

sub _split_msg
{
	my ( $self, $sockaddr, $message ) = @_;
	my $retMSG;
	my @Ext;

	# --------------------
	# split IP address, port

	( $retMSG->{ PeerPort }, $retMSG->{ PeerAddr } ) = sockaddr_in( $sockaddr );
	$retMSG->{ PeerAddr } = inet_ntoa( $retMSG->{ PeerAddr } );

	# --------------------
	# split message

	( $retMSG->{ PROTO_VER },  $retMSG->{ PacketN }, $retMSG->{ User }, 
	  $retMSG->{ Host }, $retMSG->{ Command }, @Ext) = split( $self->delim_string, $message );

	$retMSG->{ Ext } = join( ":", @Ext );
	( $retMSG->{ Ext }, $retMSG->{ Group } ) = split( $self->delim_group, $retMSG->{ Ext } );

		#	print $retMSG->{ PeerAddr },"\n";
		#	print $retMSG->{ PeerPort },"\n";
		#	print $retMSG->{ PROTO_VER },"\n";
		#	print $retMSG->{ PacketN },"\n";
		#	print $retMSG->{ User },"\n";
		#	print $retMSG->{ Host },"\n";
		#	print $retMSG->{ Command },"\n";
		#	print $retMSG->{ Ext },"\n";
		#	print $retMSG->{ Group },"\n";

	# --------------------
	# return

	return $retMSG;
}

# --------------------
# Sub1: _set_bits

sub _set_bits
{
	my ( $self, @sockets ) = @_;
	my $rin		="";

	# --------------------
	# vec

	foreach my $sock ( @sockets ){ vec( $rin, fileno( $sock ), 1 ) = 1; }

	# --------------------
	# return

	return $rin;
}

# --------------------
# Name: message

sub message
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	$options{ PeerPort }	= $self->port	if ! $options{ PeerPort } or $self->[ UseLocalPort ];

	$options{ PROTO_VER }	= $self->proto_ver		if ! defined $options{ PROTO_VER };
	$options{ PacketN }		= $self->packetn( 1 )	if ! defined $options{ PacketN };
	$options{ User }		= $self->user			if ! defined $options{ User };
	$options{ Host }		= $self->host			if ! defined $options{ Host };

	$options{ NetIF }		= $self if ! $options{ NetIF };

	# --------------------
	# make Message::Send object

	my $send = Ipmsg::Message::Send->new( %options );

	# --------------------
	# return

	return $send;
}

# --------------------
# Name: reply

sub reply
{
	my ( $self, $message ) = @_;

	# --------------------
	# make Send Message object

	my $send = $self->message(
		PeerAddr => $message->addr,
		PeerPort => $message->port,
	);

	# --------------------
	# return

	return $send;
}

# --------------------
# Name: 塼

# --------------------
# Sub1: queue

sub queue
{
	my ( $self, $msg, %options ) = @_;

	# --------------------
	# check requirements

	return $self if ! $msg;

	# --------------------
	# push Send Message to SendQueue

	if( $options{ Mode } eq 'fifo' ){

		unshift @{ $self->[ SendQueue ] }, $msg;

	}else{

		push @{ $self->[ SendQueue ] }, $msg;

	}

	# --------------------
	# return

	return $msg;
}

# --------------------
# Sub1: remainqueue

sub remainqueue
{
	my ( $self, ) = shift;

	# --------------------
	# return

	return $#{ $self->[ SendQueue ] } + 1;
}

# --------------------
# Sub1: send_queue

sub send_queue
{
	my ( $self ) = shift;

	# --------------------
	# shift Send Message from SendQueue

	my $msg = shift @{ $self->[ SendQueue ] };

	# --------------------
	# return

	if( $msg ){

		if( $self->verbose ){ $msg->verbose; }

		my $RET = $msg->send;

		unless( $RET ){
			$self->[ PacketERR ]++;
		}else{
			$self->[ PacketSEND ]++;
		}

		return $RET;

	}else{
		return undef;
	}

}

# --------------------
# Category: get_inaddrbroadcast

sub get_inaddrbroadcast
{
	my ( $self, ) = shift;
	my ( @bc, $bc );

	@bc = unpack( 'C4', INADDR_BROADCAST );
	$bc = sprintf( "%s.%s.%s.%s", $bc[ 0 ], $bc[ 1 ], $bc[ 2 ], $bc[ 3 ] );

	# --------------------
	# return

	return ( wantarray ? @bc : join( ".", @bc ) );
}

# --------------------
# Category: Setting object

# --------------------
# Name: put_value

sub put_value
{
	my $self= shift;
	my %arg	= @_;

	# --------------------
	# check requirements

	return undef if ! %arg;

	while( my ( $name, $value ) = each ( %arg ) ){
		$self->[ $_index{ $name } ] = $value;
	}

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: get_value

sub get_value
{
	my ( $self, $arg ) = @_;

	# --------------------
	# return

	return ( wantarray ? sort keys %_index : $self->[ $_index{ $arg } ] );
}

# --------------------
# Name: display

sub display
{
	my $self = shift;

	# --------------------
	# print

	foreach my $i ( sort keys %_index ){
		print "$i=>",$self->[ $_index{ $i } ],"\n";
	}
}

# --------------------
# Name: code

sub code
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ Code ];
}

# --------------------
# Name: _index

sub _index
{
	my ( $self, $index ) = @_;

	# --------------------
	# return

	return $_index{ $index };
}

1;

