#
# $Id: Message.pm,v 0.09 2003/09/20 Mikio Fukushima Exp Mikio Fukushima $
#

package Ipmsg::Message;
use Ipmsg::Common;

use strict;

use vars qw( $VERSION $THISNAME
	@ISA @EXPORT @EXPORT_OK
);

require Exporter;
require AutoLoader;

@ISA	= qw( Exporter AutoLoader Ipmsg::Common );
@EXPORT	= qw();

# --------------------
# Set global variables in this module.
#

	$VERSION	= 0.09;
	$THISNAME	= "Ipmsg Message module Ver.$VERSION";

	use constant Code	=> 0;

	use constant PeerAddr	=> 1;
	use constant PeerPort	=> 2;

	use constant PROTO_VER	=> 3;
	use constant PacketN	=> 4;
	use constant User		=> 5;
	use constant Host		=> 6;
	use constant Command	=> 7;
	use constant Ext		=> 8;
	use constant Group		=> 9;
	use constant NickName	=>10;

	use constant Mode		=>11;
	use constant Opt		=>12;

	use constant NetIF		=>13;

	my %_index = (
		Code		=> Code,

		PeerAddr	=> PeerAddr,
		PeerPort	=> PeerPort,

		PROTO_VER	=> PROTO_VER,
		PacketN		=> PacketN,
		User		=> User,
		Host		=> Host,
		Command		=> Command,
		Ext			=> Ext,
		Group		=> Group,
		NickName	=> NickName,

		Mode		=> Mode,
		Opt			=> Opt,

		NetIF		=> NetIF,
	);

# --------------------
# Category: Constructor

# --------------------
# Name: new

sub new
{
	my $class = shift;
	   $class = ref( $class ) || $class;

	my %arg = @_;
	my $self = [];

	# --------------------
	# initialize

	@{ $self }[ values %_index ] = @arg{ keys %_index };

	# --------------------
	# set default

	return undef if ! length $self->[ PeerAddr ];
	return undef if ! length $self->[ PeerPort ];
	return undef if ! defined $self->[ User ]  or ! length $self->[ User ];
	return undef if ! defined $self->[ Host ]  or ! length $self->[ Host ];

	return undef if ! defined $self->[ NetIF ];
	return undef if ! ref( $self->[ NetIF ] ) eq "Ipmsg::Net";

	$self->[ PROTO_VER ]	= $class->SUPER::proto_ver if ! length $self->[ PROTO_VER ];
	$self->[ PacketN ]		= 0		if ! defined $self->[ PacketN ];
	$self->[ Command ]		= 0		if ! defined $self->[ Command ];
	$self->[ Ext ]			= ''	if ! defined $self->[ Ext ];
	$self->[ Group ]		= ''	if ! defined $self->[ Group ];
	$self->[ NickName ]		= ''	if ! defined $self->[ NickName ];

	$self->[ Code ]			= 1;

	# --------------------
	# bless

	bless $self, $class;

	# --------------------
	# command init

	$self->command( $arg{ Command } );
	$self->_lengthcheck;

	# --------------------
	# return

	return $self;
}

# --------------------
# Category: Access methods
#
# PeerAddr
# PeerPort
#
# PROTO_VER
# PacketN
# User
# Host
# Command
# Ext
# Group
# NickName
#
# Mode
# Opt
# NetIF

# --------------------
# Name: addr

sub addr
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ PeerAddr ] = $options; return $self; }

	# --------------------
	# return

	return $self->[ PeerAddr ];
}

# --------------------
# Name: port

sub port
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ PeerPort ] = $options; return $self; }

	# --------------------
	# return

	return $self->[ PeerPort ];
}

# --------------------
# Name: proto_ver

sub proto_ver
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ PROTO_VER ] = $options; return $self; }

	# --------------------
	# return

	return $self->[ PROTO_VER ];
}

# --------------------
# Name: packetn

sub packetn
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ PacketN ] = $options; return $self; }

	# --------------------
	# return

	return $self->[ PacketN ];
}

# --------------------
# Name: user

sub user
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ User ] = $options; $self->_lengthcheck; return $self; }

	# --------------------
	# return

	return $self->[ User ];
}

# --------------------
# Name: host

sub host
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ Host ] = $options; $self->_lengthcheck; return $self; }

	# --------------------
	# return

	return $self->[ Host ];
}

# --------------------
# Name: command

sub command
{
	my ( $self, $options ) = @_;

	# --------------------
	# check requirements

	if( $options && $options =~ /[a-zA-Z]+/ ){ $options = $self->ipmsg_cmd( $options ); }

	# --------------------
	# initialize

	if( defined $options ){

		$self->[ Command ]	= $options;
		$self->[ Mode ]		= $self->get_mode( $options );
		$self->[ Opt ]		= $self->get_opt ( $options );

		return $self;

	}

	# --------------------
	# return

	return $self->[ Command ];
}

# --------------------
# Name: ext

sub ext
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ Ext ] = $options; return $self; }

	# --------------------
	# return

	return $self->[ Ext ];
}

# --------------------
# Name: group

sub group
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ Group ] = $options; return $self; }

	# --------------------
	# return

	return $self->[ Group ];
}

# --------------------
# Name: nickname

sub nickname
{
	my ( $self, $options ) = @_;

	# --------------------
	# initialize

	if( defined $options ){ $self->[ NickName ] = $options; return $self; }

	# --------------------
	# return

	return $self->[ NickName ];
}

# --------------------
# Name: mode

sub mode
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ Mode ];
}

# --------------------
# Name: opt

sub opt
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ Opt ];
}

# --------------------
# Name: netif

sub netif
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ NetIF ];
}

# --------------------
# Sub1: _lengthcheck

sub _lengthcheck
{
	my $self = shift;

	# --------------------
	# chop over $MAX_NAMEBUF( 50bytes ) strings.

	$self->[ User ] = substr( $self->[ User ], 0, $self->SUPER::namebuf );
	$self->[ Host ] = substr( $self->[ Host ], 0, $self->SUPER::namebuf );

	# --------------------
	# return

	return $self;
}

# --------------------
# Category: Methods

# --------------------
# Name: as_message

sub as_message
{
	my $self = shift;

	# --------------------
	# make message and return

	return join( $self->delim_string,
		$self->proto_ver,
		$self->packetn,
		$self->user,
		$self->host,
		$self->command,
		$self->ext,
	);
}

# --------------------
# Name: length

sub length
{
	my $self = shift;

	# --------------------
	# return

	return length $self->as_message;
}

# --------------------
# Category: Setting object

# --------------------
# Name: put_value

sub put_value
{
	my $self= shift;
	my %arg	= @_;

	# --------------------
	# check requirements

	return undef if ! %arg;

	while( my ( $name, $value ) = each ( %arg ) ){
		$self->[ $_index{ $name } ] = $value;
	}

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: get_value

sub get_value
{
	my ( $self, $arg ) = @_;

	# --------------------
	# return

	return ( wantarray ? sort keys %_index : $self->[ $_index{ $arg } ] );
}

# --------------------
# Name: display

sub display
{
	my $self = shift;

	# --------------------
	# print

	foreach my $i ( sort keys %_index ){
		print "$i=>",$self->[ $_index{ $i } ],"\n";
	}

	return $self;
}

# --------------------
# Name: code

sub code
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ Code ];
}

# --------------------
# Name: _index

sub _index
{
	my ( $self, $index ) = @_;

	# --------------------
	# return

	return $_index{ $index };
}

1;

