#
# $Id: Agent.pm,v 0.25 2004/03/08 Mikio Fukushima Exp Mikio Fukushima $
#

package Ipmsg::Agent;

use Ipmsg::Common;
use Ipmsg::Net;
use Ipmsg::HostList;
use Ipmsg::Host;

use strict;

use vars qw( $VERSION $THISNAME
	@ISA @EXPORT @EXPORT_OK
);

require Exporter;
require AutoLoader;

@ISA	= qw( Exporter AutoLoader Ipmsg::Common );
@EXPORT	= qw();

# --------------------
# Set global variables in this module.
#

	$VERSION	= 0.25;
	$THISNAME	= "Ipmsg Agent module Ver.$VERSION";

	use constant Code	=> 0;

	use constant LocalPort		=> 1;

	use constant Verbose		=> 2;
	use constant UseLocalPort	=> 3;
	use constant DirectBroadcast=> 4;

	use constant RetryInterval	=> 5;
	use constant RetryMAX		=> 6;
	use constant ExpireListGet	=> 7;

	use constant AllowListSend	=> 8;
	use constant AllowListRecv	=> 9;

	use constant BroadCastAddr	=> 10;
	use constant DialUpAddr		=> 11;

	use constant HostInfo		=> 12;
	use constant HostList		=> 13;
	use constant NetIF			=> 14;

	use constant WaitQueue		=> 15;
	use constant NoResponce		=> 16;

	my %_index = (
		Code => Code,

		LocalPort		=> LocalPort,

		Verbose			=> Verbose,
		UseLocalPort	=> UseLocalPort,
		DirectBroadcast	=> DirectBroadcast,

		RetryInterval	=> RetryInterval,
		RetryMAX		=> RetryMAX,
		ExpireListGet	=> ExpireListGet,

		AllowListSend	=> AllowListSend,
		AllowListRecv	=> AllowListRecv,

		BroadCastAddr	=> BroadCastAddr,
		DialUpAddr		=> DialUpAddr,

		HostInfo		=> HostInfo,
		HostList		=> HostList,
		NetIF			=> NetIF,

		WaitQueue		=> WaitQueue,
		NoResponce		=> NoResponce,
	);

# --------------------
# Category: Constructor

# --------------------
# Name: new

sub new
{
	my $class = shift;
	   $class = ref( $class ) || $class;

	my %arg = @_;
	my $self = [];

	# --------------------
	# initialize

	@{ $self }[ values %_index ] = @arg{ keys %_index };

	# --------------------
	# set default properties

	$self->[ Verbose ]			= 0		if ! defined $arg{ Verbose };
	$self->[ UseLocalPort ]		= 0		if ! defined $arg{ UseLocalPort };
	$self->[ DirectBroadcast ]	= 1		if ! defined $arg{ DirectBroadcast };

	$self->[ RetryInterval	]	= 2		if ! defined $arg{ RetryInterval };
	$self->[ RetryMAX ]			= 4		if ! defined $arg{ RetryMAX };
	$self->[ ExpireListGet ]	= 3		if ! defined $arg{ ExpireListGet };

	$self->[ AllowListSend ]	= 1;
	$self->[ AllowListRecv ]	= 1;

	$self->[ DialUpAddr ]		= {};
	$self->[ BroadCastAddr ]	= {};

	$self->[ HostInfo ]			= '';
	$self->[ HostList ]			= [];
	$self->[ NetIF ]			= [];

	$self->[ WaitQueue ]		= {};
	$self->[ NoResponce ]		= [];

	$self->[ Code ]				= 1;

	# --------------------
	# set host information

	$arg{ LocalPort }		= $class->SUPER::default_port if ! defined $arg{ LocalPort };
	$arg{ LocalAddr }		= $arg{ BindAddr }	if ! defined $arg{ LocalAddr };

	$arg{ PROTO_VER }		= 1		if ! defined $arg{ PROTO_VER };

	$arg{ User }			= getlogin					if ! defined $arg{ User } or ! length $arg{ User };
	$arg{ User } 			= $ENV{ 'USERNAME' }		if ! defined $arg{ User } or ! length $arg{ User };
	return undef										if ! length  $arg{ User };

	$arg{ Host } 			= $ENV{ 'HOSTNAME' }		if ! defined $arg{ Host } or ! length $arg{ Host };
	$arg{ Host } 			= $ENV{ 'COMPUTERNAME' }	if ! defined $arg{ Host } or ! length $arg{ Host };
	return undef										if ! length  $arg{ Host };

	$arg{ NickName }		= $arg{ User }				if ! defined $arg{ NickName };
	$arg{ Group }			= ""						if ! defined $arg{ Group };

	$arg{ Agent }			= "Perl/IP Messanger Ver.$VERSION" if ! defined $arg{ Agent };

	$arg{ AbsenceHead }		= '[sleep]'					if ! defined $arg{ AbsenceHead };
	$arg{ AbsenceStr }		= 'Now sleeping...'			if ! defined $arg{ AbsenceStr };
	$arg{ AbsenceStrNot }	= 'Not absence mode'		if ! defined $arg{ AbsenceStrNot };
	$arg{ Pubkey }			= ''						if ! defined $arg{ Pubkey };

	$arg{ ABSENCEOPT }		= 0 if ! defined $arg{ ABSENCEOPT };
	$arg{ SERVEROPT }		= 0 if ! defined $arg{ SERVEROPT };
	$arg{ DIALUPOPT }		= 0 if ! defined $arg{ DIALUPOPT };
	$arg{ FILEATTACHOPT }	= 0 if ! defined $arg{ FILEATTACHOPT };
	$arg{ ENCRYPTOPT }		= 0 if ! defined $arg{ ENCRYPTOPT };

	# --------------------
	# bless

	bless $self, $class;

	# --------------------
	# init

	$self->[ HostInfo ] = Ipmsg::Host->new(
		PeerAddr		=> $arg{ LocalAddr },
		PeerPort		=> $arg{ LocalPort },
		PROTO_VER		=> $arg{ PROTO_VER },
		User			=> $arg{ User },
		Host			=> $arg{ Host },
		Command			=> $arg{ Command },
		NickName		=> $arg{ NickName },
		Group			=> $arg{ Group },

		Agent			=> $arg{ Agent },
		AbsenceHead		=> $arg{ AbsenceHead },
		AbsenceStr		=> $arg{ AbsenceStr },
		AbsenceStrNot	=> $arg{ AbsenceStrNot },
	);

	$self->add_hostlist;

	$self->add_netif(
		BindAddr		=> $arg{ BindAddr },
		LocalPort		=> $arg{ LocalPort },
	) or return undef;

	# --------------------
	# command init

	return $self;
}

# --------------------
# Name: DESTROY

sub DESTROY
{
	my $self = shift;

	# --------------------
	# close socket

	for my $i ( $self->netif ){ $i->close; }
}

# --------------------
# Category: Access methods

# --------------------
# Name: Ⱦ

# --------------------
# Sub1: hostinfo

sub hostinfo
{
	my ( $self ) = shift;

	# --------------------
	# return

	return $self->[ HostInfo ];
}

# --------------------
# Name: ۥȥꥹ

# --------------------
# Sub1: add_hostlist

sub add_hostlist
{
	my ( $self, %options ) = @_;

	# --------------------
	# make hostlist object

	my $hostlist = Ipmsg::HostList->new( %options );

	push @{ $self->[ HostList ] }, $hostlist;

	# --------------------
	# return

	return $hostlist;
}

# --------------------
# Sub1: hostlist

sub hostlist
{
	my ( $self, $number ) = @_;

	# --------------------
	# check requirements

	if( $number && $number =~ /total/i )	{ return scalar @{ $self->[ HostList ] } };
	if( ! defined $number )		{ $number = 0; }

	# --------------------
	# return

	return ( wantarray ? @{ $self->[ HostList ] }: $self->[ HostList ]->[ $number ] );
}

# --------------------
# Sub1: total

sub total
{
	my ( $self, $group ) = @_;

	my $total_reghost;

	for my $i ( $self->hostlist ){
		$total_reghost += $i->total( $group );
	}

	# --------------------
	# set code

	$self->[ Code ] = $total_reghost;

	# --------------------
	# return

	return $total_reghost;
}

# --------------------
# Name: ͥåȥ󥿡ե

# --------------------
# Sub1: add_netif

sub add_netif
{
	my ( $self, %options ) = @_;

	# --------------------
	# check requirements

	$options{ UseLocalPort }	= $self->[ UseLocalPort ]	 if( ! defined $options{ UseLocalPort } );
	$options{ Verbose }			= $self->[ Verbose ]		 if( ! defined $options{ Verbose } );
	$options{ DirectBroadcast }	= $self->[ DirectBroadcast ] if( ! defined $options{ DirectBroadcast } );

	$options{ User } = $self->hostinfo->user if( ! defined $options{ User } );
	$options{ Host } = $self->hostinfo->host if( ! defined $options{ Host } );

	# --------------------
	# make hostlist object

	my $netif;

	if( $options{ NetIF } ){
		$netif = $options{ NetIF };
	}else{
		$netif = Ipmsg::Net->new( %options );
	}

	push @{ $self->[ NetIF ] }, $netif if $netif;

	# --------------------
	# set code

	$self->[ Code ] = scalar @{ $self->[ NetIF ] } - 1;

	# --------------------
	# return

	return $netif;
}

# --------------------
# Sub1: del_netif

sub del_netif
{
	my ( $self, $options ) = @_;
	my $count;

	# --------------------
	# check requirements

	return undef if ! $options;

	# --------------------
	# delete netif

	if( $options =~ /^\d+/g ){

		splice @{ $self->[ NetIF ] }, $options, 1;

	}else{

		for( my $i = 0; $i <= scalar @{ $self->[ NetIF ] } - 1; $i++ ){
			if( $self->[ NetIF ]->[ $i ] eq $options ){
				$count = $i;
			}
		}

		if( $count ){ splice @{ $self->[ NetIF ] }, $count, 1; }

	}

	# --------------------
	# set code

	$self->[ Code ] = scalar @{ $self->[ NetIF ] } - 1;

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: netif

sub netif
{
	my ( $self, $number ) = @_;

	# --------------------
	# check requirements

	if( $number && $number =~ /total/i )	{ return scalar @{ $self->[ NetIF ] } };
	if( ! defined $number )		{ $number = 0; }

	# --------------------
	# set code

	$self->[ Code ] = scalar @{ $self->[ NetIF ] };

	# --------------------
	# return

	return ( wantarray ? @{ $self->[ NetIF ] } : $self->[ NetIF ]->[ $number ] );
}

# --------------------
# Name: 塼

# --------------------
# Sub1: noresponce

sub noresponce
{
	my ( $self ) = shift;

	# --------------------
	# set code

	$self->[ Code ] = scalar @{ $self->[ NoResponce ] } - 1;

	# --------------------
	# return

	return( shift @{ $self->[ NoResponce ] } );
}

# --------------------
# Sub1: retryinterval

sub retryinterval
{
	my ( $self ) = shift;

	# --------------------
	# return

	@_	? $self->[ RetryInterval ] = shift
		: $self->[ RetryInterval ];
}

# --------------------
# Sub1: retrymax

sub retrymax
{
	my ( $self ) = shift;

	# --------------------
	# return

	@_	? $self->[ RetryMAX ] = shift
		: $self->[ RetryMAX ];
}

# --------------------
# Sub1: expirelistget

sub expirelistget
{
	my ( $self ) = shift;

	# --------------------
	# return

	@_	? $self->[ ExpireListGet ] = shift
		: $self->[ ExpireListGet ];
}

# --------------------
# Name: IPɥ쥹Ϣ
#
# --------------------
# Sub1: broadcast

# --------------------
# Sub2: add_broadcast

sub add_broadcast {

	my ( $self, %options ) = @_;
	my $NetIF;

	# --------------------
	# initialize

	$self->[ Code ] = undef;

	# --------------------
	# check requirements

	if( ! $options{ PeerPort } ){ $options{ PeerPort }	= $self->hostinfo->port; }
	if( ! $options{ NetMask } )	{ $options{ NetMask }	= '255.255.255.0'; }

	# --------------------
	# set default NetIF

	if( ref( $options{ NetIF } ) eq 'Ipmsg::Net' ){
		$NetIF = $options{ NetIF }; delete $options{ NetIF };
	}else{
		$NetIF = $self->netif( 0 );
	}

	# --------------------
	# add BroadCastAddr Network

	if( $options{ PeerNet } ){

		foreach my $i ( _calc_addrbynetwork( $options{ PeerNet }, $options{ NetMask }, 1 ) ){
			$self->[ BroadCastAddr ]{ "$i:$options{ PeerPort }" } = $NetIF;
		}

	# --------------------
	# add BroadCastAddr

	}else{

		if( ! defined $options{ PeerAddr } ){ return undef; }
		$self->[ BroadCastAddr ]{ "$options{ PeerAddr }:$options{ PeerPort }" } = $NetIF;

	}

	# --------------------
	# set code

	$self->[ Code ] = scalar keys %{ $self->[ BroadCastAddr ] };

	return $self;
}

# --------------------
# Sub2: del_broadcast

sub del_broadcast {

	my ( $self, %options ) = @_;

	# --------------------
	# initialize

	$self->[ Code ] = undef;

	# --------------------
	# delete all broadcast address

	if( ! %options ){
		$self->[ BroadCastAddr ]	= {};
		$self->[ Code ]				= 0;
		return $self;
	}

	# --------------------
	# check requirements

	return undef if ! $options{ PeerNet };
	if( ! defined $options{ PeerPort } ){ $options{ PeerPort }	= $self->hostinfo->port; }
	if( ! defined $options{ NetMask } )	{ $options{ NetMask }	= '255.255.255.0'; }

	# --------------------
	# del BroadCastAddr Network

	if( $options{ PeerNet } ){

		foreach my $i ( _calc_addrbynetwork( $options{ PeerNet }, $options{ NetMask }, 1 ) ){
			delete $self->[ BroadCastAddr ]{ "$i:$options{ PeerPort }" };
		}

	# --------------------
	# del BroadCastAddr

	}else{

		if( ! defined $options{ PeerAddr } ){ return undef; }
		delete $self->[ BroadCastAddr ]{ "$options{ PeerAddr }:$options{ PeerPort }" };

	}

	# --------------------
	# set code

	$self->[ Code ] = scalar keys %{ $self->[ BroadCastAddr ] };

	return $self;
}

# --------------------
# Sub2: get_broadcast

sub get_broadcast {

	my ( $self ) = shift;
	my @list;

	# --------------------
	# initialize

	$self->[ Code ] = undef;

	# --------------------
	# get IP address and Port No. from HostList

	for my $i ( sort keys %{ $self->[ BroadCastAddr ] } )
	{
		push @list, split( /:/, $i ), $self->[ BroadCastAddr ]->{ $i };
	}

	# -------------------
	# set code

	$self->[ Code ] = ( $#list + 1 ) / 3;

	# -------------------
	# return

	return ( wantarray ? @list : ( $#list + 1 ) / 3 );
}

# --------------------
# Sub1: dialup

# --------------------
# Sub2: add_dialup

sub add_dialup {

	my ( $self, %options ) = @_;
	my $NetIF;

	# --------------------
	# initialize

	$self->[ Code ] = undef;

	# --------------------
	# check requirements

	if( ! $options{ PeerPort } ){ $options{ PeerPort }	= $self->hostinfo->port; }
	if( ! $options{ NetMask } )	{ $options{ NetMask }	= '255.255.255.0'; }

	# --------------------
	# set default NetIF

	if( ref( $options{ NetIF } ) eq 'Ipmsg::Net' ){
		$NetIF = $options{ NetIF }; delete $options{ NetIF };
	}else{
		$NetIF = $self->netif( 0 );
	}

	# --------------------
	# add BroadCastAddr Network

	if( $options{ PeerNet } ){

		foreach my $i ( _calc_addrbynetwork( $options{ PeerNet }, $options{ NetMask }, 1 ) ){
			$self->[ DialUpAddr ]{ "$i:$options{ PeerPort }" } = $NetIF;
		}

	# --------------------
	# add BroadCastAddr

	}else{

		if( ! defined $options{ PeerAddr } ){ return undef; }
		$self->[ DialUpAddr ]{ "$options{ PeerAddr }:$options{ PeerPort }" } = $NetIF;

	}

	# --------------------
	# set code

	$self->[ Code ] = scalar keys %{ $self->[ DialUpAddr ] };

	return $self;
}

# --------------------
# Sub2: del_dialup

sub del_dialup {

	my ( $self, %options ) = @_;

	# --------------------
	# initialize

	$self->[ Code ] = undef;

	# --------------------
	# delete all broadcast address

	if( ! %options ){
		$self->[ DialUpAddr ]	= {};
		$self->[ Code ]			= 0;
		return $self;
	}

	# --------------------
	# check requirements

	return undef if ! $options{ PeerNet };
	if( ! defined $options{ PeerPort } ){ $options{ PeerPort }	= $self->hostinfo->port; }
	if( ! defined $options{ NetMask } )	{ $options{ NetMask }	= '255.255.255.0'; }

	# --------------------
	# add BroadCastAddr Network

	if( $options{ PeerNet } ){

		foreach my $i ( _calc_addrbynetwork( $options{ PeerNet }, $options{ NetMask }, 1 ) ){
			delete $self->[ DialUpAddr ]{ "$i:$options{ PeerPort }" };
		}

	# --------------------
	# add BroadCastAddr

	}else{

		if( ! defined $options{ PeerAddr } ){ return undef; }
		delete $self->[ DialUpAddr ]{ "$options{ PeerAddr }:$options{ PeerPort }" };

	}

	# --------------------
	# set code

	$self->[ Code ] = scalar keys %{ $self->[ DialUpAddr ] };

	return $self;
}

# --------------------
# Sub2: get_dialup

sub get_dialup {

	my ( $self ) = @_;
	my @list;

	# --------------------
	# initialize

	$self->[ Code ] = undef;

	# --------------------
	# get IP address and Port No. from HostList

	for my $i ( sort keys %{ $self->[ DialUpAddr ] } )
	{
		push @list, split( /:/, $i ), $self->[ DialUpAddr ]->{ $i };
	}

	# -------------------
	# set code

	$self->[ Code ] = ( $#list + 1 ) / 3;

	# -------------------
	# return

	return ( wantarray ? @list : ( $#list + 1 ) / 3 );
}

# --------------------
# Sub1: _calc_addrbynetwork

sub _calc_addrbynetwork
{
	my ( $addr, $mask, $flag ) = @_;
	my ( $i, $i0, $i1, $i2, $i3, @list, @base );
	my @addr = split( /\./, $addr );
	my @mask = split( /\./, $mask );
	my @null = split( /\./, "255.255.255.255" );

	for( $i = 0; $i <= 3; $i++ ){ $addr[ $i ] += 0; $mask[ $i ] += 0; $null[ $i ] += 0; }

	$base[ 0 ] = $addr[ 0 ] & $mask[ 0 ];
	$base[ 1 ] = $addr[ 1 ] & $mask[ 1 ];
	$base[ 2 ] = $addr[ 2 ] & $mask[ 2 ];
	$base[ 3 ] = $addr[ 3 ] & $mask[ 3 ];

	for( $i0 = 0; $i0 <= $null[ 0 ] - $mask[ 0 ]; $i0++ ){
		for( $i1 = 0; $i1 <= $null[ 1 ] - $mask[ 1 ]; $i1++ ){
			for( $i2 = 0; $i2 <= $null[ 2 ] - $mask[ 2 ]; $i2++ ){
				for( $i3 = 0; $i3 <= $null[ 3 ] - $mask[ 3 ]; $i3++ ){
					push @list, join( ".", $base[ 0 ] + $i0, $base[ 1 ] + $i1, $base[ 2 ] + $i2, $base[ 3 ] + $i3 );
				}
			}
		}
	}

	@list = sort @list;

	if( $flag ){ if( scalar @list > 3 ){ shift @list; } }

	# --------------------
	# return

	return ( wantarray ? @list : $#list + 1 );

}

# --------------------
# Name: allowlistsend

sub allowlistsend
{
	my ( $self ) = shift;

	# --------------------
	# return

	@_	? $self->[ AllowListSend ] = shift
		: $self->[ AllowListSend ];
}

# --------------------
# Name: allowlistrecv

sub allowlistrecv
{
	my ( $self ) = shift;

	# --------------------
	# return

	@_	? $self->[ AllowListRecv ] = shift
		: $self->[ AllowListRecv ];
}

# --------------------
# Name: verbose

sub verbose
{
	my ( $self ) = shift;

	# --------------------
	# return

	@_	? $self->[ Verbose ] = shift
		: $self->[ Verbose ];
}

# --------------------
# Name: uselocalport

sub uselocalport
{
	my ( $self ) = shift;

	# --------------------
	# return

	@_	? $self->[ UseLocalPort ] = shift
		: $self->[ UseLocalPort ];
}

# --------------------
# Name: directbroadcast

sub directbroadcast
{
	my ( $self ) = shift;

	# --------------------
	# return

	@_	? $self->[ DirectBroadcast ] = shift
		: $self->[ DirectBroadcast ];
}

# --------------------
# Category: Methods

# --------------------
# Name: å

# --------------------
# Sub1: recv

sub recv
{
	my ( $self, %options ) = @_;
	my $NetIF;
	my $HostList;

	# --------------------
	# set code

	$self->[ Code ] = undef;

	# --------------------
	# check requirements

	$options{ FROMCHECK }			= 0 if ! defined $options{ FROMCHECK };
	$options{ SELFACPT }			= 0 if ! defined $options{ SELFACPT };

	$options{ AUTOREJECT }			= 1 if ! defined $options{ AUTOREJECT };
	$options{ AUTOHOST }			= 1 if ! defined $options{ AUTOHOST };
	$options{ AUTOINFO }			= 1 if ! defined $options{ AUTOINFO };
	$options{ AUTOMSG }				= 1 if ! defined $options{ AUTOMSG };
	$options{ AUTOOPEN }			= 0 if ! defined $options{ AUTOOPEN };
	$options{ AUTOLIST }			= 1 if ! defined $options{ AUTOLIST };
	$options{ AUTOKEY }				= 0 if ! defined $options{ AUTOKEY };

	$options{ CHECK_WAITQUEUE }		= 1 if ! defined $options{ CHECK_WAITQUEUE };

	$options{ Mode } 				= 'fifo' if ! defined $options{ Mode };

		# --------------------
		# set default NetIF

		if( ref( $options{ NetIF } ) eq 'Ipmsg::Net' ){
			$NetIF = $options{ NetIF }; delete $options{ NetIF };
		}else{
			$NetIF = $self->netif( 0 );
		}

		# --------------------
		# set default HostList

		if( ref( $options{ HostList } ) eq 'Ipmsg::HostList' ){
			$HostList = $options{ HostList }; delete $options{ HostList };
		}else{
			$HostList = $self->hostlist( 0 );
		}

	# --------------------
	# make Message::Recv object

	my $recv = $NetIF->recv( FROMCHECK => $options{ FROMCHECK }, SELFACPT => $options{ SELFACPT } );

	if( $recv ){

		# --------------------
		# AUTOREJECT option

		if( $options{ AUTOREJECT } ){

			if( ! $self->_autoreject( $recv ) ){
				$self->[ Code ] = 2;
				return 0;
			}

		}

		# --------------------
		# AUTO* option

		$self->_autohost( $recv, $NetIF, $HostList, $options{ Mode } ) if $options{ AUTOHOST };
		$self->_autoinfo( $recv, $NetIF, $HostList, $options{ Mode } ) if $options{ AUTOINFO };
		$self->_automsg ( $recv, $NetIF, $HostList, $options{ Mode } ) if $options{ AUTOMSG  };
		$self->_autoopen( $recv, $NetIF, $HostList, $options{ Mode } ) if $options{ AUTOOPEN };
		$self->_autolist( $recv, $NetIF, $HostList, $options{ Mode } ) if $options{ AUTOLIST };
		$self->_autokey ( $recv, $NetIF, $HostList, $options{ Mode } ) if $options{ AUTOKEY } ;

	}

	# --------------------
	# CHECK_WAITQUEUE option

	$self->_check_waitqueue if $options{ CHECK_WAITQUEUE } ;

	# -------------------
	# set code

	$self->[ Code ] = $NetIF->code;

	# --------------------
	# return

	return $recv;
}

# --------------------
# Sub2: _autoreject

sub _autoreject
{
	my ( $self, $recv ) = @_;

	# --------------------
	# check requirements

	if( ! $recv ){ return undef; }

	# --------------------
	# check reject IPMSG_OKGETLIST

	if( $recv->okgetlist && ! $self->wait_queue( $recv ) ){
		print STDERR "       REJECT : IPMSG_OKGETLIST\n" if $self->verbose;
		return undef;
	}

	# --------------------
	# check reject IPMSG_ANSLIST

	if( $recv->anslist && ! $self->wait_queue( $recv ) ){
		print STDERR "       REJECT : IPMSG_ANSLIST\n" if $self->verbose;
		return undef;
	}

	# --------------------
	# check reject IPMSG_RECVMSG

	if( $recv->recvmsg && ! $self->wait_queue( $recv ) ){
		print STDERR "       REJECT : IPMSG_RECVMSG\n" if $self->verbose;
		return undef;
	}

	# --------------------
	# check reject IPMSG_GETFILEDATA

	# --------------------
	# check reject IPMSG_RELEASEFILES

	# --------------------
	# check reject IPMSG_GETDIRFILES

	# --------------------
	# check reject IPMSG_SENDINFO

	if( $recv->sendinfo && ! $self->wait_queue( $recv ) ){
		print STDERR "       REJECT : IPMSG_SENDINFO\n" if $self->verbose;
		return undef;
	}

	# --------------------
	# check reject IPMSG_SENDABSENCEINFO

	if( $recv->sendabsenceinfo && ! $self->wait_queue( $recv ) ){
		print STDERR "       REJECT : IPMSG_SENDABSENCEINFO\n" if $self->verbose;
		return undef;
	}

	# --------------------
	# check reject IPMSG_ANSPUBKEY

	if( $recv->anspubkey && ! $self->wait_queue( $recv ) ){
		print STDERR "       REJECT : IPMSG_ANSPUBKEY\n" if $self->verbose;
		return undef;
	}

	# --------------------
	# return

	return $recv;
}

# --------------------
# Sub2: _autohost

sub _autohost
{
	my ( $self, $recv, $NetIF, $HostList, $Mode ) = @_;

	# --------------------
	# check requirements

	if( ! $recv || ! $NetIF || ! $HostList ){ return undef; }

	# --------------------
	# IPMSG_BR_ENTRY, IPMSG_ANSENTRY and IPMSG_BR_ABSENCE

	if( $recv->entry or $recv->ansentry or $recv->absence ){

		# --------------------
		# add host to hostlist

		my $host = $HostList->add(
			PeerAddr	=> $recv->addr,
			PeerPort	=> $recv->port,
			PROTO_VER	=> $recv->proto_ver,
			User		=> $recv->user,
			Host		=> $recv->host,
			Command		=> $recv->command,
			NickName	=> $recv->ext,
			Group		=> $recv->group,
		);

		# --------------------
		# add host to dialuplist

		$self->add_dialup( PeerAddr => $recv->addr, PeerPort => $recv->port, NetIF => $NetIF ) if $recv->dialupopt;

		# --------------------
		# reply IPMSG_ANSENTRY

		$recv->reply->nickname( $self->hostinfo->nickname )->group( $self->hostinfo->group )->ansentry->queue( Mode => $Mode ) if $recv->entry;

		# --------------------
		# return

		return $self;

	} # End of IPMSG_BR_ENTRY

	# --------------------
	# IPMSG_BR_EXIT

	if( $recv->exit ){

		# --------------------
		# del host from hostlist

		$HostList->del( PeerAddr => $recv->addr, PeerPort => $recv->port );

		# --------------------
		# del host to dialuplist

		$self->del_dialup( PeerAddr => $recv->addr, PeerPort => $recv->port );

		# --------------------
		# return

		return $self;

	} # End of IPMSG_BR_EXIT

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub2: _autoinfo

sub _autoinfo
{
	my ( $self, $recv, $NetIF, $HostList, $Mode ) = @_;

	# --------------------
	# check requirements

	if( ! $recv || ! $NetIF || ! $HostList ){ return undef; }

	# --------------------
	# IPMSG_GETINFO

	if( $recv->getinfo ){

		# --------------------
		# reply IPMSG_SENDINFO

		$recv->reply->ext( $self->hostinfo->agent )->sendinfo->queue( Mode => $Mode );

		# --------------------
		# return

		return $self;

	} # End of IPMSG_GETINFO

	# --------------------
	# IPMSG_SENDINFO

	if( $recv->sendinfo ){

		# --------------------
		# check hostlist
		# ۥȥꥹȤ IPMSG_SENDINFO ƤۥȤϿƤ뤫å
		# 줿 Agent Ͽ
		# Ƥʤ硢IPMSG_BR_ENTRY  IPMSG_GETINFO 

		my $peerhost = $HostList->get( PeerAddr => $recv->addr, PeerPort => $recv->port );

		# --------------------
		# ۥȤϿƤ

		if( $peerhost ){

			$peerhost->agent( $recv->ext );

		# --------------------
		# ۥȤϿƤʤä

		}else{

			$recv->reply->nickname( $self->hostinfo->nickname )->group( $self->hostinfo->group )->entry->queue( Mode => $Mode );
			$self->wait_queue( $recv->reply->getinfo->queue( Mode => $Mode ) );

		}

		# --------------------
		# return

		return $self;

	} # End of IPMSG_GETINFO

	# --------------------
	# IPMSG_GETABSENCEINFO

	if( $recv->getabsenceinfo ){

		# --------------------
		# reply IPMSG_SENDABSENCEINFO

		if( $self->hostinfo->absence ){
			$recv->reply->ext( $self->hostinfo->absencestr )->sendabsenceinfo->queue( Mode => $Mode );
		}else{
			$recv->reply->ext( $self->hostinfo->absencestrnot )->sendabsenceinfo->queue( Mode => $Mode );
		}

		# --------------------
		# return

		return $self;

	} # End of IPMSG_GETINFO

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub2: _automsg

sub _automsg
{
	my ( $self, $recv, $NetIF, $HostList, $Mode ) = @_;

	# --------------------
	# check requirements

	if( ! $recv || ! $NetIF || ! $HostList ){ return undef; }

	# --------------------
	# IPMSG_AUTORETOPT & IPMSG_BROADCASTOPT

	if( $recv->autoretopt || $recv->broadcastopt ){

		return $self;

	} # End of IPMSG_AUTORETOPT and IPMSG_BROADCASTOPT

	# --------------------
	# IPMSG_SENDCHECKOPT

	if( $recv->sendmsg && $recv->sendcheckopt ){

		# --------------------
		# reply IPMSG_RECVMSG

		$recv->reply->ext( $recv->packetn )->recvmsg->queue( Mode => $Mode );

	} # End of IPMSG_SENDCHECKOPT

	# --------------------
	# If run under Absence mode. reply message "Not absence mode!!".

	if( $recv->sendmsg && $self->hostinfo->absence ){

		$recv->reply->ext( $self->hostinfo->absencestr )->sendmsg->autoretopt->queue( Mode => $Mode );

	} # End of Absence mode.

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub2: _autoopen

sub _autoopen
{
	my ( $self, $recv, $NetIF, $HostList, $Mode ) = @_;

	# --------------------
	# check requirements

	if( ! $recv || ! $NetIF || ! $HostList ){ return undef; }

	# --------------------
	# IPMSG_AUTORETOPT & IPMSG_BROADCASTOPT

	if( $recv->autoretopt || $recv->broadcastopt || ! $recv->sendmsg ){

		return $self;

	} # End of IPMSG_AUTORETOPT and IPMSG_BROADCASTOPT

	# --------------------
	# IPMSG_SECRETOPT

	if( $recv->sendmsg && $recv->secretopt ){

		# --------------------
		# reply IPMSG_READMSG

		$recv->reply->ext( $recv->packetn )->readmsg->queue( Mode => $Mode );

	} # End of IPMSG_SECRETOPT

	# --------------------
	# IPMSG_READCHECKOPT

	if( $recv->sendmsg && $recv->readcheckopt ){

		# --------------------
		# reply IPMSG_ANSREADMSG

		$recv->reply->ext( $recv->packetn )->ansreadmsg->queue( Mode => $Mode );

	} # End of IPMSG_READCHECKOPT

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub2: _autolist

sub _autolist
{
	my ( $self, $recv, $NetIF, $HostList, $Mode ) = @_;

	# --------------------
	# check requirements

	if( ! $recv || ! $NetIF || ! $HostList ){ return undef; }

	# --------------------
	# IPMSG_AUTORETOPT & IPMSG_BROADCASTOPT

	if( $recv->autoretopt || $recv->broadcastopt ){

		return $self;

	} # End of IPMSG_AUTORETOPT and IPMSG_BROADCASTOPT

	# --------------------
	# IPMSG_BR_ISGETLIST or IPMSG_BR_ISGETLIST2

	if( $recv->isgetlist or $recv->isgetlist2 ){

		# --------------------
		# reply IPMSG_OKGETLIST

		$recv->reply->okgetlist->queue( Mode => $Mode );

		# --------------------
		# return

		return $self;

	} # End of IPMSG_BR_ISGETLIST or IPMSG_BR_ISGETLIST2

	# --------------------
	# IPMSG_GETLIST

	if( $recv->getlist and $self->allowlistsend ){

		# --------------------
		# ϿƤۥȤ IPMSG_ANSLIST ֿ

		if( $HostList->total ){

			# --------------------
			# make hostlist for reply

			my $reply			= $recv->reply->anslist;

			my $req_nexthost 	= 1;

			if( $recv->ext =~ /^\d+$/ ){ $req_nexthost = $recv->ext; }

			my ( $nexthost, $total, $list ) = $HostList->get_hostlist(
													Count		=> $req_nexthost,
													Length		=> $self->udpbuf - $reply->length,
													DropCommand => 1,
											);

			my $hostlist = sprintf( "%5d%s%5d%s%s",
				$nexthost ? $nexthost : 0,
				$self->delim_hostlist,
				$total,
				$self->delim_hostlist,
				$list );

				#	print "Request  => $req_nexthost\n";
				#	printf "%5d\n", $nexthost;
				#	printf "%5d\n", $total;
				#	printf "%s\n",  $self->as_string( $list );

			# --------------------
			# reply IPMSG_ANSLIST

			$reply->ext( $hostlist )->queue( Mode => $Mode );

		} # End of make_hostlist

		# --------------------
		# return

		return $self;

	} # End of IPMSG_GETLIST

	# --------------------
	# IPMSG_ANSLIST

	if( $recv->anslist and $self->allowlistrecv ){

		# --------------------
		# ۥȥꥹȤۥȥꥹȤɲä
		# ɲäۥȤ dialup ۥȤä륢åץꥹȤˤɲä

		foreach my $i ( $HostList->add_hostlist( $recv->ext ) ){
			$self->add_dialup( PeerAddr => $i->addr, PeerPort => $i->port, NetIF => $NetIF ) if $i->dialup;
		}

		# --------------------
		# ³ΥۥȥꥹȤ IPMSG_GETLIST wait_queue 

		$self->wait_queue( $recv->reply->getlist->ext( $HostList->code )->queue( Mode => $Mode ) ) if $HostList->code;

		# --------------------
		# return

		return $self;

	}

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub2: _autokey

sub _autokey
{
	my ( $self, $recv, $NetIF, $HostList, $Mode ) = @_;

	# --------------------
	# check requirements

	if( ! $recv || ! $NetIF || ! $HostList ){ return undef; }

	# --------------------
	# IPMSG_AUTORETOPT & IPMSG_BROADCASTOPT

	if( $recv->autoretopt || $recv->broadcastopt ){

		return $self;

	} # End of IPMSG_AUTORETOPT and IPMSG_BROADCASTOPT

	# --------------------
	# IPMSG_GETPUBKEY

	if( $recv->getpubkey ){

		my ( $power, $exp, $key ) = $self->hostinfo->pubkey;

		if( $power ){
			$self->wait_queue( $recv->reply->getpubkey->ext( $power . ":" . $exp . "-" . $key )->queue( Mode => $Mode ) );
		}

		# --------------------
		# return

		return $self;

	} # End of IPMSG_GETPUBKEY

	# --------------------
	# IPMSG_ANSPUBKEY

	if( $recv->anspubkey ){

		# --------------------
		# check hostlist
		# ۥȥꥹȤ IPMSG_ANSPUB ƤۥȤϿƤ뤫å
		# 줿 Agent Ͽ
		# Ƥʤ硢IPMSG_BR_ENTRY  IPMSG_GETPUBKEY 

		my $peerhost = $HostList->get( PeerAddr => $recv->addr, PeerPort => $recv->port );

		# --------------------
		# ۥȤϿƤ

		if( $peerhost ){

			$peerhost->pubkey( $recv->ext );

		# --------------------
		# ۥȤϿƤʤä

		}else{

			$recv->reply->nickname( $self->hostinfo->nickname )->group( $self->hostinfo->group )->entry->queue( Mode => $Mode );

			my ( $power, $exp, $key ) = $self->hostinfo->pubkey;

			if( $power ){
				$self->wait_queue( $recv->reply->getpubkey->ext( $power )->queue( Mode => $Mode ) );
			}

		}

		# --------------------
		# return

		return $self;

	} # End of IPMSG_ANSPUBKEY

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub2: _check_waitqueue

sub _check_waitqueue
{
	my ( $self ) = shift;

	# --------------------
	# check WaitQueue

	foreach my $key ( keys %{ $self->[ WaitQueue ] } ){

			#	print $key,"\n";
			#	print time(),"\n";
			#	print "Count         => ", $self->[ WaitQueue ]{ $key }->[ 0 ] ,"\n";
			#	print "RetryMAX      => ", $self->[ WaitQueue ]{ $key }->[ 1 ] ,"\n";
			#	print "time          => ", $self->[ WaitQueue ]{ $key }->[ 2 ] ,"\n";
			#	print "RetryInterval => ", $self->[ WaitQueue ]{ $key }->[ 3 ] ,"\n";
			#	print "Messege Obj   => ", $self->[ WaitQueue ]{ $key }->[ 4 ] ,"\n";
			#	print "Expire        => ", $self->[ WaitQueue ]{ $key }->[ 5 ] ,"\n";

		# --------------------
		# Expire
		#
		# Expire  time() 꾮 Expire  0 ǤϤʤ WaitQueue 
		#

		if(
			$self->[ WaitQueue ]{ $key }->[ 5 ] <= time() &&
			$self->[ WaitQueue ]{ $key }->[ 5 ] != 0
		){

			# --------------------
			# expire

			push @{ $self->[ NoResponce ] }, $self->[ WaitQueue ]{ $key }->[ 4 ];
			$self->_del_waitqueue( $key );

			# --------------------
			# next

			next;
		}

		# --------------------
		# retry send

		if(
			$self->[ WaitQueue ]{ $key }->[ 0 ] < $self->[ WaitQueue ]{ $key }->[ 1 ]
			&&
			$self->[ WaitQueue ]{ $key }->[ 2 ] + $self->[ WaitQueue ]{ $key }->[ 3 ] <= time()
		){

			# --------------------
			# count up, update retry time and retry

			$self->[ WaitQueue ]{ $key }->[ 0 ]++;
			$self->[ WaitQueue ]{ $key }->[ 2 ] = time();
			$self->[ WaitQueue ]{ $key }->[ 4 ]->queue;

		}

	}

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: å

# --------------------
# Sub1: message

sub message
{
	my ( $self, %options ) = @_;
	my $NetIF;

	# --------------------
	# check requirements

	# --------------------
	# set default NetIF

	if( ref( $options{ NetIF } ) eq 'Ipmsg::Net' ){
		$NetIF = $options{ NetIF }; delete $options{ NetIF };
	}else{
		$NetIF = $self->netif( 0 );
	}

	# --------------------
	# make Message::Send object

	my $send = $NetIF->message( %options );

	# --------------------
	# return

	return $send;
}

# --------------------
# Sub1: broadcast

sub broadcast
{
	my ( $self, %options ) = @_;

	# --------------------
	# make broadcast list

	my @tmp;

	for my $i ( $self->netif ){
		push @tmp, scalar $self->netif->get_inaddrbroadcast, $self->hostinfo->port, $i;
	}

	push @tmp, $self->get_broadcast;
	push @tmp, $self->get_dialup;

	# --------------------
	# make messages

	for( my $i = 0; $i <= $#tmp; $i = $i + 3 ){

		$self->message(
			NetIF		=> $tmp[ $i + 2 ],
			PeerAddr	=> $tmp[ $i ],
			PeerPort	=> $tmp[ $i + 1 ],
			Command		=> $options{ Command },
			Ext			=> $options{ Ext },
		)->queue;

	}

	# --------------------
	# set code

	$self->[ Code ] = $#tmp + 1;

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: Login

sub Login
{
	my ( $self ) = shift;

	# --------------------
	# return

	my $EXT;
	$EXT  = $self->hostinfo->nickname;
	$EXT .= $self->delim_group . $self->hostinfo->group;

	return $self->broadcast( Command => $self->ipmsg_cmd( 'IPMSG_BR_ENTRY' ) + $self->hostinfo->command, Ext => $EXT );
}

# --------------------
# Sub1: Logout

sub Logout
{
	my $self = shift;

	# --------------------
	# return

	return $self->broadcast( Command => 'IPMSG_BR_EXIT' );
}

# --------------------
# Name: 塼

# --------------------
# Sub1: remainqueue

sub remainqueue
{
	my ( $self, %options ) = @_;
	my $NetIF;

	# --------------------
	# check requirements

	# --------------------
	# set default NetIF

	if( ref( $options{ NetIF } ) eq 'Ipmsg::Net' ){
		$NetIF = $options{ NetIF }; delete $options{ NetIF };
	}else{
		$NetIF = $self->netif( 0 );
	}

	# --------------------
	# return

	return $NetIF->remainqueue;
}

# --------------------
# Sub1: send_queue

sub send_queue
{
	my ( $self, %options ) = @_;
	my $NetIF;

	# --------------------
	# check requirements

	$options{ CHECK_WAITQUEUE } = 1 if ! defined $options{ CHECK_WAITQUEUE };

	# --------------------
	# set default NetIF

	if( ref( $options{ NetIF } ) eq 'Ipmsg::Net' ){
		$NetIF = $options{ NetIF }; delete $options{ NetIF };
	}else{
		$NetIF = $self->netif( 0 );
	}

	# --------------------
	# CHECK_WAITQUEUE option

	$self->_check_waitqueue if $options{ CHECK_WAITQUEUE } ;

	# --------------------
	# return

	return $NetIF->send_queue;
}

# --------------------
# Name: 塼

# --------------------
# Sub1: wait_queue

sub wait_queue
{
	my ( $self, $msg, %options ) = @_;
	my $key;

	# --------------------
	# check requirements

	if( ! defined $msg ){ return keys %{ $self->[ WaitQueue ] }; }

	if( ! defined $options{ RetryMAX } )		{ $options{ RetryMAX }		= $self->retrymax; }
	if( ! defined $options{ RetryInterval } )	{ $options{ RetryInterval }	= $self->retryinterval; }
	if( ! defined $options{ Expire } )			{ $options{ Expire } = $options{ RetryMAX } * $options{ RetryInterval }; }

# --------------------
# Sub2: Message::Send

	if( ref( $msg ) eq "Ipmsg::Message::Send" ){

		# --------------------
		# if $msg is IPMSG_AUTORETOPT or IPMSG_BROADCASTOPT

		if(
			$msg->opt & $msg->ipmsg_cmd( 'IPMSG_AUTORETOPT' ) ||
			$msg->opt & $msg->ipmsg_cmd( 'IPMSG_BROADCASTOPT' )
		){
			return $self;
		}

		# --------------------
		# if $msg is IPMSG_BR_ISGETLIST or IPMSG_BR_ISGETLIST2 => IPMSG_OKGETLIST

		if(
			$msg->mode == $msg->ipmsg_cmd( 'IPMSG_BR_ISGETLIST' ) ||
			$msg->mode == $msg->ipmsg_cmd( 'IPMSG_BR_ISGETLIST2' )
		 ){

			$options{ Expire } = $self->expirelistget if ! $options{ Expire };

			$self->_add_waitqueue( $msg,
				RetryMAX		=> 0,
				RetryInterval	=> 0,
				Expire			=> $options{ Expire },
				Command			=> $msg->ipmsg_cmd( 'IPMSG_OKGETLIST' ),
			);
			return $self;

		}

		# --------------------
		# if $msg is IPMSG_GETLIST => IPMSG_ANSLIST

		if( $msg->mode == $msg->ipmsg_cmd( 'IPMSG_GETLIST' ) ){

			$self->_add_waitqueue( $msg,
				RetryMAX		=> $options{ RetryMAX },
				RetryInterval	=> $options{ RetryInterval },
				Expire			=> $options{ Expire },
				Command			=> $msg->ipmsg_cmd( 'IPMSG_ANSLIST' ),
			);
			return $self;

		}

		# --------------------
		# if $msg is IPMSG_SENDMSG + IPMSG_SENDCHECKOPT => IPMSG_RECVMSG

		if(
			$msg->mode == $msg->ipmsg_cmd( 'IPMSG_SENDMSG' ) &&
			$msg->opt  &  $msg->ipmsg_cmd( 'IPMSG_SENDCHECKOPT' )
		){

			$self->_add_waitqueue( $msg,
				RetryMAX		=> $options{ RetryMAX },
				RetryInterval	=> $options{ RetryInterval },
				Expire			=> $options{ Expire },
				Command			=> $msg->ipmsg_cmd( 'IPMSG_RECVMSG' ),
				Ext				=> $msg->packetn,
			);
			return $self;

		}

		# --------------------
		# if $msg is IPMSG_GETINFO => IPMSG_SENDINFO

		if( $msg->mode == $msg->ipmsg_cmd( 'IPMSG_GETINFO' ) ){

			$self->_add_waitqueue( $msg,
				RetryMAX		=> $options{ RetryMAX },
				RetryInterval	=> $options{ RetryInterval },
				Expire			=> $options{ Expire },
				Command			=> $msg->ipmsg_cmd( 'IPMSG_SENDINFO' ),
			);
			return $self;

		}

		# --------------------
		# if $msg is IPMSG_GETABSENCEINFO => IPMSG_SENDABSENCEINFO

		if( $msg->mode == $msg->ipmsg_cmd( 'IPMSG_GETABSENCEINFO' ) ){

			$self->_add_waitqueue( $msg,
				RetryMAX		=> $options{ RetryMAX },
				RetryInterval	=> $options{ RetryInterval },
				Expire			=> $options{ Expire },
				Command			=> $msg->ipmsg_cmd( 'IPMSG_SENDABSENCEINFO' ),
			);
			return $self;

		}

		# --------------------
		# if $msg is IPMSG_GETPUBKEY => IPMSG_ANSPUBKEY

		if( $msg->mode == $msg->ipmsg_cmd( 'IPMSG_GETPUBKEY' ) ){

			$self->_add_waitqueue( $msg,
				RetryMAX		=> $options{ RetryMAX },
				RetryInterval	=> $options{ RetryInterval },
				Expire			=> $options{ Expire },
				Command			=> $msg->ipmsg_cmd( 'IPMSG_ANSPUBKEY' ),
			);
			return $self;

		}

	} # End of Message::Send

# --------------------
# Sub2: Message::Recv

	if( ref( $msg ) eq "Ipmsg::Message::Recv" ){

		# --------------------
		# if $msg is IPMSG_ANSLIST

		if( $msg->anslist ){
			return $self->_del_waitqueue( $msg->addr . ":" . $msg->port . ":" . $msg->ipmsg_cmd( 'IPMSG_ANSLIST' ) );
		}

		# --------------------
		# if $msg is IPMSG_OKGETLIST

		if( $msg->okgetlist ){
			return $self->_del_waitqueue( $msg->addr . ":" . $msg->port . ":" . $msg->ipmsg_cmd( 'IPMSG_OKGETLIST' ) );
		}

		# --------------------
		# if $msg is IPMSG_RECVMSG

		if( $msg->recvmsg ){
			return $self->_del_waitqueue( $msg->addr . ":" . $msg->port . ":" . $msg->ipmsg_cmd( 'IPMSG_RECVMSG' ) . ":" . $msg->ext );
		}

		# --------------------
		# if $msg is IPMSG_SENDINFO

		if( $msg->sendinfo ){
			return $self->_del_waitqueue( $msg->addr . ":" . $msg->port . ":" . $msg->ipmsg_cmd( 'IPMSG_SENDINFO' ) );
		}

		# --------------------
		# if $msg is IPMSG_SENDABSENCEINFO

		if( $msg->sendabsenceinfo ){
			return $self->_del_waitqueue( $msg->addr . ":" . $msg->port . ":" . $msg->ipmsg_cmd( 'IPMSG_SENDABSENCEINFO' ) );
		}

		# --------------------
		# if $msg is IPMSG_ANSPUBKEY

		if( $msg->anspubkey ){
			return $self->_del_waitqueue( $msg->addr . ":" . $msg->port . ":" . $msg->ipmsg_cmd( 'IPMSG_ANSPUBKEY' ) );
		}

	} # End of Message::Recv

	# --------------------
	# return

	return $self;
}

# --------------------
# Sub1: _add_waitqueue

sub _add_waitqueue
{
	my ( $self, $msg, %options ) = @_;

	# --------------------
	# check requirements

	if( ! defined $msg )						{ return undef; }

	if( ! defined $options{ RetryMAX } )		{ $options{ RetryMAX }		= $self->retrymax; }
	if( ! defined $options{ RetryInterval } )	{ $options{ RetryInterval }	= $self->retryinterval; }
	if( ! defined $options{ Expire } )			{ $options{ Expire }		= ( $options{ RetryMAX } + 1 ) * $options{ RetryInterval }; }
	if( $options{ Expire } )					{ $options{ Expire }	   += time(); }

	# --------------------
	# make key

	my $key = join( ":",
		$msg->addr,
		$msg->port,
		$options{ Command },
	);

	$key .= ":" . $options{ Ext } if defined $options{ Ext };

	#	print "_add_waitqueue : $key\n";
	#	print "  now          : ", time(),"\n";
	#	print "  RetryMAX     : ", $options{ RetryMAX },"\n";
	#	print "  RetryInterval: ", $options{ RetryInterval },"\n";
	#	print "  Expire       : ", $options{ Expire },"\n";

	# --------------------
	# push Message::Send to WaitQueue

	$self->[ WaitQueue ]{ $key } =
		[
			0,							# Retry count
			$options{ RetryMAX },		# Retry Maximum
			time(),						# Retry time
			$options{ RetryInterval },	# Retry Interval
			$msg,						# Send object
			$options{ Expire },			# Expire
		];

	# --------------------
	# return

	return $msg;
}

# --------------------
# Sub1: _del_waitqueue

sub _del_waitqueue
{
	my ( $self, $key ) = @_;

	# --------------------
	# check requirements

	return undef if ! defined $key;

	#	print "       Message key : $key\n";

	# --------------------
	# push this message to NoResponce and delete WaitQueue

	if( exists $self->[ WaitQueue ]{ $key } ){

	#	print "_del_waitqueue : $key\n";
	#	print "  Expire       : ", time(),"\n";

		return delete $self->[ WaitQueue ]{ $key };

	}else{

		return undef;

	}
}

# --------------------
# Name: absence

sub absence
{
	my ( $self ) = shift;

	# --------------------
	# absence

	if( $self->hostinfo->absence( 1 ) ){

		$self->broadcast(
			Command	=> $self->ipmsg_cmd( 'IPMSG_BR_ABSENCE' ) + $self->ipmsg_cmd( 'IPMSG_ABSENCEOPT' ),
			Ext		=> substr( $self->hostinfo->nickname . $self->hostinfo->absencehead . $self->delim_group . $self->hostinfo->group, 0, $self->namebuf ),
		);

	# --------------------
	# presence

	}else{

		$self->broadcast(
			Command	=> $self->ipmsg_cmd( 'IPMSG_BR_ABSENCE' ),
			Ext		=> substr( $self->hostinfo->nickname . $self->delim_group . $self->hostinfo->group, 0, $self->namebuf ),
		);

	}

	# --------------------
	# return

	return $self->hostinfo->absence;
}

# --------------------
# Category: Setting object

# --------------------
# Name: put_value

sub put_value
{
	my $self= shift;
	my %arg	= @_;

	# --------------------
	# check requirements

	return undef if ! %arg;

	while( my ( $name, $value ) = each ( %arg ) ){
		$self->[ $_index{ $name } ] = $value;
	}

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: get_value

sub get_value
{
	my ( $self, $arg ) = @_;

	# --------------------
	# return

	return ( wantarray ? sort keys %_index : $self->[ $_index{ $arg } ] );
}

# --------------------
# Name: display

sub display
{
	my $self = shift;

	# --------------------
	# print

	foreach my $i ( sort keys %_index ){
		print "$i=>",$self->[ $_index{ $i } ],"\n";
	}

	# --------------------
	# return

	return $self;
}

# --------------------
# Name: code

sub code
{
	my $self = shift;

	# --------------------
	# return

	return $self->[ Code ];
}

# --------------------
# Name: _index

sub _index
{
	my ( $self, $index ) = @_;

	# --------------------
	# return

	return $_index{ $index };
}

1;

