package junit.extensions.eclipse.quick;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.junit.wizards.JUnitWizard;
import org.eclipse.jdt.internal.junit.wizards.NewTestCaseCreationWizard;
import org.eclipse.jdt.internal.ui.actions.SelectionConverter;
import org.eclipse.jdt.internal.ui.wizards.NewClassCreationWizard;
import org.eclipse.jdt.internal.ui.wizards.NewElementWizard;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.internal.ide.IHelpContextIds;

public class OpenTestingPairAction extends QuickJUnitAction {

    private IType getTypeOfJavaEditor() throws JavaModelException {
        if (javaEditor == null)
            return null;
        return SelectionConverter.getTypeAtOffset(javaEditor);
    }

    private IType getTypeOfJavaElement() {
        return JavaElements.getPrimaryTypeOf(javaElement);
    }

    private IType getTargetType() throws JavaModelException {
        IType targetType = getTypeOfJavaEditor();
        if (targetType != null)
            return targetType;
        return getTypeOfJavaElement();
    }

    private static class TypeLabelProvider extends LabelProvider {
        public String getText(Object element) {
            return ((IType) element).getFullyQualifiedName();
        }
    }

    private IType selectType(IAction action, List types) {
        PopupTableSelector selector = new PopupTableSelector(getShell(), types);
        selector.setTitle(action.getText());
        selector.setCommandForward(action.getActionDefinitionId());
        selector.setLabelProvider(new TypeLabelProvider());
        return (IType) selector.select();
    }

    private void openPairWizard(IType targetType, String[] pairNames) throws JavaModelException {
        IWorkbench workbench = PlatformUI.getWorkbench();
        IWizard wizard = createNewPairWizard(targetType, workbench);
		Shell parent = workbench.getActiveWorkbenchWindow().getShell();
		WizardDialog dialog = new WizardDialog(parent, wizard);
		dialog.create();
		setUpNewClassCreationWizard(wizard, targetType, pairNames);
		WorkbenchHelp.setHelp(dialog.getShell(), IHelpContextIds.NEW_WIZARD_SHORTCUT);
		dialog.open();
    }

    private String extractTypeName(String fullName) {
        int index = fullName.lastIndexOf('.');
        if (index == -1)
            return fullName;
        return fullName.substring(index + 1);
    }
    
    private void setUpNewClassCreationWizard(IWizard wizard, IType targetType, String[] pairNames) {
        if (pairNames.length == 0)
            return;
        try {
            String typeName = extractTypeName(pairNames[0]);
            Field pageField = wizard.getClass().getDeclaredField("fPage");
            pageField.setAccessible(true);
            Object page = pageField.get(wizard);
            Method setTypeNameMethod = page.getClass().getMethod("setTypeName",
                    new Class[] { String.class, boolean.class });
            setTypeNameMethod.setAccessible(true);
            setTypeNameMethod.invoke(page, new Object[]{typeName, Boolean.TRUE});
        } catch (Exception e) {
            // TODO: Oɏ
            e.printStackTrace();
        }
    }

    private IWizard createNewPairWizard(IType targetType, IWorkbench workbench) throws JavaModelException {
         if (JavaTypes.isTest(targetType)) {
            NewElementWizard wizard = new NewClassCreationWizard();
            IStructuredSelection selection = new StructuredSelection(targetType.getPackageFragment());
            wizard.init(workbench, selection);
            return wizard;
        }
        else {
            JUnitWizard wizard = new NewTestCaseCreationWizard();
            IStructuredSelection selection = new StructuredSelection(targetType);
            wizard.init(workbench, selection);
            return wizard;
        }
    }

    public void run(IAction action) {
        try {
            IType targetType = getTargetType();
            if (targetType == null)
                return;

            String[] pairNames = createTestPair().getPairClassNames(targetType.getFullyQualifiedName());
            List testPairTypes = findPairTypes(pairNames);
            if (testPairTypes.isEmpty()) {
                runWhenTestingPairNotExist(action, targetType, pairNames);
            } else {
                openTestingPair(action, testPairTypes);
            }
        } catch (JavaModelException e) {
            QuickJUnitPlugin.getDefault().handleSystemError(e, this);
        } catch (CoreException e) {
            QuickJUnitPlugin.getDefault().handleSystemError(e, this);
        }
    }

    private List findPairTypes(String[] pairNames) throws JavaModelException {
        IJavaProject[] projects = getJavaProjects();
        Set result = new LinkedHashSet();
        for (int i = 0; i < projects.length; ++i) {
            IJavaProject project = projects[i];
            for (int j = 0; j < pairNames.length; ++j) {
                IType pairType= project.findType(pairNames[j]);
                if (pairType != null) {
                    result.add(pairType);
                }
            }
        }
        return new ArrayList(result);
    }

    private void runWhenTestingPairNotExist(IAction action, IType targetType, String[] pairNames) throws JavaModelException {
        if (pairNames.length == 0) {
            openInformation(action, Messages.getString("OpenTestingPairAction.testPairNotFound", targetType.getFullyQualifiedName()));
            return;
        }
        if (MessageDialog.openQuestion(getShell(), action.getText(), Messages.getString("OpenTestingPairAction.testPairNotFoundAndConfirm"))) {
            openPairWizard(targetType, pairNames);
        }
    }

    private void openTestingPair(IAction action, List testPairTypes) throws CoreException {
        IType selectedType = null;
        if (testPairTypes.size() >= 2) {
            selectedType = selectType(action, testPairTypes);
        } else {
            selectedType = (IType) testPairTypes.get(0);
        }
        if (selectedType == null)
            return;
        JavaUI.openInEditor(selectedType);
    }

    private TestingPair createTestPair() {
        TestingPair testPair = new TestingPair();
        IPreferenceStore store = QuickJUnitPlugin.getDefault().getPreferenceStore();
        String[] rules = new NamingRules(store).getEnableValues();
        for (int i = 0; i < rules.length; ++i) {
            testPair.addNamingRule(rules[i]);            
        }
        return testPair;
    }
}
