package junit.extensions.eclipse.quick;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.actions.SelectionConverter;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.LabelProvider;

public class OpenTestingPairAction extends QuickJUnitAction {

    private static final String SPLITS_EDITOR_STORE_ID = "OpenTestingPairAction.splitsEditor";
    private static final boolean SPLITS_EDITOR_DEFAULT = false;

    private IType getTypeOfJavaEditor() throws JavaModelException {
        if (javaEditor == null)
            return null;
        return SelectionConverter.getTypeAtOffset(javaEditor);
    }

    private IType getTypeOfJavaElement() {
        return JavaElements.getPrimaryTypeOf(javaElement);
    }

    private static class TypeLabelProvider extends LabelProvider {
        public String getText(Object element) {
            return ((IType) element).getFullyQualifiedName();
        }
    }

    private IType selectType(IAction action, List types) {
        PopupTableSelector selector = new PopupTableSelector(getShell(), types);
        selector.setTitle(action.getText());
        selector.setCommandForward(action.getActionDefinitionId());
        selector.setLabelProvider(new TypeLabelProvider());
        return (IType) selector.select();
    }

    public void run(IAction action) {
        try {
            IType targetType = getTypeOfJavaEditor();
            if (targetType == null)
                targetType = getTypeOfJavaElement();
            if (targetType == null)
                return;

            IJavaProject[] projects = getJavaProjects();
            List testPairTypes = findPairTypes(createTestPair(), projects, targetType);
            if (testPairTypes.isEmpty()) {
                openInformation(action, Messages.getString("OpenTestingPairAction.testPairNotFound", targetType.getFullyQualifiedName())); //$NON-NLS-1$
                return;
            } else {
                IType selectedType = null;
                if (testPairTypes.size() >= 2) {
                    selectedType = selectType(action, testPairTypes);
                } else {
                    selectedType = (IType) testPairTypes.get(0);
                }
                if (selectedType != null)
                    openInEditor(selectedType);  
            }
        } catch (JavaModelException e) {
            QuickJUnitPlugin.getDefault().handleSystemError(e, this);
        } catch (CoreException e) {
            QuickJUnitPlugin.getDefault().handleSystemError(e, this);
        }
    }

    private void openInEditor(IType selectedType) throws CoreException {
        if (javaEditor == null || !splitsEditor()) {
            JavaUI.openInEditor(selectedType);
            return;
        }

        IResource rsrc = selectedType.getResource();
        if (!(rsrc instanceof IFile)) {
            JavaUI.openInEditor(selectedType);
            return;
        }
        EditorSplitter splitter = new EditorSplitter(javaEditor);
        splitter.open((IFile) rsrc);
    }

    private TestingPair createTestPair() {
        TestingPair testPair = new TestingPair();
        IPreferenceStore store = QuickJUnitPlugin.getDefault().getPreferenceStore();
        String[] rules = new NamingRules(store).getEnableValues();
        for (int i = 0; i < rules.length; ++i) {
            testPair.addNamingRule(rules[i]);            
        }
        return testPair;
    }

    private List findPairTypes(TestingPair testPair, IJavaProject[] projects, IType type) throws JavaModelException {
        Set result = new LinkedHashSet();
        String[] pairNames = testPair.getPairClassNames(type.getFullyQualifiedName());
        for (int i = 0; i < projects.length; ++i) {
            IJavaProject project = projects[i];
            for (int j = 0; j < pairNames.length; ++j) {
                IType pairType= project.findType(pairNames[j]);
                if (pairType != null) {
                    result.add(pairType);
                }
            }
        }
        return new ArrayList(result);
    }

    private static void beforeAccessPreferenceStore(IPreferenceStore store) {
        if (!store.contains(SPLITS_EDITOR_STORE_ID))
            store.setDefault(SPLITS_EDITOR_STORE_ID, SPLITS_EDITOR_DEFAULT);
    }

    static boolean splitsEditor(IPreferenceStore store) {
        beforeAccessPreferenceStore(store);
        return store.getBoolean(SPLITS_EDITOR_STORE_ID);
    }

    public static boolean splitsEditor() {
        return splitsEditor(QuickJUnitPlugin.getDefault().getPreferenceStore());
    }

    static void setSplitsEditor(IPreferenceStore store, boolean value) {
        beforeAccessPreferenceStore(store);
        store.setValue(SPLITS_EDITOR_STORE_ID, value);
    }
    
    public static void setSplitsEditor(boolean value) {
        setSplitsEditor(QuickJUnitPlugin.getDefault().getPreferenceStore(), value);
    }

    static boolean getDefaultSplitsEditor(IPreferenceStore store) {
        beforeAccessPreferenceStore(store);
        return store.getDefaultBoolean(SPLITS_EDITOR_STORE_ID);
    }

    public static boolean getDefalutSplitsEditor() {
        return getDefaultSplitsEditor(QuickJUnitPlugin.getDefault().getPreferenceStore());
    }
}
