package junit.extensions.eclipse.quick;

import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.internal.EditorArea;
import org.eclipse.ui.internal.EditorPane;
import org.eclipse.ui.internal.EditorPresentation;
import org.eclipse.ui.internal.EditorWorkbook;
import org.eclipse.ui.internal.ILayoutContainer;
import org.eclipse.ui.internal.LayoutPart;
import org.eclipse.ui.internal.RootLayoutContainer;
import org.eclipse.ui.internal.WorkbenchPage;

public class EditorSplitter {
    private IWorkbenchPage page;
    private EditorArea area;
    
    public EditorSplitter(IWorkbenchPage page) throws CoreException {
        this.page = page;
        this.area = getEditorArea(page);
    }

    public EditorSplitter(IWorkbenchPart part) throws CoreException {
        this(part.getSite().getPage());
    }

    private static EditorArea getEditorArea(IWorkbenchPage page) throws CoreException {
        if (!(page instanceof WorkbenchPage))
            throw new WorkbenchException("cannot get WorkbenchPage object.");
        EditorPresentation presentation = ((WorkbenchPage) page).getEditorPresentation();
        return (EditorArea) presentation.getLayoutPart();
    }

    public void open(IFile file) throws CoreException {
        IEditorPart currentEditor = page.getActiveEditor();

        EditorPane thisPane = getEditorPane(currentEditor);
        RootLayoutContainer root = area.getRootContainer();
        boolean zoomed = root.isZoomed();

        IEditorPart otherEditor = page.openEditor(file, null, false);
        EditorPane otherPane = getEditorPane(otherEditor);

        if (otherEditor == currentEditor) {
            otherPane.setFocus();
            otherPane.showFocus(true);
            return;
        }

        root.getControl().setRedraw(false);
        try {
            EditorWorkbook newWorkbook = null;

            if (thisPane.getWorkbook() == otherPane.getWorkbook()) {
                removeLayoutPart(otherPane);
                EditorWorkbook insertWorkbook = findWorkbook(thisPane.getWorkbook());
                if (insertWorkbook == null) {
                    insertWorkbook = new EditorWorkbook(area);
                    newWorkbook = insertWorkbook;
                }
                insertWorkbook.add(otherPane);
            } else if (zoomed && !otherPane.isVisible()) {
                newWorkbook = otherPane.getWorkbook();
                removeLayoutPart(newWorkbook);
            }

            if (newWorkbook != null) {
                area.add(newWorkbook, IPageLayout.BOTTOM, 0.5f, thisPane.getWorkbook());
            }

            if (zoomed) {
                root.zoomIn(area);
            }
        } finally {
            root.getControl().setRedraw(true);
            otherPane.setFocus();
            otherPane.showFocus(true);
        }
    }

    private EditorPane getEditorPane(IEditorPart editorPart) {
        List workbooks = area.getEditorWorkbooks();
        for (Iterator i = workbooks.iterator(); i.hasNext(); ) {
            EditorWorkbook workbook = (EditorWorkbook) i.next();
            EditorPane[] panes = workbook.getEditors();
            for (int j = 0; j < panes.length; j++) {
                IEditorReference reference = panes[j].getEditorReference();
                if (reference == null)
                    continue;
                if (editorPart == reference.getEditor(false))
                    return panes[j];
            }
        }
        return null;
    }

    private void removeLayoutPart(LayoutPart part) {
        ILayoutContainer oldContainer = part.getContainer();
        part.reparent(area.getParent());
        if (oldContainer == null) 
            return;
        oldContainer.remove(part);
        LayoutPart[] children = oldContainer.getChildren();
        if (children == null || children.length == 0){
            if (oldContainer instanceof LayoutPart) {
                LayoutPart parent = (LayoutPart)oldContainer;
                ILayoutContainer parentContainer = parent.getContainer();
                if (parentContainer != null) {
                    parentContainer.remove(parent);
                    parent.dispose();
                }
            }
        }
    }

    private EditorWorkbook findWorkbook(EditorWorkbook excludedWorkbook) {
        List workbooks = area.getEditorWorkbooks();
        for (Iterator i = workbooks.iterator(); i.hasNext(); ) {
            EditorWorkbook workbook = (EditorWorkbook) i.next();
            if (excludedWorkbook != workbook) {
                return workbook;
            }
        }
        return null;
    }
}
