package junit.extensions.eclipse.quick;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.internal.commands.Manager;
import org.eclipse.ui.internal.commands.Sequence;
import org.eclipse.ui.internal.commands.Stroke;

public class PopupTableSelector {
    private Shell shell;
    private List items;
    private Object selection;
    private String commandForward;
    private String commandBackward;
    private ILabelProvider labelProvider;
    private String title = "";  //$NON-NLS-1$

    public PopupTableSelector(Shell shell, List items) {
        this.shell = shell;
        this.items = items;
    }

    public void setCommandBackward(String string) {
        commandBackward = string;
    }

    public void setCommandForward(String string) {
        commandForward = string;
    }

    public void setTitle(String string) {
        title = string;
    }

    public void setLabelProvider(ILabelProvider provider) {
        labelProvider = provider;
    }

    public Object select() {
        final int MAX_ITEMS = 22;

        selection = null;
        final Shell dialog = new Shell(shell, SWT.MODELESS);
        Display display = dialog.getDisplay();
        dialog.setLayout(new FillLayout());

        final Table table = new Table(dialog, SWT.SINGLE | SWT.FULL_SELECTION);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);
        TableColumn tc = new TableColumn(table, SWT.NONE);
        tc.setResizable(false);
        tc.setText(title);
        addItems(table, items);
        int tableItemCount = table.getItemCount();
        if (tableItemCount > 0)
            table.setSelection(0);
        tc.pack();
        table.pack();
        Rectangle tableBounds = table.getBounds();
        tableBounds.height = Math.min(tableBounds.height, table.getItemHeight() * MAX_ITEMS);
        table.setBounds(tableBounds);
        dialog.pack();

        tc.setWidth(table.getClientArea().width);
        table.setFocus();
        table.addFocusListener(new FocusListener() {
            public void focusGained(FocusEvent e) {
            }

            public void focusLost(FocusEvent e) {
                cancel(dialog);
            }
        });

        Rectangle dialogBounds = dialog.getBounds();
        Rectangle displayBounds = display.getClientArea();
        Rectangle parentBounds = dialog.getParent().getBounds();

        //Place it in the center of its parent;
        dialogBounds.x = parentBounds.x + ((parentBounds.width - dialogBounds.width) / 2);
        dialogBounds.y = parentBounds.y + ((parentBounds.height - dialogBounds.height) / 2);
        if (!displayBounds.contains(dialogBounds.x, dialogBounds.y)
            || !displayBounds.contains(
                dialogBounds.x + dialogBounds.width,
                dialogBounds.y + dialogBounds.height)) {
            //Place it in the center of the display if it is not visible
            //when placed in the center of its parent;
            dialogBounds.x = (displayBounds.width - dialogBounds.width) / 2;
            dialogBounds.y = (displayBounds.height - dialogBounds.height) / 2;
        }
        dialogBounds.height = dialogBounds.height + 5 - table.getHorizontalBar().getSize().y;
        dialog.setBounds(dialogBounds);

        try {
            dialog.open();
            addMouseListener(table, dialog);
            addKeyListener(table, dialog);

            while (!dialog.isDisposed())
                if (!display.readAndDispatch())
                    display.sleep();
        } finally {
            if (!dialog.isDisposed())
                cancel(dialog);
        }
        return selection;
    }

    private void addItems(Table table, List items) {
        TableItem tableItem = null;
        for (int i = 0; i < items.size(); ++i) {
            Object item = items.get(i);
            tableItem = new TableItem(table, SWT.NONE);
            tableItem.setText(labelProvider.getText(item));
            tableItem.setData(item);
        }
    }

    private void addMouseListener(final Table table, final Shell dialog) {
        table.addMouseListener(new MouseListener() {
            public void mouseDoubleClick(MouseEvent e) {
                ok(dialog, table);
            }

            public void mouseDown(MouseEvent e) {
                ok(dialog, table);
            }

            public void mouseUp(MouseEvent e) {
                ok(dialog, table);
            }
        });
    }

    private void addKeyListener(final Table table, final Shell dialog) {
        table.addKeyListener(new KeyListener() {
            private boolean firstKey = true;
            private boolean quickReleaseMode = false;
                
            public void keyPressed(KeyEvent e) {
                int keyCode = e.keyCode;
                int stateMask = e.stateMask;
                char character = e.character;
                int accelerator = stateMask | (keyCode != 0 ? keyCode : convertCharacter(character));
    
                //System.out.println("\nPRESSED");
                //printKeyEvent(e);
                //System.out.println("accelerat:\t" + accelerator + "\t (" + KeySupport.formatStroke(Stroke.create(accelerator), true) + ")");              
                    
                boolean acceleratorForward = false;
                boolean acceleratorBackward = false;
    
                if (commandForward != null) {
                    Map commandMap = Manager.getInstance().getKeyMachine().getCommandMap();     
                    SortedSet sequenceSet = (SortedSet) commandMap.get(commandForward);
            
                    if (sequenceSet != null) {
                        Iterator iterator = sequenceSet.iterator();
                            
                        while (iterator.hasNext()) {
                            Sequence sequence = (Sequence) iterator.next();
                            List strokes = sequence.getStrokes();
                            int size = strokes.size();
    
                            if (size > 0 && accelerator == ((Stroke) strokes.get(size - 1)).getValue()) {
                                acceleratorForward = true;
                                break;
                            }
                        }
                    }
                }
    
                if (commandBackward != null) {
                    Map commandMap = Manager.getInstance().getKeyMachine().getCommandMap();     
                    SortedSet sequenceSet = (SortedSet) commandMap.get(commandBackward);
            
                    if (sequenceSet != null) {
                        Iterator iterator = sequenceSet.iterator();
                            
                        while (iterator.hasNext()) {
                            Sequence sequence = (Sequence) iterator.next();
                            List strokes = sequence.getStrokes();
                            int size = strokes.size();
    
                            if (size > 0 && accelerator == ((Stroke) strokes.get(size - 1)).getValue()) {
                                acceleratorBackward = true;
                                break;
                            }
                        }
                    }
                }
    
                if (character == SWT.CR || character == SWT.LF)
                    ok(dialog, table);
                else if (acceleratorForward) {
                    if (firstKey && e.stateMask != 0)
                        quickReleaseMode = true;                
                        
                    int index = table.getSelectionIndex();
                    table.setSelection((index + 1) % table.getItemCount());
                } else if (acceleratorBackward) {
                    if (firstKey && e.stateMask != 0)
                        quickReleaseMode = true;                    
    
                    int index = table.getSelectionIndex();
                    table.setSelection(index >= 1 ? index - 1 : table.getItemCount() - 1);
                } else if (keyCode != SWT.ALT && keyCode != SWT.COMMAND && keyCode != SWT.CTRL && keyCode != SWT.SHIFT && 
                    keyCode != SWT.ARROW_DOWN && keyCode != SWT.ARROW_UP && keyCode != SWT.ARROW_LEFT && keyCode != SWT.ARROW_RIGHT)
                    cancel(dialog);
    
                firstKey = false;
            }
                
            public void keyReleased(KeyEvent e) {       
                int keyCode = e.keyCode;
                int stateMask = e.stateMask;
                // char character = e.character;
                // int accelerator = stateMask | (keyCode != 0 ? keyCode : convertCharacter(character));
    
                //System.out.println("\nRELEASED");
                //printKeyEvent(e);
                //System.out.println("accelerat:\t" + accelerator + "\t (" + KeySupport.formatStroke(Stroke.create(accelerator), true) + ")");
                    
                if ((firstKey || quickReleaseMode) && keyCode == stateMask)
                    ok(dialog, table);
            }
        });
    }

    private static char convertCharacter(char c) {
        return c >= 0 && c <= 31 ? (char) (c + '@') : Character.toUpperCase(c);
    }

    private void cancel(Shell dialog) {
        selection = null;
        dialog.close();
    }

    private void ok(Shell dialog, final Table table) {
        TableItem[] items = table.getSelection();
        if (items != null && items.length == 1)
            selection = items[0].getData();
        dialog.close();
    }
}
