// YUVReader.h
// 2009/05/19

#pragma once

// YUVReader
class YUVReader {

	FILE* m_fp;

	INT32 m_CX;
	INT32 m_CY;

	UINT8* m_Y;
	UINT8* m_U;
	UINT8* m_V;

public:

	YUVReader() : m_fp(0), m_CX(0), m_CY(0), m_Y(0), m_U(0), m_V(0)
	{
	}

	~YUVReader()
	{
		if (m_fp != 0 && m_fp != stdin) {
			fclose(m_fp);
		}

		free(m_Y);
		free(m_U);
		free(m_V);
	}

	bool Open(LPCWSTR path)
	{
		FILE* fp = 0;

		if (wcscmp(path, L"-") == 0) {
			_setmode(
				_fileno(stdin),
				_O_BINARY);

			fp = stdin;

		} else {
			_wfopen_s(&fp, path, L"rb");
			if (fp == 0) {
				return false;
			}
		}

		m_fp = fp;

		char buf[80+1]; buf[80] = '\0';
		for (INT32 i = 0; i < 80; i++) {
			INT32 ch = fgetc(m_fp);
			if (ch == '\n' || ch == EOF) {
				buf[i] = '\0';
				break;
			}
			buf[i] = ch;
		}

		if (strncmp(buf, "YUV4MPEG2 ", 10) != 0) {
			return false;
		}

		int n = 10;
		while (buf[n] != '\0') {
			if (buf[n] == 'W') {
				break;
			}
			n++;
		}

		int cx = -1;
		int cy = -1;
		if (sscanf_s(buf + n, "W%d H%d", &cx, &cy) != 2) {
			return false;
		}

		if (cx <= 0 || cy <= 0) {
			return false;
		}

		m_CX = cx;
		m_CY = cy;

		m_Y = (UINT8*)malloc(m_CX * m_CY    );
		m_U = (UINT8*)malloc(m_CX * m_CY / 4);
		m_V = (UINT8*)malloc(m_CX * m_CY / 4);

		if (m_Y == 0 || m_U == 0 || m_V == 0) {
			return false;
		}

		return true;
	}

	bool ReadFrame()
	{
		char buf[80+1]; buf[80] = '\0';
		for (INT32 i = 0; i < 80; i++) {
			INT32 ch = fgetc(m_fp);
			if (ch == '\n' || ch == EOF) {
				buf[i] = '\0';
				break;
			}
			buf[i] = ch;
		}

		if (strncmp(buf, "FRAME", 5) != 0) {
			return false;
		}

		fread(m_Y, m_CX * m_CY,     1, m_fp);
		fread(m_U, m_CX * m_CY / 4, 1, m_fp);
		fread(m_V, m_CX * m_CY / 4, 1, m_fp);

		return true;
	}

	bool Compare(QT_Output_t* f)
	{
		INT32 x, y;

		const UINT8* Y = f->Plane[0] + m_CX * (m_CY - 1);
		const UINT8* U = f->Plane[1] + m_CX / 2 * (m_CY / 2 - 1);
		const UINT8* V = f->Plane[2] + m_CX / 2 * (m_CY / 2 - 1);

		const UINT8* R = m_Y;

		INT32 error = 0;

		for (y = 0; y < m_CY; y++, Y -= m_CX) {
			const UINT8* T = Y;
			for (x = 0; x < m_CX; x++, R++, T++) {
				if (*R != *T) {
					if (error < 3) {
						printf("Y: %d,%d : %02X -> %02X\n", x, y, *R, *T);
					}
					error++;
				}
			}
		}

		if (error > 2000) {
			INT32 pp = error * 100 / (m_CX * m_CY);
			printf("Y Count: %d : %d %%\n", error, pp);
			return false;
		}

		const UINT8* R0 = m_U;
		const UINT8* R1 = m_V;

		for (y = 0; y < m_CY / 2; y++, U -= m_CX / 2, V -= m_CX / 2) {
			const UINT8* T0 = U;
			const UINT8* T1 = V;
			for (x = 0; x < m_CX / 2; x++, R0++, T0++, R1++, T1++) {
				if (*R0 != *T0) {
					printf("U: %d,%d : %02X -> %02X\n", x, y, *R0, *T0);

					return false;
				}

				if (*R1 != *T1) {
					printf("V: %d,%d : %02X -> %02X\n", x, y, *R1, *T1);

					return false;
				}
			}
		}

		if (error > 0) {
			return false;
		}

		return true;
	}

}; // YUVReader

