// MainDialog.h
// 2009/05/29

#pragma once

#include "resource.h"

#include "QDecoder.h"

// MainDialog
class MainDialog :
	public ATL::CDialogImpl<MainDialog>,
	public WTL::CMessageFilter {

	enum {
		TID_TICK     =  1,
		TICK_TIMEOUT = 20
	};

	enum {
		WM_EVENT_NOTIFY = WM_APP + 1
	};

	enum {
		ID_MENU_DEFAULT = 100,
		ID_MENU_YUY2,
		ID_MENU_YV12,

		ID_MENU_ISTREAM
	};

	ATL::CComPtr<IGraphBuilder> m_FilterGraph;

	ATL::CComPtr<IBaseFilter> m_Decoder;

	DWORD m_dwCookie;

	INT32 m_VideoMode;

	bool m_bIStream;

public:

	/* */

	enum { IDD = IDD_MAIN };

	/* */

	MainDialog() : m_dwCookie(0), m_VideoMode(ID_MENU_DEFAULT), m_bIStream(true)
	{
	}

	~MainDialog()
	{
	}

	/* */

	BEGIN_MSG_MAP_EX(MainDialog)
		MESSAGE_HANDLER(WM_INITDIALOG, OnInitDialog)
		MESSAGE_HANDLER(WM_COMMAND,    OnCommand   )

		MSG_WM_DESTROY(OnDestroy)

		MSG_WM_TIMER(OnTimer)

		MSG_WM_CONTEXTMENU(OnContextMenu)

		MESSAGE_HANDLER(WM_DROPFILES,    OnDropFiles  )
		MESSAGE_HANDLER(WM_EVENT_NOTIFY, OnEventNotify)
	END_MSG_MAP()

	/* */

	LRESULT OnInitDialog(
		UINT   uMsg,
		WPARAM wParam,
		LPARAM lParam,
		BOOL&  bHandled)
	{
		SetWindowPos(
			0,
			0, 0,
			0, 0,
			SWP_NOZORDER | SWP_NOSIZE);

		SetDlgItemTextW(IDC_PATH,     L"");
		SetDlgItemTextW(IDC_VIDEO,    L"");
		SetDlgItemTextW(IDC_AUDIO,    L"");
		SetDlgItemTextW(IDC_DURATION, L"");
		SetDlgItemTextW(IDC_CURRENT,  L"");

		WTL::CButton b1(GetDlgItem(IDC_E_VIDEO));
		b1.SetCheck(1);

		WTL::CButton b2(GetDlgItem(IDC_E_AUDIO));
		b2.SetCheck(1);

		SetTimer(TID_TICK, TICK_TIMEOUT);

		DragAcceptFiles(TRUE);

		return TRUE;
	}

	LRESULT OnCommand(
		UINT   uMsg,
		WPARAM wParam,
		LPARAM lParam,
		BOOL&  bHandled)
	{
		int id   = LOWORD(wParam);
		int code = HIWORD(wParam);

		switch (id) {
		case IDC_OPEN:
			Open();
			break;

		case IDC_RUN:
			Run();
			break;

		case IDC_PAUSE:
			Pause();
			break;

		case IDC_RELEASE:
			ReleaseFilterGraph();
			SetDlgItemTextW(IDC_PATH,     L"");
			SetDlgItemTextW(IDC_VIDEO,    L"");
			SetDlgItemTextW(IDC_AUDIO,    L"");
			SetDlgItemTextW(IDC_DURATION, L"");
			SetDlgItemTextW(IDC_CURRENT,  L"");
			break;

		case IDOK:
		case IDCANCEL:
		case IDCLOSE:
			ReleaseFilterGraph();
			DestroyWindow();
			break;
		}

		return 0;
	}

	/* */

	void OnDestroy()
	{
		PostQuitMessage(0);
	}

	/* */

	void OnTimer(UINT_PTR nIDEvent)
	{
		if (nIDEvent == TID_TICK) {
			OnTick();
		}
	}

	/* */

	void OnContextMenu(HWND hwnd, WTL::CPoint pt)
	{
		WTL::CMenu menu;

		menu.CreatePopupMenu();

		menu.InsertMenu(0, MF_BYPOSITION | ((m_VideoMode == ID_MENU_DEFAULT) ? MF_CHECKED : 0), ID_MENU_DEFAULT, L"Video Default");
		menu.InsertMenu(1, MF_BYPOSITION | ((m_VideoMode == ID_MENU_YUY2)    ? MF_CHECKED : 0), ID_MENU_YUY2,    L"Video YUY2"   );
		menu.InsertMenu(2, MF_BYPOSITION | ((m_VideoMode == ID_MENU_YV12)    ? MF_CHECKED : 0), ID_MENU_YV12,    L"Video YV12"   );
		menu.InsertMenu(3, MF_BYPOSITION | MF_SEPARATOR,                                        -1,              L""             );
		menu.InsertMenu(4, MF_BYPOSITION | (m_bIStream                       ? MF_CHECKED : 0), ID_MENU_ISTREAM, L"Use IStream"  );

		INT32 cmd = menu.TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RETURNCMD, pt.x, pt.y, hwnd);

		switch (cmd) {
		case ID_MENU_DEFAULT:
		case ID_MENU_YUY2:
		case ID_MENU_YV12:
			m_VideoMode = cmd;
			break;

		case ID_MENU_ISTREAM:
			m_bIStream = !m_bIStream;
			break;
		}
	}

	/* */

	virtual BOOL PreTranslateMessage(MSG* pMsg)
	{
		HWND hwnd = *this;

		if (hwnd != 0) {
			if (pMsg->message           == WM_KEYDOWN &&
				::GetParent(pMsg->hwnd) == hwnd) {
				ATLTRACE("DOWN\n");
				INT32 code = (INT32)pMsg->wParam;
				if (code >= '0' && code <= '9') {
					INT32 p = code - '0';
					Seek(p);
					return TRUE;
				}
			}

			if (::IsDialogMessageW(hwnd, pMsg)) {
				return TRUE;
			}
		}

		return FALSE;
	}

	/* */

	void OnTick()
	{
		if (m_FilterGraph != 0) {
			ATL::CComQIPtr<IMediaSeeking> seek(m_FilterGraph);

			INT64 pos = 0;
			seek->GetCurrentPosition(&pos);

			INT32 ds = (INT32)(pos / (1000*1000));
			INT32 ss = ds / 10;

			ATL::CStringW buf;
			buf.Format(L"%d:%02d:%d", ss / 60, ss % 60, ds % 10);

			SetDlgItemText(IDC_CURRENT, buf);
		}
	}

	/* */

private:

	/* */

	void ReleaseFilterGraph()
	{
		if (m_dwCookie != 0) {
			RemoveFromRot(m_dwCookie);
			m_dwCookie = 0;
		}

		if (m_FilterGraph != 0) {
			ATL::CComQIPtr<IMediaControl> ctrl(m_FilterGraph);
			ctrl->Stop();

			ATL::CComQIPtr<IMediaEventEx> me(m_FilterGraph);
			me->SetNotifyWindow(0, 0, 0);
		}

		m_Decoder.Release();

		m_FilterGraph.Release();
	}

	/* */

	void Open()
	{
		WTL::CFileDialog dlg(
			TRUE,
			L"mkv",
			0,
			OFN_HIDEREADONLY,
			L"Matroska Files (*.mkv;*.mka)\0*.mkv;*.mka\0All Files (*.*)\0*.*\0\0");

		if (dlg.DoModal() == IDOK) {
			ReleaseFilterGraph();
			Prepare(dlg.m_szFileName);
		}
	}

	/* */

	void Prepare(LPCWSTR path)
	{
		if (m_FilterGraph != 0) {
			return;
		}

		SetDlgItemTextW(IDC_PATH,     path);
		SetDlgItemTextW(IDC_VIDEO,    L"");
		SetDlgItemTextW(IDC_AUDIO,    L"");
		SetDlgItemTextW(IDC_DURATION, L"");
		SetDlgItemTextW(IDC_CURRENT,  L"");

		GetDlgItem(IDC_RUN)  .EnableWindow(FALSE);
		GetDlgItem(IDC_PAUSE).EnableWindow(FALSE);
		GetDlgItem(IDC_RUN)  .ShowWindow(SW_HIDE);
		GetDlgItem(IDC_PAUSE).ShowWindow(SW_HIDE);

		WTL::CButton b1(GetDlgItem(IDC_E_VIDEO));
		WTL::CButton b2(GetDlgItem(IDC_E_AUDIO));

		UpdateWindow();

		WTL::CWaitCursor wait;

		HRESULT hRslt = m_FilterGraph.CoCreateInstance(
			CLSID_FilterGraph,
			0,
			CLSCTX_INPROC_SERVER);
		if (FAILED(hRslt)) {
			return;
		}

		{
			ATL::CComQIPtr<IMediaEventEx> me(m_FilterGraph);

			HWND hwnd = *this;
			hRslt = me->SetNotifyWindow(
				(OAHWND)hwnd,
				WM_EVENT_NOTIFY,
				0);
			if (FAILED(hRslt)) {
				return;
			}
		}

		ATL::CComPtr<IBaseFilter> filter;
		if (m_bIStream) {
			ATL::CComPtr<IStream> stm;
			hRslt = SHCreateStreamOnFileW(
				path,
				STGM_READ,
				&stm);
			if (FAILED(hRslt)) {
				return;
			}

			hRslt = QD_CreateDecoder_IStream(
				stm,
				&filter);
			if (FAILED(hRslt)) {
				return;
			}

		} else {
			hRslt = QD_CreateDecoder(
				path,
				&filter);
			if (FAILED(hRslt)) {
				return;
			}
		}

		hRslt = m_FilterGraph->AddFilter(
			filter,
			L"QDecoder");
		if (FAILED(hRslt)) {
			return;
		}

		m_Decoder = filter;

		/* Video */

		if (b1.GetCheck() != 0) {
			ATL::CComPtr<IPin> pin;
			hRslt = filter->FindPin(L"Video", &pin);
			if (FAILED(hRslt)) {
				return;
			}

			ATL::CComPtr<IEnumMediaTypes> em;
			hRslt = pin->EnumMediaTypes(&em);
			if (FAILED(hRslt)) {
				return;
			}

			bool b = false;

			while (!b) {
				AM_MEDIA_TYPE* mt = 0;
				ULONG c = 0;
				hRslt = em->Next(1, &mt, &c);
				if (FAILED(hRslt)) {
					return;
				}
				if (mt == 0) {
					break;
				}

				if (mt->formattype == FORMAT_VideoInfo2 &&
					mt->pbFormat != 0 &&
					mt->cbFormat >= sizeof(VIDEOINFOHEADER2)) {

					VIDEOINFOHEADER2* v   = (VIDEOINFOHEADER2*)(mt->pbFormat);
					BITMAPINFOHEADER* bmi = &(v->bmiHeader);

					INT32 cx = bmi->biWidth;
					INT32 cy = bmi->biHeight;
					if (cy < 0) cy = -cy;

					ATL::CStringW buf;
					buf.Format(L"V: %d x %d", cx, cy);
					SetDlgItemTextW(IDC_VIDEO, buf);

					b = true;
				}

				QD_FreeMediaType(mt);
			}

			if (b) {
				ATL::CComPtr<IBaseFilter> rend;
				hRslt = rend.CoCreateInstance(
					CLSID_VideoMixingRenderer9,
					0,
					CLSCTX_INPROC_SERVER);
				if (FAILED(hRslt)) {
					return;
				}

				hRslt = m_FilterGraph->AddFilter(
					rend,
					L"V-Renderer");
				if (FAILED(hRslt)) {
					return;
				}

				ATL::CComPtr<IEnumPins> e;
				hRslt = rend->EnumPins(&e);
				if (FAILED(hRslt)) {
					return;
				}

				ATL::CComPtr<IPin> ip;
				ULONG c = 0;
				hRslt = e->Next(1, &ip, &c);
				if (FAILED(hRslt)) {
					return;
				}

				AM_MEDIA_TYPE mt = { 0 };

				AM_MEDIA_TYPE* pmt = 0;
				switch (m_VideoMode) {
				case ID_MENU_YUY2:
					mt.majortype = MEDIATYPE_Video;
					mt.subtype   = MEDIASUBTYPE_YUY2;
					pmt = &mt;
					break;

				case ID_MENU_YV12:
					mt.majortype = MEDIATYPE_Video;
					mt.subtype   = MEDIASUBTYPE_YV12;
					pmt = &mt;
					break;
				}

				hRslt = m_FilterGraph->ConnectDirect(
					pin,
					ip,
					pmt);
				if (FAILED(hRslt)) {
					ATLTRACE("ERROR ConnectDirect (V).\n");
					return;
				}

			} else {
				SetDlgItemTextW(IDC_VIDEO, L"NO Video");
			}
		}

		/* Audio */

		if (b2.GetCheck() != 0) {
			ATL::CComPtr<IPin> pin;
			hRslt = filter->FindPin(L"Audio", &pin);
			if (FAILED(hRslt)) {
				return;
			}

			ATL::CComPtr<IEnumMediaTypes> em;
			hRslt = pin->EnumMediaTypes(&em);
			if (FAILED(hRslt)) {
				return;
			}

			bool b = false;

			while (!b) {
				AM_MEDIA_TYPE* mt = 0;
				ULONG c = 0;
				hRslt = em->Next(1, &mt, &c);
				if (FAILED(hRslt)) {
					return;
				}
				if (mt == 0) {
					break;
				}

				if (mt->formattype == FORMAT_WaveFormatEx &&
					mt->pbFormat != 0 &&
					mt->cbFormat >= sizeof(WAVEFORMATEX)) {

					WAVEFORMATEX* w = (WAVEFORMATEX*)(mt->pbFormat);

					ATL::CStringW buf;
					buf.Format(L"A: %d ch %d Hz", w->nChannels, w->nSamplesPerSec);
					SetDlgItemTextW(IDC_AUDIO, buf);

					b = true;
				}

				QD_FreeMediaType(mt);
			}

			if (b) {
				ATL::CComPtr<IBaseFilter> rend;
				hRslt = rend.CoCreateInstance(
					CLSID_DSoundRender,
					0,
					CLSCTX_INPROC_SERVER);
				if (FAILED(hRslt)) {
					return;
				}

				hRslt = m_FilterGraph->AddFilter(
					rend,
					L"A-Renderer");
				if (FAILED(hRslt)) {
					return;
				}

				ATL::CComPtr<IEnumPins> e;
				hRslt = rend->EnumPins(&e);
				if (FAILED(hRslt)) {
					return;
				}

				ATL::CComPtr<IPin> ip;
				ULONG c = 0;
				hRslt = e->Next(1, &ip, &c);
				if (FAILED(hRslt)) {
					return;
				}

				hRslt = m_FilterGraph->ConnectDirect(
					pin,
					ip,
					0);
				if (FAILED(hRslt)) {
					ATLTRACE("ERROR ConnectDirect (A).\n");
					return;
				}

			} else {
				SetDlgItemTextW(IDC_AUDIO, L"NO Audio");
			}
		}

		/* */

		{
			ATL::CComQIPtr<IMediaSeeking> seek(m_FilterGraph);
			if (seek != 0) {
				INT64 duration = 0;
				seek->GetDuration(&duration);

				INT32 ds = (INT32)(duration / (1000*1000));
				INT32 ss = ds / 10;

				ATL::CStringW buf;
				buf.Format(L"%d:%02d:%d", ss / 60, ss % 60, ds % 10);

				SetDlgItemText(IDC_DURATION, buf);
			}
		}

		{
			ATL::CComQIPtr<IVideoWindow> v(m_FilterGraph);
			if (v != 0) {
				INT32 x = 0;

				RECT rc = { 0 };
				GetWindowRect(&rc);

				x = rc.right + 8;

				v->put_Top(0);
				v->put_Left(x);
			}
		}

		{
			ATL::CComQIPtr<IBasicVideo> v(m_FilterGraph);
			if (v != 0) {
				LONG cx = 0;
				LONG cy = 0;

				v->get_VideoWidth (&cx);
				v->get_VideoHeight(&cy);

				ATLTRACE("IBasicVideo: VideoSize %d x %d\n", cx, cy);
			}
		}

		/* */

#ifdef _DEBUG
		AddToRot(m_FilterGraph, &m_dwCookie);
#endif

		/* */

		GetDlgItem(IDC_RUN)  .ShowWindow(SW_NORMAL);
		GetDlgItem(IDC_PAUSE).ShowWindow(SW_NORMAL);
		GetDlgItem(IDC_RUN)  .EnableWindow(TRUE);
		GetDlgItem(IDC_PAUSE).EnableWindow(TRUE);
	}

	/* */

	void Run()
	{
		if (m_FilterGraph == 0) {
			return;
		}

		ATL::CComQIPtr<IMediaControl> ctrl(m_FilterGraph);
		ctrl->Run();
	}

	void Pause()
	{
		if (m_FilterGraph == 0) {
			return;
		}

		ATL::CComQIPtr<IMediaControl> ctrl(m_FilterGraph);
		ctrl->Pause();
	}

	/* */

	void Seek(INT32 p)
	{
		if (m_FilterGraph != 0) {
			ATL::CComQIPtr<IMediaSeeking> seek(m_FilterGraph);
			if (seek != 0) {
				INT64 duration = 0;
				seek->GetDuration(&duration);

				INT64 pos = p * (duration / 10);
				seek->SetPositions(
					&pos,
					AM_SEEKING_AbsolutePositioning,
					0,
					AM_SEEKING_NoPositioning);
			}
		}
	}

	/* */

	LRESULT OnDropFiles(
		UINT   uMsg,
		WPARAM wParam,
		LPARAM lParam,
		BOOL&  bHandled)
	{
		HDROP hDrop = (HDROP)wParam;

		UINT c = DragQueryFileW(hDrop, (UINT)-1, 0, 0);
		if (c > 0) {
			WCHAR path[MAX_PATH];
			DragQueryFileW(hDrop, 0, path, MAX_PATH);

			ReleaseFilterGraph();
			Prepare(path);
		}

		DragFinish(hDrop);

		return 0;
	}

	/* */

	LRESULT OnEventNotify(
		UINT   uMsg,
		WPARAM wParam,
		LPARAM lParam,
		BOOL&  bHandled)
	{
		if (m_FilterGraph == 0) {
			return 0;
		}

		ATL::CComQIPtr<IMediaEvent> me(m_FilterGraph);
		if (me != 0) {
			LONG lEventCode = 0;

			LONG_PTR lParam1 = 0;
			LONG_PTR lParam2 = 0;

			HRESULT hRslt = me->GetEvent(
				&lEventCode,
				&lParam1,
				&lParam2,
				0);
			if (SUCCEEDED(hRslt)) {
				if (lEventCode == EC_COMPLETE) {
					ATLTRACE("EC_COMPLETE\n");

					ATL::CComQIPtr<IMediaControl> ctrl(m_FilterGraph);
					ctrl->Stop();
				}

				me->FreeEventParams(
					lEventCode,
					lParam1,
					lParam2);
			}
		}

		return 0;
	}

	/* */

	static HRESULT AddToRot(IUnknown* pUnkGraph, DWORD* pdwRegister)
	{
		ATL::CComPtr<IRunningObjectTable> rot;
		HRESULT hRslt = GetRunningObjectTable(0, &rot);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		WCHAR name[256];
		swprintf_s(name, 256, L"FilterGraph %08x pid %08x", (DWORD_PTR)pUnkGraph, GetCurrentProcessId());

		ATL::CComPtr<IMoniker> moniker;
		hRslt = CreateItemMoniker(L"!", name, &moniker);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		hRslt = rot->Register(
			ROTFLAGS_REGISTRATIONKEEPSALIVE,
			pUnkGraph,
			moniker,
			pdwRegister);
		if (FAILED(hRslt)) {
			return hRslt;
		}

		return S_OK;
	}

	static void RemoveFromRot(DWORD dwRegister)
	{
		ATL::CComPtr<IRunningObjectTable> rot;
		HRESULT hRslt = GetRunningObjectTable(0, &rot);
		if (FAILED(hRslt)) {
			return;
		}

		rot->Revoke(dwRegister);
	}

	/* */

}; // MainDialog

/* */

