/* StreamView.c */
/* 2009/05/20   */

#include "StdAfx.h"

#include "StreamView.h"

BOOL StreamView_Create(
	StreamView_t*      t,
	QM_Allocator_t*    allocator,
	QM_StreamReader_t* reader,
	INT32              size)
{
	if (t == NULL) {
		return FALSE;
	}

	t->Allocator = allocator;

	t->Reader = reader;

	t->Pointer = 0;

	t->Buffer = NULL;;
	t->Size   = size;

	t->Current = 0;
	t->Filled  = 0;

	t->Buffer = (UINT8*)allocator->Allocate(allocator->Context, size);
	if (t->Buffer == NULL) {
		return FALSE;
	}

	return TRUE;
}

void StreamView_Release(
	StreamView_t* t)
{
	if (t != NULL) {
		t->Allocator->Free(t->Allocator->Context, t->Buffer);
		t->Buffer = NULL;
	}
}

INT32 StreamView_Read(
	StreamView_t* t,
	INT32         size)
{
	INT32 r = 0, r0;

	INT32 start = t->Current + t->Filled;
	INT32 rest  = t->Size - t->Filled;

	INT32 to_r = size;
	if (to_r > rest) {
		to_r = rest;
	}

	if (start < t->Size) {
		INT32 r1 = t->Size - start;
		if (r1 > to_r) {
			r1 = to_r;
		}

		r0 = t->Reader->Read(
			t->Reader->Context,
			t->Buffer + start,
			r1);
		if (r0 < 0) {
			return r0;
		}

		t->Pointer += r0;
		t->Filled  += r0;

		r += r0;

		start += r0;
		to_r  -= r0;
	}

	if (start >= t->Size) {
		start -= t->Size;
	}

	if (to_r > 0) {
		INT32 r2 = t->Current - start;
		if (r2 > to_r) {
			r2 = to_r;
		}

		r0 = t->Reader->Read(
			t->Reader->Context,
			t->Buffer + start,
			r2);
		if (r0 < 0) {
			return r0;
		}

		t->Pointer += r0;
		t->Filled  += r0;

		r += r0;

		start += r0;
		to_r  -= r0;
	}

	return r;
}

/* */

void StreamView_MapChunk(
	StreamView_t*  t,
	StreamChunk_t* chunk)
{
	chunk->Buffer = t->Buffer;

	chunk->start = t->Current;
	chunk->mask  = t->Size - 1;

	chunk->p   = t->Current;
	chunk->end = t->Current + t->Filled;
}

void StreamView_RetireChunk(
	StreamView_t*  t,
	StreamChunk_t* chunk)
{
	INT32 sz = chunk->p - chunk->start;
	t->Current  = (t->Current + sz) & (t->Size - 1);
	t->Filled  -= sz;
}

INT64 StreamView_TellPosition(
	StreamView_t* t)
{
	return t->Pointer - t->Filled;
}

/* */

BOOL StreamView_Seek(
	StreamView_t* t,
	INT64         pos)
{
	INT64 start = t->Pointer - t->Filled;
	if (pos >= start && pos < t->Pointer) {
		INT32 offset = (INT32)(pos - start);

		t->Current  = (t->Current + offset) & (t->Size - 1);
		t->Filled  -= offset;

		return TRUE;
	}

	if (!t->Reader->Seek(
		t->Reader->Context,
		pos)) {
		return FALSE;
	}

	t->Pointer = pos;
	t->Current = (INT32)(pos & (t->Size - 1));

	t->Filled = t->Reader->Read(
		t->Reader->Context,
		t->Buffer + t->Current,
		t->Size   - t->Current);
	if (t->Filled <= 0) {
		return FALSE;
	}

	t->Pointer += t->Filled;

	return TRUE;
}

/* */

static BOOL TryLoadElement(
	StreamView_t*  t,
	StreamChunk_t* chunk,
	Element_t*     e)
{
	StreamView_MapChunk(t, chunk);

	if (!StreamChunk_GetElement(chunk, e)) {
		return FALSE;
	}

	return TRUE;
}

BOOL StreamView_LoadElement(
	StreamView_t*  t,
	INT32          id,
	StreamChunk_t* chunk,
	Element_t*     e)
{
	INT32 i;
	for (i = 0; ; i++) {
		if (i > 0) {
			if (t->Filled >= t->Size) {
				return FALSE;

			} else {
				INT32 r = 0;

				INT32 start = t->Current + t->Filled;
				if (start < t->Size) {
					r = t->Size - start;
				} else {
					start -= t->Size;
					r = t->Current - start;
				}

				r = t->Reader->Read(
					t->Reader->Context,
					t->Buffer + start,
					r);
				if (r <= 0) {
					return FALSE;
				}

				t->Pointer += r;
				t->Filled  += r;
			}
		}

		if (TryLoadElement(t, chunk, e)) {
			if (e->Id != id) {
				return FALSE;
			}

			if (!StreamChunk_CheckSize(chunk, e->Size)) {
				continue;
			}

			return TRUE;
		}
	}

	return FALSE;
}

BOOL StreamView_CheckElement(
	StreamView_t*  t,
	INT32          id,
	StreamChunk_t* chunk,
	Element_t*     e)
{
	INT32 i;
	for (i = 0; ; i++) {
		if (i > 0) {
			if (t->Filled >= t->Size) {
				return FALSE;

			} else {
				INT32 r = 0;

				INT32 start = t->Current + t->Filled;
				if (start < t->Size) {
					r = t->Size - start;
				} else {
					start -= t->Size;
					r = t->Current - start;
				}

				r = t->Reader->Read(
					t->Reader->Context,
					t->Buffer + start,
					r);
				if (r <= 0) {
					return FALSE;
				}

				t->Pointer += r;
				t->Filled  += r;
			}
		}

		if (TryLoadElement(t, chunk, e)) {
			if (e->Id != id) {
				return FALSE;
			}

			return TRUE;
		}
	}

	return FALSE;
}

BOOL StreamView_ReadElement(
	StreamView_t*  t,
	StreamChunk_t* chunk,
	Element_t*     e)
{
	INT32 i;
	for (i = 0; ; i++) {
		if (i > 0) {
			if (t->Filled >= t->Size) {
				return FALSE;

			} else {
				INT32 r = 0;

				INT32 start = t->Current + t->Filled;
				if (start < t->Size) {
					r = t->Size - start;
				} else {
					start -= t->Size;
					r = t->Current - start;
				}

				r = t->Reader->Read(
					t->Reader->Context,
					t->Buffer + start,
					r);
				if (r <= 0) {
					return FALSE;
				}

				t->Pointer += r;
				t->Filled  += r;
			}
		}

		if (TryLoadElement(t, chunk, e)) {
			if (!StreamChunk_CheckSize(chunk, e->Size)) {
				continue;
			}

			return TRUE;
		}
	}

	return FALSE;
}

/* */

#define GET_NEXT_BYTE(X) \
	if (t->p >= t->end) return -1; \
	(X) = *(t->Buffer + (((t->p)++) & t->mask));

INT32 StreamChunk_GetElementId(
	StreamChunk_t* t)
{
	INT32 mask = 0x80;
	INT32 id, head;

	GET_NEXT_BYTE(head)

	id = head;

	while ((head & mask) == 0) {
		INT32 by;

		GET_NEXT_BYTE(by)

		id = (id << 8) | by;

		mask >>= 1;
		if (mask == 0x08) {
			return -1;
		}
	}

	return id;
}

static const INT8 ES_LEN[0x100] = {
	0,8,7,7,6,6,6,6,5,5,5,5,5,5,5,5,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
	3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1
};

INT64 StreamChunk_GetElementSize(
	StreamChunk_t* t)
{
	INT64 size;

	INT32 len, head;

	GET_NEXT_BYTE(head)

	len = ES_LEN[head];
	if (len == 0) {
		return -1;
	}

	size = head & (0xff >> len);

	while (--len > 0) {
		INT32 by;

		GET_NEXT_BYTE(by)

		size = (size << 8) | by;
	}

	return size;
}

BOOL StreamChunk_GetElement(
	StreamChunk_t* t,
	Element_t*     e)
{
	e->Id = StreamChunk_GetElementId(t);
	if (e->Id < 0) {
		return FALSE;
	}

	e->Size = StreamChunk_GetElementSize(t);
	if (e->Size < 0) {
		return FALSE;
	}

	return TRUE;
}

BOOL StreamChunk_GetBinaryData(
	StreamChunk_t* t,
	VOID*          pv,
	INT64          size)
{
	UINT8* p   = (UINT8*)pv;
	UINT8* end = p + size;

	if (t->p + size > t->end) {
		return FALSE;
	}

	while (p < end) {
		*(p++) = *(t->Buffer + (((t->p)++) & t->mask));
	}

	return TRUE;
}

BOOL StreamChunk_GetInt32Data(
	StreamChunk_t* t,
	INT32*         value,
	INT64          size)
{
	INT32 v = 0;

	INT32 len = (INT32)size;

	if (size > 4 || t->p + size > t->end) {
		return FALSE;
	}

	while (len-- > 0) {
		v = (v << 8) | *(t->Buffer + (((t->p)++) & t->mask));
	}

	*value = v;

	return TRUE;
}

BOOL StreamChunk_GetSInt32Data(
	StreamChunk_t* t,
	INT32*         value,
	INT64          size)
{
	INT32 v = 0;

	INT32 len = (INT32)size;

	if (size > 4 || t->p + size > t->end) {
		return FALSE;
	}

	if (len-- > 0) {
		v = (INT8)(*(t->Buffer + (((t->p)++) & t->mask)));
	}

	while (len-- > 0) {
		v = (v << 8) | *(t->Buffer + (((t->p)++) & t->mask));
	}

	*value = v;

	return TRUE;
}

BOOL StreamChunk_GetInt64Data(
	StreamChunk_t* t,
	INT64*         value,
	INT64          size)
{
	INT64 v = 0;

	INT32 len = (INT32)size;

	if (size > 8 || t->p + size > t->end) {
		return FALSE;
	}

	while (len-- > 0) {
		v = (v << 8) | *(t->Buffer + (((t->p)++) & t->mask));
	}

	*value = v;

	return TRUE;
}

BOOL StreamChunk_GetFloatData(
	StreamChunk_t* t,
	DOUBLE*        value,
	INT64          size)
{
	if (size == 4) {
		INT32 i;
		UINT8 by[4];

		FLOAT v = 0;

		if (!StreamChunk_GetBinaryData(t, by, 4)) {
			return FALSE;
		}

		for (i = 0; i < 4; i++) {
			((UINT8*)(&v))[i] = by[3 - i];
		}

		*value = v;

	} else if (size == 8) {
		INT32 i;
		UINT8 by[8];

		DOUBLE v = 0;

		if (!StreamChunk_GetBinaryData(t, by, 8)) {
			return FALSE;
		}

		for (i = 0; i < 8; i++) {
			((UINT8*)(&v))[i] = by[7 - i];
		}

		*value = v;

	} else {
		return FALSE;
	}

	return TRUE;
}

BOOL StreamChunk_CopyBinaryData(
	StreamChunk_t* t,
	VOID*          pv,
	INT64          size)
{
	UINT8* p   = (UINT8*)pv;
	UINT8* end = p + size;

	if (t->p + size > t->end) {
		return FALSE;
	}

	if (p < end) {
		INT32 b = (INT32)(end - p);
		INT32 e = t->mask + 1;
		if (t->p < e) {
			INT32 s = e - t->p;
			if (s > b) {
				s = b;
			}

			memcpy(
				p,
				t->Buffer + (t->p & t->mask),
				s);

			p    += s;
			t->p += s;
		}
	}

	if (p < end) {
		INT32 b = (INT32)(end - p);
		INT32 s = t->end - t->p;
		if (s > b) {
			s = b;
		}

		memcpy(
			p,
			t->Buffer + (t->p & t->mask),
			s);

		p    += s;
		t->p += s;
	}

	return TRUE;
}

/* */

BOOL StreamChunk_CheckSize(
	StreamChunk_t* t,
	INT64          size)
{
	INT64 rest = t->end - t->p;
	if (rest < size) {
		return FALSE;
	}

	return TRUE;
}

BOOL StreamChunk_Skip(
	StreamChunk_t* t,
	INT64          size)
{
	INT64 next = t->p + size;
	if (next > t->end) {
		return FALSE;
	}

	t->p += (INT32)size;

	return TRUE;
}

INT32 StreamChunk_GetEndPoint(
	StreamChunk_t* t,
	INT64          size)
{
	return (INT32)(t->p + size);
}

INT32 StreamChunk_CheckEndPoint(
	StreamChunk_t* t,
	INT32          ep)
{
	if (t->p < ep) {
		return 0;
	}

	if (t->p == ep) {
		return 1;
	}

	return -1;
}

