/* MatroskaReader.c */
/* 2009/05/21       */

#include "StdAfx.h"

#include "MatroskaReader.h"

/* */

#define VIEW_BUFFER_SIZE 0x100000

#define BLOCK_BUFFER_SIZE 0x100000

#define PRELOAD_SIZE 0x10000

/* */

struct Tracks {

	QM_Track_t Track;

	Tracks_t* Next;

}; /* Tracks */

/* */

static BOOL CheckHeader(
	MatroskaReader_t* t)
{
	INT32 valid = 0;

	INT32 id, ep;
	INT64 sz;

	StreamChunk_t chunk;
	StreamView_MapChunk(&(t->View), &chunk);

	id = StreamChunk_GetElementId(&chunk);
	if (id != EBML_ID_HEADER) {
		return FALSE;
	}

	sz = StreamChunk_GetElementSize(&chunk);
	if (sz < 0) {
		return FALSE;
	}

	if (!StreamChunk_CheckSize(&chunk, sz)) {
		return FALSE;
	}

	ep = StreamChunk_GetEndPoint(&chunk, sz);

	for (; ; ) {
		Element_t e;

		INT32 code = StreamChunk_CheckEndPoint(&chunk, ep);
		if (code < 0) {
			return FALSE;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(&chunk, &e)) {
			return FALSE;
		}

		if (e.Id == EBML_ID_DOCTYPE) {
			UINT8 dt[8];

			if (e.Size != 8) {
				return FALSE;
			}

			if (!StreamChunk_GetBinaryData(&chunk, dt, 8)) {
				return FALSE;
			}

			if (memcmp(dt, "matroska", 8) != 0) {
				return FALSE;
			}

			valid = 1;

		} else {
			if (!StreamChunk_Skip(&chunk, e.Size)) {
				return FALSE;
			}
		}
	}

	StreamView_RetireChunk(&(t->View), &chunk);

	if (valid != 1) {
		return FALSE;
	}

	return TRUE;
}

static BOOL CheckSegment(
	MatroskaReader_t* t)
{
	INT32 id;
	INT64 sz;

	StreamChunk_t chunk;
	StreamView_MapChunk(&(t->View), &chunk);

	id = StreamChunk_GetElementId(&chunk);
	if (id != MATROSKA_ID_SEGMENT) {
		return FALSE;
	}

	sz = StreamChunk_GetElementSize(&chunk);
	if (sz < 0) {
		return FALSE;
	}

	StreamView_RetireChunk(&(t->View), &chunk);

	return TRUE;
}

static BOOL CheckMetaSeekInfomation(
	MatroskaReader_t* t)
{
	INT32 id, ep;
	INT64 sz;

	StreamChunk_t chunk;

	t->MetaOrigin = StreamView_TellPosition(&(t->View));

	StreamView_MapChunk(&(t->View), &chunk);

	id = StreamChunk_GetElementId(&chunk);
	if (id != MATROSKA_ID_SEEKHEAD) {
		return FALSE;
	}

	sz = StreamChunk_GetElementSize(&chunk);
	if (sz < 0) {
		return FALSE;
	}

	if (!StreamChunk_CheckSize(&chunk, sz)) {
		return FALSE;
	}

	ep = StreamChunk_GetEndPoint(&chunk, sz);

	for (; ; ) {
		Element_t e;

		INT32 code = StreamChunk_CheckEndPoint(&chunk, ep);
		if (code < 0) {
			return FALSE;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(&chunk, &e)) {
			return FALSE;
		}

		if (e.Id == MATROSKA_ID_SEEKENTRY) {
			INT32 mid = -1;
			INT64 pos = -1;

			INT32 ep1 = StreamChunk_GetEndPoint(&chunk, e.Size);

			for (; ; ) {
				Element_t e1;

				code = StreamChunk_CheckEndPoint(&chunk, ep1);
				if (code < 0) {
					return FALSE;
				}
				if (code != 0) {
					break;
				}

				if (!StreamChunk_GetElement(&chunk, &e1)) {
					return FALSE;
				}

				switch (e1.Id) {
				case MATROSKA_ID_SEEKID:
					if (!StreamChunk_GetInt32Data(&chunk, &mid, e1.Size)) {
						return FALSE;
					}
					break;

				case MATROSKA_ID_SEEKPOSITION:
					if (!StreamChunk_GetInt64Data(&chunk, &pos, e1.Size)) {
						return FALSE;
					}
					break;

				default:
					if (!StreamChunk_Skip(&chunk, e1.Size)) {
						return FALSE;
					}
					break;
				}
			}

			switch (mid) {
			case MATROSKA_ID_INFO:
				t->MetaPosition[META_INFO] = pos;
				break;

			case MATROSKA_ID_TRACKS:
				t->MetaPosition[META_TRACKS] = pos;
				break;

			case MATROSKA_ID_CUES:
				t->MetaPosition[META_CUES] = pos;
				break;

			case MATROSKA_ID_SEEKHEAD:
				t->MetaPosition[META_SEEK] = pos;
				break;
			}

		} else {
			if (!StreamChunk_Skip(&chunk, e.Size)) {
				return FALSE;
			}
		}
	}

	StreamView_RetireChunk(&(t->View), &chunk);

	return TRUE;
}

/* */

static BOOL CheckSegmentInformation(
	MatroskaReader_t* t)
{
	StreamChunk_t chunk;
	Element_t e;
	INT32 ep;

	INT64 pos = t->MetaOrigin + t->MetaPosition[META_INFO];

	if (!StreamView_Seek(&(t->View), pos)) {
		return FALSE;
	}

	if (!StreamView_LoadElement(
		&(t->View),
		MATROSKA_ID_INFO,
		&chunk,
		&e)) {
		return FALSE;
	}

	ep = StreamChunk_GetEndPoint(&chunk, e.Size);

	for (; ; ) {
		INT32 code = StreamChunk_CheckEndPoint(&chunk, ep);
		if (code < 0) {
			return FALSE;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(&chunk, &e)) {
			return FALSE;
		}

		switch (e.Id) {
		case MATROSKA_ID_TIMECODESCALE:
			if (!StreamChunk_GetInt64Data(&chunk, &(t->TimeCodeScale), e.Size)) {
				return FALSE;
			}
			break;

		case MATROSKA_ID_DURATION:
			if (!StreamChunk_GetFloatData(&chunk, &(t->Duration), e.Size)) {
				return FALSE;
			}
			break;

		default:
			if (!StreamChunk_Skip(&chunk, e.Size)) {
				return FALSE;
			}
			break;
		}
	}

	StreamView_RetireChunk(&(t->View), &chunk);

	return TRUE;
}

/* */

static QM_TrackVideo_t* CheckTrackVideo(
	MatroskaReader_t* t,
	StreamChunk_t*    chunk,
	INT64             size)
{
	INT32 ep = StreamChunk_GetEndPoint(chunk, size);

	QM_TrackVideo_t* v = (QM_TrackVideo_t*)QM_MemoryPool_Allocate(
		&(t->Pool),
		t->Allocator,
		sizeof(QM_TrackVideo_t));
	if (v == NULL) {
		return NULL;
	}

	memset(v, 0, sizeof(QM_TrackVideo_t));

	for (; ; ) {
		Element_t e;

		INT32 code = StreamChunk_CheckEndPoint(chunk, ep);
		if (code < 0) {
			return NULL;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(chunk, &e)) {
			return NULL;
		}

		switch (e.Id) {
		case MATROSKA_ID_VIDEOPIXELWIDTH:
			if (!StreamChunk_GetInt32Data(chunk, &(v->PixelWidth), e.Size)) {
				return NULL;
			}
			break;

		case MATROSKA_ID_VIDEOPIXELHEIGHT:
			if (!StreamChunk_GetInt32Data(chunk, &(v->PixelHeight), e.Size)) {
				return NULL;
			}
			break;

		case MATROSKA_ID_VIDEOFRAMERATE:
			if (!StreamChunk_GetFloatData(chunk, &(v->FrameRate), e.Size)) {
				return NULL;
			}
			break;

		default:
			if (!StreamChunk_Skip(chunk, e.Size)) {
				return NULL;
			}
			break;
		}
	}

	return v;
}

static QM_TrackAudio_t* CheckTrackAudio(
	MatroskaReader_t* t,
	StreamChunk_t*    chunk,
	INT64             size)
{
	INT32 ep = StreamChunk_GetEndPoint(chunk, size);

	QM_TrackAudio_t* v = (QM_TrackAudio_t*)QM_MemoryPool_Allocate(
		&(t->Pool),
		t->Allocator,
		sizeof(QM_TrackAudio_t));
	if (v == NULL) {
		return NULL;
	}

	memset(v, 0, sizeof(QM_TrackAudio_t));

	for (; ; ) {
		Element_t e;

		INT32 code = StreamChunk_CheckEndPoint(chunk, ep);
		if (code < 0) {
			return NULL;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(chunk, &e)) {
			return NULL;
		}

		switch (e.Id) {
		case MATROSKA_ID_AUDIOCHANNELS:
			if (!StreamChunk_GetInt32Data(chunk, &(v->Channels), e.Size)) {
				return NULL;
			}
			break;

		case MATROSKA_ID_AUDIOSAMPLINGFREQ:
			if (!StreamChunk_GetFloatData(chunk, &(v->SamplingFrequency), e.Size)) {
				return NULL;
			}
			break;

		default:
			if (!StreamChunk_Skip(chunk, e.Size)) {
				return NULL;
			}
			break;
		}
	}

	return v;
}

/* */

static BOOL CheckTrackInformation(
	MatroskaReader_t* t)
{
	StreamChunk_t chunk;
	Element_t e;
	INT32 ep;

	INT64 pos = t->MetaOrigin + t->MetaPosition[META_TRACKS];

	if (!StreamView_Seek(&(t->View), pos)) {
		return FALSE;
	}

	if (!StreamView_LoadElement(
		&(t->View),
		MATROSKA_ID_TRACKS,
		&chunk,
		&e)) {
		return FALSE;
	}

	ep = StreamChunk_GetEndPoint(&chunk, e.Size);

	for (; ; ) {
		INT32 code = StreamChunk_CheckEndPoint(&chunk, ep);
		if (code < 0) {
			return FALSE;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(&chunk, &e)) {
			return FALSE;
		}

		switch (e.Id) {
		case MATROSKA_ID_TRACKENTRY:
		{
			INT32 ep1 = StreamChunk_GetEndPoint(&chunk, e.Size);

			Tracks_t* track = (Tracks_t*)QM_MemoryPool_Allocate(
				&(t->Pool),
				t->Allocator,
				sizeof(Tracks_t));
			if (track == NULL) {
				return FALSE;
			}

			memset(track, 0, sizeof(Tracks_t));

			track->Next = t->TrackList;

			t->TrackList   = track;
			t->TrackCount += 1;

			for (; ; ) {
				Element_t e1;

				code = StreamChunk_CheckEndPoint(&chunk, ep1);
				if (code < 0) {
					return FALSE;
				}
				if (code != 0) {
					break;
				}

				if (!StreamChunk_GetElement(&chunk, &e1)) {
					return FALSE;
				}

				switch (e1.Id) {
				case MATROSKA_ID_TRACKNUMBER:
					if (!StreamChunk_GetInt32Data(&chunk, &(track->Track.TrackNo), e1.Size)) {
						return FALSE;
					}
					break;

				case MATROSKA_ID_TRACKFLAGLACING:
					if (!StreamChunk_GetInt32Data(&chunk, &(track->Track.TrackLacing), e1.Size)) {
						return FALSE;
					}
					break;

				case MATROSKA_ID_TRACKTYPE:
					if (!StreamChunk_GetInt32Data(&chunk, &(track->Track.TrackType), e1.Size)) {
						return FALSE;
					}
					break;

				case MATROSKA_ID_CODECID:
				{
					char* id;

					if (e1.Size > 0x100) {
						return FALSE;
					}

					id = (char*)QM_MemoryPool_Allocate(
						&(t->Pool),
						t->Allocator,
						(SIZE_T)e1.Size + 1);
					if (id == NULL) {
						return FALSE;
					}

					if (!StreamChunk_GetBinaryData(&chunk, id, e1.Size)) {
						return FALSE;
					}

					id[e1.Size] = '\0';

					track->Track.CodecId = id;

					break;
				}

				case MATROSKA_ID_TRACKDEFAULTDURATION:
					if (!StreamChunk_GetInt64Data(&chunk, &(track->Track.TrackDuration), e1.Size)) {
						return FALSE;
					}
					break;

				case MATROSKA_ID_CODECPRIVATE:
				{
					VOID* pv;

					if (e1.Size > 0x10000) {
						return FALSE;
					}

					pv = QM_MemoryPool_Allocate(
						&(t->Pool),
						t->Allocator,
						(SIZE_T)e1.Size);
					if (pv == NULL) {
						return FALSE;
					}

					if (!StreamChunk_GetBinaryData(&chunk, pv, e1.Size)) {
						return FALSE;
					}

					track->Track.CodecPrivate     = pv;
					track->Track.CodecPrivateSize = (INT32)e1.Size;

					break;
				}

				case MATROSKA_ID_TRACKVIDEO:
					track->Track.Video = CheckTrackVideo(t, &chunk, e1.Size);
					if (track->Track.Video == NULL) {
						return FALSE;
					}
					break;

				case MATROSKA_ID_TRACKAUDIO:
					track->Track.Audio = CheckTrackAudio(t, &chunk, e1.Size);
					if (track->Track.Audio == NULL) {
						return FALSE;
					}
					break;

				default:
					if (!StreamChunk_Skip(&chunk, e1.Size)) {
						return FALSE;
					}
					break;
				}
			}

			break;
		}

		default:
			if (!StreamChunk_Skip(&chunk, e.Size)) {
				return FALSE;
			}
			break;
		}
	}

	StreamView_RetireChunk(&(t->View), &chunk);

	if (t->TrackCount > 0) {
		INT32 i;
		Tracks_t* n = t->TrackList;

		t->Tracks = (QM_Track_t*)QM_MemoryPool_Allocate(
			&(t->Pool),
			t->Allocator,
			t->TrackCount * sizeof(QM_Track_t));
		if (t->Tracks == NULL) {
			return FALSE;
		}

		for (i = t->TrackCount - 1; i >= 0; i--, n = n->Next) {
			t->Tracks[i] = n->Track;
		}
	}

	return TRUE;
}

/* */

static BOOL CheckCueingData(
	MatroskaReader_t* t)
{
	StreamChunk_t chunk;
	Element_t e;
	INT32 ep;

	INT64 pos = t->MetaOrigin + t->MetaPosition[META_CUES];

	t->QuePointSize  = 0x10;
	t->QuePointCount = 0;

	t->QuePoints = (QM_QuePoint_t*)t->Allocator->Allocate(
		t->Allocator->Context,
		t->QuePointSize * sizeof(QM_QuePoint_t));
	if (t->QuePoints == NULL) {
		return FALSE;
	}

	if (!StreamView_Seek(&(t->View), pos)) {
		return FALSE;
	}

	if (!StreamView_LoadElement(
		&(t->View),
		MATROSKA_ID_CUES,
		&chunk,
		&e)) {
		return FALSE;
	}

	ep = StreamChunk_GetEndPoint(&chunk, e.Size);

	for (; ; ) {
		INT32 code = StreamChunk_CheckEndPoint(&chunk, ep);
		if (code < 0) {
			return FALSE;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(&chunk, &e)) {
			return FALSE;
		}

		switch (e.Id) {
		case MATROSKA_ID_POINTENTRY:
		{
			INT64 cue_time  = -1;
			INT32 cue_track = -1;
			INT64 cue_pos   = -1;

			INT32 ep1 = StreamChunk_GetEndPoint(&chunk, e.Size);

			for (; ; ) {
				Element_t e1;

				code = StreamChunk_CheckEndPoint(&chunk, ep1);
				if (code < 0) {
					return FALSE;
				}
				if (code != 0) {
					break;
				}

				if (!StreamChunk_GetElement(&chunk, &e1)) {
					return FALSE;
				}

				switch (e1.Id) {
				case MATROSKA_ID_CUETIME:
					if (!StreamChunk_GetInt64Data(&chunk, &cue_time, e1.Size)) {
						return FALSE;
					}
					break;

				case MATROSKA_ID_CUETRACKPOSITION:
				{
					INT32 ep2 = StreamChunk_GetEndPoint(&chunk, e1.Size);

					for (; ; ) {
						Element_t e2;

						code = StreamChunk_CheckEndPoint(&chunk, ep2);
						if (code < 0) {
							return FALSE;
						}
						if (code != 0) {
							break;
						}

						if (!StreamChunk_GetElement(&chunk, &e2)) {
							return FALSE;
						}

						switch (e2.Id) {
						case MATROSKA_ID_CUETRACK:
							if (!StreamChunk_GetInt32Data(&chunk, &cue_track, e2.Size)) {
								return FALSE;
							}
							break;

						case MATROSKA_ID_CUECLUSTERPOSITION:
							if (!StreamChunk_GetInt64Data(&chunk, &cue_pos, e2.Size)) {
								return FALSE;
							}
							break;

						default:
							if (!StreamChunk_Skip(&chunk, e2.Size)) {
								return FALSE;
							}
							break;
						}
					}

					if (cue_time > 0) {
						// printf("CUE: %8d : %d : %08X\n", (INT32)cue_time, (INT32)cue_track, (INT32)cue_pos);

						if (t->QuePointCount >= t->QuePointSize) {
							QM_QuePoint_t* q;

							t->QuePointSize *= 2;
							q = (QM_QuePoint_t*)t->Allocator->Reallocate(
								t->Allocator->Context,
								t->QuePoints,
								t->QuePointSize * sizeof(QM_QuePoint_t));
							if (q == NULL) {
								return FALSE;
							}

							t->QuePoints = q;
						}

						t->QuePoints[t->QuePointCount].CueTime    = cue_time;
						t->QuePoints[t->QuePointCount].TrackNo    = cue_track;
						t->QuePoints[t->QuePointCount].ClusterPos = cue_pos;

						t->QuePointCount += 1;
					}

					break;
				}

				default:
					if (!StreamChunk_Skip(&chunk, e1.Size)) {
						return FALSE;
					}
					break;
				}
			}

			break;
		}

		default:
			if (!StreamChunk_Skip(&chunk, e.Size)) {
				return FALSE;
			}
			break;
		}
	}

	StreamView_RetireChunk(&(t->View), &chunk);

	return TRUE;
}

static BOOL CheckSeekHead(
	MatroskaReader_t* t)
{
	StreamChunk_t chunk;
	Element_t e;
	INT32 ep;

	INT64 pos = t->MetaOrigin + t->MetaPosition[META_SEEK];

	t->ClusterSize  = 0x10;
	t->ClusterCount = 0;

	t->Clusters = (INT64*)t->Allocator->Allocate(
		t->Allocator->Context,
		t->ClusterSize * sizeof(INT64));
	if (t->Clusters == NULL) {
		return FALSE;
	}

	if (!StreamView_Seek(&(t->View), pos)) {
		return FALSE;
	}

	if (!StreamView_LoadElement(
		&(t->View),
		MATROSKA_ID_SEEKHEAD,
		&chunk,
		&e)) {
		return FALSE;
	}

	ep = StreamChunk_GetEndPoint(&chunk, e.Size);

	for (; ; ) {
		INT32 code = StreamChunk_CheckEndPoint(&chunk, ep);
		if (code < 0) {
			return FALSE;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(&chunk, &e)) {
			return FALSE;
		}

		switch (e.Id) {
		case MATROSKA_ID_SEEKENTRY:
		{
			INT32 seek_id  = -1;
			INT64 seek_pos = -1;

			INT32 ep1 = StreamChunk_GetEndPoint(&chunk, e.Size);

			for (; ; ) {
				Element_t e1;

				code = StreamChunk_CheckEndPoint(&chunk, ep1);
				if (code < 0) {
					return FALSE;
				}
				if (code != 0) {
					break;
				}

				if (!StreamChunk_GetElement(&chunk, &e1)) {
					return FALSE;
				}

				switch (e1.Id) {
				case MATROSKA_ID_SEEKID:
					if (!StreamChunk_GetInt32Data(&chunk, &seek_id, e1.Size)) {
						return FALSE;
					}
					break;

				case MATROSKA_ID_SEEKPOSITION:
					if (!StreamChunk_GetInt64Data(&chunk, &seek_pos, e1.Size)) {
						return FALSE;
					}
					break;

				default:
					if (!StreamChunk_Skip(&chunk, e1.Size)) {
						return FALSE;
					}
					break;
				}
			}

			// printf("SEEK: %08X : %08X\n", (INT32)seek_id, (INT32)seek_pos);

			if (seek_id == MATROSKA_ID_CLUSTER) {
				if (t->ClusterCount >= t->ClusterSize) {
					INT64* q;

					t->ClusterSize *= 2;
					q = (INT64*)t->Allocator->Reallocate(
						t->Allocator->Context,
						t->Clusters,
						t->ClusterSize * sizeof(INT64));
					if (q == NULL) {
						return FALSE;
					}

					t->Clusters = q;
				}

				t->Clusters[t->ClusterCount] = seek_pos;

				t->ClusterCount += 1;
			}

			break;
		}

		default:
			if (!StreamChunk_Skip(&chunk, e.Size)) {
				return FALSE;
			}
			break;
		}
	}

	StreamView_RetireChunk(&(t->View), &chunk);

	return TRUE;
}

/* */

BOOL MatroskaReader_Init(
	MatroskaReader_t* t,
	QM_Allocator_t*   allocator)
{
	if (t == NULL) {
		return FALSE;
	}

	memset(t, 0, sizeof(MatroskaReader_t));

	t->Allocator = allocator;

	QM_MemoryPool_Init(&(t->Pool));

	t->TrackList  = NULL;
	t->TrackCount = 0;

	t->Tracks = NULL;

	t->QuePoints     = NULL;
	t->QuePointSize  = 0;
	t->QuePointCount = 0;

	t->Clusters     = NULL;
	t->ClusterSize  = 0;
	t->ClusterCount = 0;

	t->Block = NULL;

	return TRUE;
}

BOOL MatroskaReader_Open(
	MatroskaReader_t*  t,
	QM_StreamReader_t* reader)
{
	INT32 r;

	if (t == NULL) {
		return FALSE;
	}

	if (!StreamView_Create(
		&(t->View),
		t->Allocator,
		reader,
		VIEW_BUFFER_SIZE)) {
		return FALSE;
	}

	r = StreamView_Read(
		&(t->View),
		PRELOAD_SIZE);
	if (r <= 0) {
		return FALSE;
	}

	if (!CheckHeader(t)) {
		return FALSE;
	}

	if (!CheckSegment(t)) {
		return FALSE;
	}

	if (!CheckMetaSeekInfomation(t)) {
		return FALSE;
	}

	if (!CheckSegmentInformation(t)) {
		return FALSE;
	}

	if (!CheckTrackInformation(t)) {
		return FALSE;
	}

	if (!CheckCueingData(t)) {
		return FALSE;
	}

	if (!CheckSeekHead(t)) {
		return FALSE;
	}

	if (t->ClusterCount == 0) {
		return FALSE;
	}

	{
		INT64 pos = t->MetaOrigin + t->Clusters[0];
		if (!StreamView_Seek(&(t->View), pos)) {
			return FALSE;
		}

		t->State = 0;
	}

	t->Block = (UINT8*)t->Allocator->Allocate(
		t->Allocator->Context,
		BLOCK_BUFFER_SIZE);
	if (t->Block == NULL) {
		return FALSE;
	}

	return TRUE;
}

void MatroskaReader_Release(
	MatroskaReader_t* t)
{
	if (t != NULL) {
		StreamView_Release(&(t->View));

		QM_MemoryPool_Release(
			&(t->Pool),
			t->Allocator);

		t->Allocator->Free(t->Allocator->Context, t->Block);

		t->Allocator->Free(t->Allocator->Context, t->QuePoints);

		t->Allocator->Free(t->Allocator->Context, t->Clusters);
	}
}

/* */

static BOOL CheckCluster(
	MatroskaReader_t* t)
{
	INT64 cluster, end;

	StreamChunk_t chunk;
	Element_t e;

	cluster = StreamView_TellPosition(&(t->View));

	if (!StreamView_CheckElement(
		&(t->View),
		MATROSKA_ID_CLUSTER,
		&chunk,
		&e)) {
		return FALSE;
	}

	StreamView_RetireChunk(&(t->View), &chunk);

	end = StreamView_TellPosition(&(t->View)) + e.Size;

	t->ClusterPos = cluster;
	t->ClusterEnd = end;

	return TRUE;
}

/* */

static BOOL ReadBlockPayload(
	MatroskaReader_t*  t,
	StreamChunk_t*     chunk,
	INT64              size,
	QM_Block_t*        block)
{
	UINT8 h[4];

	if (size < 4) {
		return FALSE;
	}

	if (!StreamChunk_GetBinaryData(chunk, h, 4)) {
		return FALSE;
	}

	if ((h[0] & 0x80) == 0) {
		return FALSE;
	}

	block->ClusterTime = t->ClusterTime;
	block->TrackNo   = h[0] & 0x7f;
	block->TimeDelta = (((INT8)h[1]) << 8) | h[2];
	block->Flags     = h[3];

	block->Payload = t->Block;
	block->Size    = (INT32)(size - 4);

	if (!StreamChunk_CopyBinaryData(chunk, t->Block, block->Size)) {
		return FALSE;
	}

	return TRUE;
}

static BOOL ReadBlockGroup(
	MatroskaReader_t*  t,
	StreamChunk_t*     chunk,
	INT64              size,
	QM_Block_t*        block)
{
	INT32 ep = StreamChunk_GetEndPoint(chunk, size);

	for (; ; ) {
		Element_t e;

		INT32 code = StreamChunk_CheckEndPoint(chunk, ep);
		if (code < 0) {
			return FALSE;
		}
		if (code != 0) {
			break;
		}

		if (!StreamChunk_GetElement(chunk, &e)) {
			return FALSE;
		}

		switch (e.Id) {
		case MATROSKA_ID_BLOCK:
		{
			if (!ReadBlockPayload(t, chunk, e.Size, block)) {
				return FALSE;
			}
			break;
		}

		case MATROSKA_ID_BLOCKDURATION:
			if (!StreamChunk_GetInt32Data(chunk, &(block->Duration), e.Size)) {
				return FALSE;
			}
			break;

		case MATROSKA_ID_BLOCKREFERENCE:
			if (!StreamChunk_GetSInt32Data(chunk, &(block->Reference), e.Size)) {
				return FALSE;
			}
			break;

		default:
			if (!StreamChunk_Skip(chunk, e.Size)) {
				return FALSE;
			}
			break;
		}
	}

	return TRUE;
}

INT32 MatroskaReader_Read(
	MatroskaReader_t*  t,
	QM_Block_t*        block)
{
	if (t == NULL || block == NULL) {
		return -1;
	}

	memset(block, 0, sizeof(QM_Block_t));

	for (; ; ) {
		if (t->State == 0) {
			if (!CheckCluster(t)) {
				return -1;
			}

			t->State = 1;

		} else {
			INT32 have = 0;

			StreamChunk_t chunk;
			Element_t e;

			INT64 pos = StreamView_TellPosition(&(t->View));
			if (pos >= t->ClusterEnd) {
				if (t->ClusterPos >= t->MetaOrigin + t->Clusters[t->ClusterCount - 1]) { /* EOS */
					return 0;
				}

				t->State = 0;
				continue;
			}

			if (!StreamView_ReadElement(&(t->View), &chunk, &e)) {
				return -1;
			}

			switch (e.Id) {
			case MATROSKA_ID_CLUSTERTIMECODE:
				if (!StreamChunk_GetInt64Data(&chunk, &(t->ClusterTime), e.Size)) {
					return -1;
				}
				break;

			case MATROSKA_ID_SIMPLEBLOCK:
			{
				if (!ReadBlockPayload(t, &chunk, e.Size, block)) {
					return -1;
				}

				have = 1;

				break;
			}

			case MATROSKA_ID_BLOCKGROUP:
			{
				if (!ReadBlockGroup(t, &chunk, e.Size, block)) {
					return FALSE;
				}

				have = 1;

				break;
			}

			default:
				if (!StreamChunk_Skip(&chunk, e.Size)) {
					return -1;
				}
				break;
			}

			StreamView_RetireChunk(&(t->View), &chunk);

			if (have) {
				break;
			}
		}
	}

	return 1;
}

/* */

BOOL MatroskaReader_SeekCluster(
	MatroskaReader_t* t,
	INT64             cpos)
{
	INT64 pos = t->MetaOrigin + cpos;
	if (!StreamView_Seek(&(t->View), pos)) {
		return FALSE;
	}

	t->State = 0;

	return TRUE;
}

/* */

INT64 MatroskaReader_LookupCluster(
	MatroskaReader_t* t,
	INT64             ts)
{
	const QM_QuePoint_t* s = t->QuePoints;
	const QM_QuePoint_t* e = s + t->QuePointCount;

	const QM_QuePoint_t* p = s + (e - s) / 2;

	while (e - s > 1) {
		if (ts == p->CueTime) {
			break;
		} else if (ts < p->CueTime) {
			e = p;
		} else {
			s = p;
		}

		p = s + (e - s) / 2;
	}

	if (ts < p->CueTime) {
		return t->Clusters[0];
	}

	return p->ClusterPos;
}

/* */

