/* MatroskaLib.c */
/* 2009/05/26    */

#include "StdAfx.h"

#include "MatroskaReader.h"

#include "BlockLacing.h"

struct QM_Reader {

	MatroskaReader_t Reader;

	QM_StreamReader_t Stream;

	FILE* fp;

	IStream* p;

}; /* QM_Reader */

QM_Reader_t* QM_CreateReader(void)
{
	QM_Allocator_t* allocator = Allocator_Get();

	QM_Reader_t* t = (QM_Reader_t*)malloc(sizeof(QM_Reader_t));
	if (t == NULL) {
		return NULL;
	}

	memset(t, 0, sizeof(QM_Reader_t));

	MatroskaReader_Init(&(t->Reader), allocator);

	t->fp = NULL;
	t->p  = NULL;

	return t;
}

void QM_ReleaseReader(QM_Reader_t* t)
{
	if (t != NULL) {
		MatroskaReader_Release(&(t->Reader));

		if (t->fp != NULL) {
			fclose(t->fp);
		}

		if (t->p != NULL) {
			t->p->lpVtbl->Release(t->p);
		}

		free(t);
	}
}

BOOL QM_OpenReader(
	QM_Reader_t* t,
	const WCHAR* path)
{
	FILE* fp = NULL;

	_wfopen_s(&fp, path, L"rb");
	if (fp == NULL) {
		return FALSE;
	}

	QM_StreamReader_Init(&(t->Stream), fp);

	if (!MatroskaReader_Open(&(t->Reader), &(t->Stream))) {
		fclose(fp);
		return FALSE;
	}

	t->fp = fp;

	return TRUE;
}

BOOL QM_OpenReader_IStream(
	QM_Reader_t* t,
	IStream*     p)
{
	QM_StreamReader_IStream(&(t->Stream), p);

	if (!MatroskaReader_Open(&(t->Reader), &(t->Stream))) {
		return FALSE;
	}

	t->p = p;
	t->p->lpVtbl->AddRef(t->p);

	return TRUE;
}

INT64 QM_GetTimeCodeScale(
	QM_Reader_t* t)
{
	return t->Reader.TimeCodeScale;
}

DOUBLE QM_GetDuraion(
	QM_Reader_t* t)
{
	return t->Reader.Duration;
}

INT32 QM_GetTrackCount(
	QM_Reader_t* t)
{
	return t->Reader.TrackCount;
}

const QM_Track_t* QM_GetTracks(
	QM_Reader_t* t)
{
	return t->Reader.Tracks;
}

INT32 QM_ReadBlock(
	QM_Reader_t* t,
	QM_Block_t*  block)
{
	INT32 code = MatroskaReader_Read(&(t->Reader), block);

	return code;
}

BOOL QM_SeekCluster(
	QM_Reader_t* t,
	INT64        pos)
{
	INT64 cpos = MatroskaReader_LookupCluster(&(t->Reader), pos);

	return MatroskaReader_SeekCluster(&(t->Reader), cpos);
}

/* */

struct QM_FrameExtractor {

	BlockLacing_t Decoder;

}; /* QM_FrameExtractor */

QM_FrameExtractor_t* QM_CreateFrameExtractor(void)
{
	QM_Allocator_t* allocator = Allocator_Get();

	QM_FrameExtractor_t* t = (QM_FrameExtractor_t*)malloc(sizeof(QM_FrameExtractor_t));
	if (t == NULL) {
		return NULL;
	}

	if (!BlockLacing_Create(&(t->Decoder), allocator, 0x100)) {
		BlockLacing_Release(&(t->Decoder));
		free(t);
		return NULL;
	}

	return t;
}

void QM_ReleaseFrameExtractor(QM_FrameExtractor_t* t)
{
	if (t != NULL) {
		BlockLacing_Release(&(t->Decoder));
		free(t);
	}
}

BOOL QM_DecodeLacingBlock(
	QM_FrameExtractor_t* t,
	const QM_Block_t*    block)
{
	if (!BlockLacing_Decode(
		&(t->Decoder),
		block->Payload,
		block->Size,
		block->Flags)) {
		return FALSE;
	}

	return TRUE;
}

INT32 QM_ReadFrame(
	QM_FrameExtractor_t* t,
	QM_Frame_t*          frame)
{
	INT32 code = BlockLacing_ReadNext(
		&(t->Decoder),
		frame);

	return code;
}

/* */

