// QReader.h
// 2009/06/04

#pragma once

#define QV_VORBIS_MAX_FRAME_SIZE 0x10000

// QReader
class QReader {

	QM_Reader_t* m_reader;

	QM_FrameExtractor* m_extra;

	INT64 m_duration;

	INT32 m_vid;
	INT32 m_aid;

	const QM_Track_t* m_vtrack;
	const QM_Track_t* m_atrack;

	QTheoraDecoderSetup_t* m_vsetup;

	QTheoraDecoder_t* m_vdecoder;

	QVorbisDecoderSetup_t* m_asetup;

	QVorbisDecoder_t* m_adecoder;

	QT_Format_t m_vf;

	INT64 m_scale;
	INT64 m_vrate;

	INT32 m_ascale;

	bool  m_bVideoSeek;
	bool  m_bAudioSeek;

	bool  m_bVideoEOS;
	bool  m_bAudioEOS;

	INT64 m_StartPos;
	INT64 m_EndPos;

public:

	QReader() :
		m_reader(0), m_extra(0),
		m_duration(0),
		m_vid(0), m_aid(0),
		m_vtrack(0), m_atrack(0),
		m_vsetup(0), m_vdecoder(0),
		m_asetup(0), m_adecoder(0),
		m_scale(10*1000), m_vrate(0), m_ascale(1000),
		m_bVideoSeek(false), m_bAudioSeek(false),
		m_bVideoEOS(false),  m_bAudioEOS(false),
		m_StartPos(0), m_EndPos(-1)
	{
		memset(&m_vf, 0, sizeof(m_vf));
	}

	~QReader()
	{
		QM_ReleaseReader(m_reader);
		QM_ReleaseFrameExtractor(m_extra);

		QT_ReleaseDecoder(m_vdecoder);
		QT_ReleaseDecoderSetup(m_vsetup);

		QV_ReleaseDecoder(m_adecoder);
		QV_ReleaseDecoderSetup(m_asetup);
	}

	/* */

	INT64 GetDuration()
	{
		return m_duration;
	}

	const QM_Track_t* GetVideoTrack()
	{
		return m_vtrack;
	}

	const QM_Track_t* GetAudioTrack()
	{
		return m_atrack;
	}

	const QT_Format_t* GetVideoFormat()
	{
		return &m_vf;
	}

	/* */

	bool Open(LPCWSTR path)
	{
		m_reader = QM_CreateReader();
		if (m_reader == 0) {
			return false;
		}

		if (!QM_OpenReader(m_reader, path)) {
			return false;
		}

		return true;
	}

	bool Open(IStream* p)
	{
		m_reader = QM_CreateReader();
		if (m_reader == 0) {
			return false;
		}

		if (!QM_OpenReader_IStream(m_reader, p)) {
			return false;
		}

		return true;
	}

	bool Setup()
	{
		INT64 scale = QM_GetTimeCodeScale(m_reader);
#if 0
		if (scale != 1000*1000) {
			return false;
		}
#else
		// matroska ts -> RT : 1ns -> 100ns
		m_scale  = (INT32)(scale / 100);
		// 1s / matroska ts
		m_ascale = (INT32)(1000*1000*1000 / scale);
#endif

		DOUBLE d = QM_GetDuraion(m_reader);
		m_duration = (INT64)(d * m_scale);

		const QM_Track_t* tracks = QM_GetTracks(m_reader);

		INT32 c = QM_GetTrackCount(m_reader);
		for (INT32 i = 0; i < c; i++) {
			const QM_Track_t* track = tracks + i;

			if (strcmp(track->CodecId, "V_THEORA") == 0) {
				if (m_vid == 0) {
					m_vid = track->TrackNo;

					m_vtrack = track;
				}

			} else if (strcmp(track->CodecId, "A_VORBIS") == 0) {
				if (m_aid == 0) {
					m_aid = track->TrackNo;

					m_atrack = track;
				}
			}
		}

		if (m_vid > 0) {
			m_vsetup = QT_CreateDecoderSetup();
			if (m_vsetup == 0) {
				return false;
			}

			if (!QT_SetupDecoderSetupLacing(
				m_vsetup,
				m_vtrack->CodecPrivate,
				m_vtrack->CodecPrivateSize)) {
				return false;
			}

			if (!QT_GetFormatSetup(m_vsetup, &m_vf)) {
				return false;
			}

			m_vdecoder = QT_CreateDecoder();
			if (m_vdecoder == 0) {
				return false;
			}

			if (!QT_SetupDecoder(m_vdecoder, m_vsetup)) {
				return false;
			}

			m_vrate = m_vtrack->TrackDuration / 100; /* ns -> RT (100ns) */
		}

		if (m_aid > 0) {
			m_extra = QM_CreateFrameExtractor();
			if (m_extra == 0) {
				return false;
			}

			m_asetup = QV_CreateDecoderSetup();
			if (m_asetup == 0) {
				return false;
			}

			if (!QV_SetupDecoderSetupLacing(
				m_asetup,
				m_atrack->CodecPrivate,
				m_atrack->CodecPrivateSize)) {
				return false;
			}

			m_adecoder = QV_CreateDecoder();
			if (m_adecoder == 0) {
				return false;
			}

			if (!QV_SetupDecoder(m_adecoder, m_asetup, QV_VORBIS_MAX_FRAME_SIZE)) {
				return false;
			}
		}

		m_bVideoSeek = true;
		m_bAudioSeek = true;

		m_bVideoEOS = false;
		m_bAudioEOS = false;

		m_StartPos = 0;
		m_EndPos   = -1;

		return true;
	}

	/* */

	INT32 Read(
		IMemAllocator*   valloc,
		IMemInputPin*    vpin,
		QImageConverter* image,
		IMemAllocator*   aalloc,
		IMemInputPin*    apin,
		QWaveOutputer*   aout)
	{
		if (m_bVideoEOS && m_bAudioEOS) {
			return 0;
		}

		QM_Block_t block = { 0 };
		INT32 code = QM_ReadBlock(
			m_reader,
			&block);
		if (code < 0) {
			return -1;
		}
		if (code == 0) {
			return 0;
		}

		if (block.TrackNo == m_vid && !m_bVideoEOS) {
			INT64 ts = block.ClusterTime + block.TimeDelta;

			if (m_bVideoSeek) {
				if ((block.Flags & QM_FLAGS_KEYFRAME) == 0) {
					return 1;
				}

				m_bVideoSeek = false;
			}

			QT_Output_t output = { 0 };
			if (!QT_DecodeFrame(
				m_vdecoder,
				block.Payload,
				block.Size,
				&output)) {
				return -1;
			}

			if (ts < m_StartPos) {
				return 1;
			}

			if (m_EndPos > 0 && ts >= m_EndPos) {
				m_bVideoEOS = true;
				return 1;
			}

			if (valloc == 0 || vpin == 0) {
				return 1;
			}

			REFERENCE_TIME rts = (ts - m_StartPos) * m_scale;
			REFERENCE_TIME rte = rts + m_vrate;

			ATL::CComPtr<IMediaSample> samp;
			HRESULT hRslt = valloc->GetBuffer(&samp, &rts, &rte, 0);
			if (FAILED(hRslt)) {
				return -1;
			}

			BYTE* pb = 0;
			samp->GetPointer(&pb);
			if (pb != 0) {
				image->Copy(pb, &output);
			}

			samp->SetTime(&rts, &rte);

			hRslt = vpin->Receive(samp);
			if (FAILED(hRslt)) {
				return -1;
			}

		} else if (block.TrackNo == m_aid && !m_bAudioEOS) {
			INT64 ts = block.ClusterTime + block.TimeDelta;
			INT32 ss = 0;

			if (QM_DecodeLacingBlock(m_extra, &block)) {
				for (; ; ) {
					QM_Frame_t frame = { 0 };
					INT32 code = QM_ReadFrame(
						m_extra,
						&frame);
					if (code < 0) {
						return -1;
					}
					if (code == 0) {
						break;
					}

					/* */

					if (m_bAudioSeek) {
						QV_ResetDecoder(m_adecoder);

						m_bAudioSeek = false;
					}

					QV_Output_t output = { 0 };
					if (!QV_DecodeFrame(
						m_adecoder,
						frame.Payload,
						frame.Size,
						&output)) {
						return -1;
					}

					INT64 t0 = ts + aout->GetTime(ss * m_ascale);

					ss += aout->GetSamples(output.Length);

					if (t0 < m_StartPos) {
						continue;
					}

					if (m_EndPos > 0 && t0 >= m_EndPos) {
						m_bAudioEOS = true;
						return 1;
					}

					REFERENCE_TIME rts = (t0 - m_StartPos) * m_scale;

					if (output.Length > 0) {
						QV_Convert_t conv = { 0 };
						if (!QV_ConvertFrame(
							m_adecoder,
							&output,
							&conv)) {
							return -1;
						}

						if (!aout->Output(
							aalloc,
							apin,
							rts,
							conv.Sample,
							conv.Samples)) {
							return -1;
						}
					}
				}
			}
		}

		return 1;
	}

	/* */

	bool Seek(INT64 pos)
	{
		INT64 ts = pos / m_scale;

		if (!QM_SeekCluster(m_reader, ts)) {
			return false;
		}

		m_bVideoSeek = true,
		m_bAudioSeek = true,

		m_bVideoEOS = false;
		m_bAudioEOS = false;

		m_StartPos = ts;
		m_EndPos   = -1;

		return true;
	}

	/* */

}; // QReader

