package org.qrone.one.inner;

import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Set;
import java.util.Vector;

import javax.servlet.ServletContext;

import org.qrone.XMLTools;
import org.qrone.one.QrONEServer;
import org.qrone.one.QrONEService;
import org.qrone.one.QrONEServiceConfig;
import org.qrone.one.QrONEServiceContext;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class QrONEAppDocument {
	private QrONEServer server;
	private ServletContext context;
	private QrONEServiceContext servicecontext;
	private Hashtable contextparam = new Hashtable();
	private Hashtable contextattribute = new Hashtable();
	
	private String displayName;
	private String serviceDomain;
	private Service[] serviceList;
	private ServiceMapping[] serviceMapList;

	public QrONEAppDocument(QrONEServer server, Document doc, ServletContext context){
		this.server = server;
		this.context = context;
		Vector service = new Vector();
		Vector servicemap = new Vector();

		NodeList list = doc.getDocumentElement().getChildNodes();
		for(int i=0;i<list.getLength();i++){
			Node node = list.item(i);
			if(node.getNodeType() == Node.ELEMENT_NODE){
				Element enode = (Element)node;
				
				if(enode.getNodeName().equals("display-name")){
					displayName = XMLTools.getTextContent(enode);
				}else if(enode.getNodeName().equals("service-domain")){
					serviceDomain = XMLTools.getTextContent(enode);
				}else if(enode.getNodeName().equals("service")){
					service.add(new Service(enode));
				}else if(enode.getNodeName().equals("service-mapping")){
					servicemap.add(new ServiceMapping(enode));
				}else if(enode.getNodeName().equals("context-param")){
					NodeList nlist = enode.getElementsByTagName("param-name");
					NodeList vlist = enode.getElementsByTagName("param-value");
					
					if(nlist.getLength()>0 && vlist.getLength()>0){
						contextparam.put(XMLTools.getTextContent(nlist.item(0)),
								XMLTools.getTextContent(vlist.item(0)));
					}
				}
			}
		}
		
		serviceList = (Service[])service.toArray(new Service[0]);
		serviceMapList = (ServiceMapping[])servicemap.toArray(new ServiceMapping[0]);
	}
	
	public String getDisplayName() {
		return displayName;
	}

	public String getServiceDomain() {
		return serviceDomain;
	}

	public Service[] getServices() {
		return serviceList;
	}

	public ServiceMapping[] getServiceMappings() {
		return serviceMapList;
	}
	
	public ServiceMapping getServiceMappingByUri(String uri){
		for (int i = 0; i < serviceMapList.length; i++) {
			ServiceMapping map = serviceMapList[i];
			if(map.match(uri)){
				return map;
			}
		}
		return null;
	}
	
	public Service getServiceByName(String serviceName){
		for (int i = 0; i < serviceList.length; i++) {
			Service ser = serviceList[i];
			if(ser.getServiceName().equals(serviceName)){
				return ser;
			}
		}
		return null;
	}
	
	public QrONEServiceConfig createService(final String uri) throws ServiceException{
		ServiceMapping map = getServiceMappingByUri(uri);
		if(map == null) return null;
		
		final Service ser = getServiceByName(map.getServiceName());
		if(ser == null) return null;
		
		final QrONEService service = ser.newInstance();
		return new QrONEServiceConfig(){
			public String getInitParameter(String name) {
				return ser.getInitParameter(name);
			}
			public Enumeration getInitParameterNames() {
				return ser.getInitParameterNames();
			}
			public QrONEService getService() {
				return service;
			}
			public String getServiceURI() {
				return uri;
			}
			public String getServiceDomainURI() {
				return serviceDomain;
			}
			public QrONEServiceContext getServiceContext() {
				return QrONEAppDocument.this.getServiceContext();
			}
			public String getServiceName() {
				return ser.getDisplayName();
			}
			public void invalidate() {
				server.removeService(service);
			}
		};
	}
	
	public QrONEServiceContext getServiceContext(){
		if(servicecontext==null) servicecontext = new QrONEServiceContext(){
			public QrONEServiceContext getContext(String uripath) {
				return server.getServiceContext(uripath);
			}
			public String getInitParameter(String name) {
				return (String)contextparam.get(name);
			}
			public Enumeration getInitParameterNames() {
				return contextparam.keys();
			}
			public int getMajorVersion() {
				return QrONEServer.MAJOR_VERSION;
			}
			public int getMinorVersion() {
				return QrONEServer.MINOR_VERSION;
			}
			public String getServerInfo() {
				return QrONEServer.INFO;
			}
			public String getRealPath(String path) {
				return context.getRealPath(path);
			}
			public URL getResource(String path) throws MalformedURLException {
				return context.getResource(path);
			}
			public InputStream getResourceAsStream(String path) {
				return context.getResourceAsStream(path);
			}
			public Set getResourcePaths(String path) {
				return context.getResourcePaths(path);
			}
			public String getServiceContextName() {
				return displayName;
			}
			public void log(String msg) {
				context.log(msg);
			}
			public void log(String message, Throwable throwable) {
				context.log(message, throwable);
			}
			public Object getAttribute(String name) {
				return contextattribute.get(name);
			}
			public Enumeration getAttributeNames() {
				return contextattribute.keys();
			}
			public void removeAttribute(String name) {
				contextattribute.remove(name);
			}
			public void setAttribute(String name, Object object) {
				contextattribute.put(name,object);
			}
		};
		return servicecontext;
	}
	
	public class Service{
		private String description;
		private String displayName;
		private String serviceName;
		private String serviceClass;
		
		private Hashtable initparam = new Hashtable();
		
		public Service(Element element){
			NodeList list = element.getChildNodes();
			for(int i=0;i<list.getLength();i++){
				Node node = list.item(i);
				if(node.getNodeType() == Node.ELEMENT_NODE){
					Element enode = (Element)node;
					if(enode.getNodeName().equals("display-name")){
						displayName = XMLTools.getTextContent(enode);
					}else if(enode.getNodeName().equals("description")){
						description = XMLTools.getTextContent(enode);
					}else if(enode.getNodeName().equals("service-name")){
						serviceName = XMLTools.getTextContent(enode);
					}else if(enode.getNodeName().equals("service-class")){
						serviceClass = XMLTools.getTextContent(enode);
					}else if(enode.getNodeName().equals("init-param")){
						NodeList nlist = enode.getElementsByTagName("param-name");
						NodeList vlist = enode.getElementsByTagName("param-value");
						
						if(nlist.getLength()>0 && vlist.getLength()>0){
							initparam.put(XMLTools.getTextContent(nlist.item(0)),
									XMLTools.getTextContent(vlist.item(0)));
						}
					}
				}
			}
		}
		
		public String getDescription() {
			return description;
		}
		public String getDisplayName() {
			return displayName;
		}
		public String getServiceClass() {
			return serviceClass;
		}
		public String getServiceName() {
			return serviceName;
		}
		public String getInitParameter(String name) {
			return (String)initparam.get(name);
		}
		public Enumeration getInitParameterNames() {
			return initparam.keys();
		}
		
		public QrONEService newInstance() throws ServiceException{
			try {
				Class cls = Class.forName(serviceClass);
				return (QrONEService)cls.newInstance();
			} catch (ClassNotFoundException e){
				throw new ServiceException("ClassNotFound : " +serviceClass,e);
			} catch (InstantiationException e) {
				throw new ServiceException("Cant create instance of this class",e);
			} catch (IllegalAccessException e) {
				throw new ServiceException("Constractor not public",e);
			} catch (ClassCastException e){
				throw new ServiceException("Services must extend QrONEService",e);
			}
		}
	}
	
	public class ServiceMapping{
		private String serviceName;
		private String urlPattern;
		
		public ServiceMapping(Element element){
			NodeList list = element.getChildNodes();
			for(int i=0;i<list.getLength();i++){
				Node node = list.item(i);
				if(node.getNodeType() == Node.ELEMENT_NODE){
					Element enode = (Element)node;
					
					if(enode.getNodeName().equals("service-name")){
						serviceName = XMLTools.getTextContent(enode);
					}else if(enode.getNodeName().equals("url-pattern")){
						urlPattern = XMLTools.getTextContent(enode);
					}
				}
			}
		}
		
		public String getServiceName() {
			return serviceName;
		}
		public String getUrlPattern() {
			return urlPattern;
		}
		
		public boolean match(String uri){
			if(urlPattern.endsWith("*")){
				if(uri.startsWith(urlPattern.substring(0,urlPattern.length()-1))){
					return true;
				}
			}else{
				if(uri.equals(serviceDomain + urlPattern)){
					return true;
				}
			}
			
			return false;
		}
	}
}
