/*!
  \file
  \brief VAʐM̎ (Windows)

  \author Satofumi KAMIMURA

  $Id: SerialDevice_win.cpp 1209 2009-08-05 13:11:59Z satofumi $
*/

#include "DetectOS.h"
#include <windows.h>
#include <string>

using namespace std;

#ifdef MSC
#define snprintf _snprintf
#endif

#undef min
#undef max


class RawSerialDevice
{
  string error_message_;
  HANDLE hCom_;
  HANDLE hEvent_;
  OVERLAPPED overlapped_;


public:
  RawSerialDevice::RawSerialDevice(void)
    : error_message_("no error."),
      hCom_(INVALID_HANDLE_VALUE), hEvent_(INVALID_HANDLE_VALUE)
  {
  }


  const char* RawSerialDevice::what(void)
  {
    return error_message_.c_str();
  }


  bool RawSerialDevice::connect(const char* device, long baudrate)
  {
    /* COM |[gJ */
    enum { NameLength = 11 };
    char adjusted_device[NameLength];
    snprintf(adjusted_device, NameLength, "\\\\.\\%s", device);
    hCom_ = CreateFileA(adjusted_device, GENERIC_READ | GENERIC_WRITE, 0,
                        NULL, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    if (hCom_ == INVALID_HANDLE_VALUE) {
      error_message_ = string("open failed: ") + device;
      return false;
    }

    hEvent_ = CreateEvent(NULL, TRUE, FALSE, NULL);
    if(hEvent_ == INVALID_HANDLE_VALUE) {
      // !!! G[bZ[WXVׂ
      return false;
    }
    overlapped_.hEvent = hEvent_;
    overlapped_.Offset = 0;
    overlapped_.OffsetHigh = 0;

    // ʐMTCY̍XV
    SetupComm(hCom_, 4096 * 8, 4096);

    // {[[g̕ύX
    bool ret = setBaudrate(baudrate);
    if (! ret) {
      error_message_ = "fail SerialDevice::setBaudrate()";
    }
    return true;
  }


  void RawSerialDevice::disconnect(void)
  {
    if (hCom_ != INVALID_HANDLE_VALUE) {
      CloseHandle(hCom_);
      hCom_ = INVALID_HANDLE_VALUE;
    }
    if (hEvent_) {
      CloseHandle(hEvent_);
      hEvent_ = INVALID_HANDLE_VALUE;
    }
  }


  bool RawSerialDevice::isConnected(void)
  {
    return (hCom_ == INVALID_HANDLE_VALUE) ? false : true;
  }


  bool RawSerialDevice::setBaudrate(long baudrate)
  {
    long baudrate_value;
    switch (baudrate) {

    case 4800:
      baudrate_value = CBR_4800;
      break;

    case 9600:
      baudrate_value = CBR_9600;
      break;

    case 19200:
      baudrate_value = CBR_19200;
      break;

    case 38400:
      baudrate_value = CBR_38400;
      break;

    case 57600:
      baudrate_value = CBR_57600;
      break;

    case 115200:
      baudrate_value = CBR_115200;
      break;

    default:
      baudrate_value = baudrate;
    }

    DCB dcb;
    GetCommState(hCom_, &dcb);
    dcb.BaudRate = baudrate_value;
    dcb.ByteSize = 8;
    dcb.Parity = NOPARITY;
    dcb.fParity = FALSE;
    dcb.StopBits = ONESTOPBIT;
    if (SetCommState(hCom_, &dcb) == 0) {
      flush();
      return false;
    } else {
      return true;
    }
  }


  int RawSerialDevice::send(const char* data, size_t count)
  {
    if (count <= 0) {
      return 0;
    }

    DWORD n;
    if (WriteFile(hCom_, data, (DWORD)count, &n, &overlapped_)) {
      return n;
    } else {
      if (GetLastError() != ERROR_IO_PENDING) {
        return -1;
      }
    }

    WaitForSingleObject(hEvent_, INFINITE);
    GetOverlappedResult(hCom_, &overlapped_, &n, TRUE);

    return n;
  }


  int receive(char data[], int count, int timeout)
  {
    if (count <= 0) {
      return 0;
    }

    DWORD n;
    int filled = 0;
    int read_n;

    do {
      read_n = count - filled;
      if (ReadFile(hCom_, &data[filled], count - filled, &n, &overlapped_)) {
        return filled + n;
      } else {
        if (GetLastError() != ERROR_IO_PENDING) {
          return -1;
        }
      }

      int ret = WaitForSingleObject(hEvent_, timeout);
      if (ret == WAIT_TIMEOUT) {
        break;
      }
      GetOverlappedResult(hCom_, &overlapped_, &n, FALSE);
      filled += n;
    } while ((filled < count) && (n != 0));

    return filled;
  }


  void RawSerialDevice::flush(void)
  {
    PurgeComm(hCom_,
              PURGE_RXABORT | PURGE_TXABORT | PURGE_RXCLEAR | PURGE_TXCLEAR);
  }
};
