--! \file
-- \brief コルーチンのスケジューラ
--
-- getTicks() で経過時間を msec で返せる必要がある
-- !!! 引数を渡したときに、適切に動作しないことがある
--
-- \author Satofumi KAMIMURA
--
-- $Id: scheduler.lua 243 2009-02-27 12:44:50Z satofumi $


Scheduler = {}


function Scheduler:new()

   local members = {
      functions = {},           --!< コルーチンの格納用

      alive_index = {},         --!< ID とコルーチンとの関係の保持用
      alive_index_first = 1,    --!< 最初に有効な ID

      dead_index = {},           --!< 使ってない index の管理テーブル
      dead_index_last = 0        --!< 使ってない index の最後の値
   }

   Scheduler.__index = Scheduler
   setmetatable(members, Scheduler)

   return members
end


-- 関数をコルーチンに登録
function Scheduler:registerFunction(f, a)

   local handling_alive_index = -1

   if self.dead_index_last == 0 then
      -- コルーチンの新規作成
      local next_alive_index = #self.alive_index + 1
      self.functions[next_alive_index] = {
         func = f,
         args = a,
         co = nil,
         wakeup_ticks = false
      }
      local self_function = self.functions[next_alive_index]
      self_function.co =
         coroutine.create(function()
                             while true do
                                self_function.func(self_function.args)
                                coroutine.yield(false)
                             end
                          end
                       )
      table.insert(self.alive_index, next_alive_index)
      handling_alive_index = next_alive_index

   else
      -- コルーチンの再利用。関数を再設定して初期化
      local function_id = self.dead_index[self.dead_index_last]
      self.functions[function_id].func = f
      self.functions[function_id].args = a
      self.dead_index_last = self.dead_index_last - 1

      table.insert(self.alive_index, function_id)
      --self.alive_index[function_id] = function_id

      handling_alive_index = function_id
   end

   return handling_alive_index
end


-- コルーチン実行を待機させる
function Scheduler:wait(msec)

   coroutine.yield(msec)
end



-- コルーチン実行を中断して処理を戻す
function Scheduler:yield()

   coroutine.yield(true)
end


-- スケジューラの実行
function Scheduler:execute()

   local alive_coroutines = 0

   local alive_index_size = #self.alive_index
   for i = self.alive_index_first, alive_index_size do

      local id = self.alive_index[i]

      if id == false then
         if self.alive_index_first == i then
            -- 最初の ID が無効ならば、次回からの開始位置を進める
            self.alive_index_first = self.alive_index_first + 1
         end
      else
         -- 終了したコルーチン以外を処理する
         local alive_function = self.functions[id]

         -- 次の実行時間になっていなければ、処理しない
         if (alive_function.wakeup_ticks == false)
         or (getTicks() > alive_function.wakeup_ticks) then

         alive_function.wakeup_ticks = false
         local ret, alive_flag = coroutine.resume(alive_function.co,
                                                  alive_function.args)
         if alive_flag == false then
            -- コルーチンの終了処理
            alive_function.func = false
            alive_function.args = false
            self.alive_index[i] = false
            self.dead_index[self.dead_index_last + 1] = i
            self.dead_index_last = self.dead_index_last + 1
         else
            if alive_flag ~= true then
               -- 次のコルーチン再開時刻の保持
               local msec = alive_flag
               alive_function.wakeup_ticks = getTicks() + msec
            end
            -- 動作中のコルーチンを数える
            alive_coroutines = alive_coroutines + 1
         end
      end
   end
end

return alive_coroutines
end


-- 指定した ID のコルーチンが有効かを返す
function Scheduler:isActive(id)

   local function_id = self.alive_index[id]
   if (function_id == false) or (function_id == nil) then
      return false
   else
      return true
   end
end


-- 指定されたサイズのコルーチンを作成する
function Scheduler:resize(size)

   local add_size = size - #self.functions

   for i = 1, (add_size + 1) do
      registerFunction(function()
                          -- 空の関数
                       end)
   end
end
