from twisted.trial.unittest import TestCase
from twisted.web.resource import Resource
from twisted.web import server
from twisted.internet import reactor, defer
import moduleloader
import os, os.path, shutil

class ModuleLoaderTest(TestCase):
    def testNormal(self):
        plugindirname = "my/test/module"
        pluginPath=["."]
        if not os.path.exists(plugindirname):
            os.makedirs(plugindirname)
        fp = open(plugindirname + "/" + "mytest.py", "w")
        fp.close()
        
        mod = moduleloader.loadModule("my.test.module.mytest", pluginPath)
        self.assertEquals(not mod, False)
        self.assertRaises(ImportError, moduleloader.loadModule, "my.test.module.mytest2", pluginPath)
        self.assertRaises(ImportError, moduleloader.loadModule, "filter.nomod", pluginPath)


class MyResource(Resource):
    isLeaf = False
    def getChild(self, name, request):
        if not name:
            return self
        return Resource.getChild(self, name, request)

class MyPyResource(Resource):
    def render_GET(self, request):
        return 'print("iyan")'

from twisted.python import failure

class ModuleNetLoaderTest(TestCase):
    def setUp(self):
        moduleloader._moduleFactories = {}
        root = MyResource()
        leaf1 = MyResource()
        leaf2 = MyResource()
        root.putChild("base", leaf1)
        leaf1.putChild("m", leaf2)
        leaf2.putChild("mymod.py", MyPyResource())
        root.putChild("mymod.py", MyResource)
        site = server.Site(root)
        self.port = self._listen(site)
        self.portno = self.port.getHost().port

    def tearDown(self):
        try:
            shutil.rmtree("./tmpDownload")
        except:
            pass
        return self.port.stopListening()

    def _listen(self, site):
        return reactor.listenTCP(0, site, interface="127.0.0.1")

    def getURL(self, path):
        return "http://127.0.0.1:%d/%s" % (self.portno, path)
    
    def _checkExist(self, result, *args, **kwds):
        self.assertEquals(os.path.exists("./tmpDownload/m/mymod.py"), True)
        return result

    def testDownload(self):
        d = moduleloader.downloadModule(self.getURL("base"),
                                        "m.mymod",
                                        "./tmpDownload")
        return d.addCallback(self._checkExist)

    def testDownloadFail(self):
        d = moduleloader.downloadModule(self.getURL("base"),
                                        "m.mymod2",
                                        "./tmpDownload")
        def got(result):
            self.assertEquals(1, 0)

        def gotErr(failure):
            pass
        
        return d.addCallback(got).addErrback(gotErr)

    def testNetLoader(self):
        d = moduleloader.loadModuleFromNet("m.mymod", ["./tmpDownload"],
                                         url = self.getURL("base"),
                                        )
        if isinstance(d, defer.Deferred):
            return d.addCallback(self._checkExist)
        
        self.assertEquals(not d, False)

    def testNetLoaderExist(self):
        os.makedirs("./tmpDownload/m")
        fp = open("./tmpDownload/m/mymod.py", "w")
        fp.write("import sys")
        fp.close()
        d = moduleloader.loadModuleFromNet("m.mymod", ["./tmpDownload"],
                                         url = self.getURL("base"),
                                        )
        if isinstance(d, defer.Deferred):
            self.assertEquals(0, 1)
        
        self.assertEquals(not d, False)


    def testNetLoaderBadPy(self):
        os.makedirs("./tmpDownload/m")
        fp = open("./tmpDownload/m/mymod.py", "w")
        fp.write("hogehoge fuga")
        fp.close()
        self.assertRaises(Exception, moduleloader.loadModuleFromNet,
                          "m.mymod", ["./tmpDownload"],
                                         url = self.getURL("base"),
                                        )

