from twisted.trial.unittest import TestCase
from twisted.internet import defer
import cron
import time

class CronTest(TestCase):
    def testParseExpression(self):
        data = [("hourly", ["0", "*", "*", "*", "*"]),
                ("daily", ["0", "2", "*", "*", "*"]),
                ("weekly", ["0", "0", "0", "0", "1"]),
                ("monthly", ["0", "0", "1", "*", "*"]),
                ("every 10 min", ["*/10", "*", "*", "*", "*"]),
                ("every 15 min", ["*/15", "*", "*", "*", "*"]),
                ("every 30 min", ["*/30", "*", "*", "*", "*"]),
                ("* * * * *", ["*", "*", "*", "*", "*"]),
                ("1 2 3 4 *", ["1", "2", "3", "4", "*"]),
                ("1,2,3 2 3 4 *", ["1,2,3", "2", "3", "4", "*"]),
                ]
        for item in data:
            self.assertEquals(cron.parseExpression(item[0]), item[1])

    def testFailParseExpression(self):
        data = ("dayly", "*/10 * *a * *", "1 2 3 4 *b", 
                "2 3 4", "1 2 3 4 5 6",
                "     ", "1 2 3 4  *",
                "1 2 3 4 * ", "    ")
        for item in data:
            self.assertRaises(cron.ParseExpressionError,
                              cron.parseExpression, item)

    
    def testMatchWild(self):
        obj = cron.Cron()
        self.assertEquals(obj.matchWild(1, "*"), True)
        self.assertEquals(obj.matchWild(0, "*"), True)
        self.assertEquals(obj.matchWild(10, "*/10"), True)
        self.assertEquals(obj.matchWild(20, "*/10"), True)
        self.assertEquals(obj.matchWild(1, "*/10"), False)
        self.assertRaises(ValueError, obj.matchWild, 1, "1/*")
        self.assertRaises(ValueError, obj.matchWild, 1, "*/a")

    def testMatch(self):
        obj = cron.Cron()
        self.assertEquals(obj.match(10, "1"), False)
        self.assertEquals(obj.match(10, "1,2,3"), False)
        self.assertEquals(obj.match(1, "1,2,3"), True)
        self.assertEquals(obj.match(1, "1"), True)
        self.assertEquals(obj.match(1, "*"), True)
        self.assertEquals(obj.match(10, "*/10"), True)
        self.assertEquals(obj.match(20, "*/10"), True)
        self.assertEquals(obj.match(1, "*/10"), False)
        self.assertRaises(ValueError, obj.match, 1, "1/*")
        self.assertRaises(ValueError, obj.match, 1, "*/a")

    def testMatchAll(self):
        # tm_year,tm_mon,tm_day,tm_hour,tm_min,tm_sec,tm_wday ,tm_yday,tm_isdst)
        data = (((2007, 9, 26, 15, 1, 0, 2, 269, 0), "* * * * *", True),
                ((2007, 9, 26, 15, 1, 0, 2, 269, 0), "1 * * * *", True),
                ((2007, 9, 26, 15, 1, 0, 2, 269, 0), "0 * * * *", False),
                ((2007, 9, 26, 15, 1, 0, 2, 269, 0), "*/10 * * * *", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "*/10 * * * *", True),
                ((2007, 9, 26, 15, 0, 0, 2, 269, 0), "*/10 * * * *", True),

                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* 1 * * *", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* 15 * * *", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* */5 * * *", True),

                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 25 * *", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* 1 26 * *", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* 15 */2 * *", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * */2 * *", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 * *", True),

                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * * 9 *", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * * 10 *", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 */3 *", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* 15 */2 */3 *", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * */2 */3 *", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 9 *", True),


                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * * 9 */3", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * * 10 */3", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 */3 */3", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* 15 */2 */3 */3", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * */2 */3 */3", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 9 */3", True),
                
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * * 9 3", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * * 10 3", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 */3 3", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* 15 */2 */3 3", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * */2 */3 3", True),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 9 3", True),

                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * * 9 2", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * * 10 2", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 */3 2", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* 15 */2 */3 2", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * */2 */3 2", False),
                ((2007, 9, 26, 15, 10, 0, 2, 269, 0), "* * 26 9 2", False),

                )
        obj = cron.Cron()
        for item in data:
            self.assertEquals(obj.matchAll(item[0], item[1].split(" ")), item[2])


    def testNextTime(self):
        now = time.mktime(time.localtime())
        obj = cron.Cron()
        nextTm = obj.nextTime()
        self.assertEquals(int(nextTm) % cron.SCHEDULE_INTERVAL, 0)
        self.assertEquals(nextTm > now, True)
        self.assertEquals(nextTm <= now + cron.SCHEDULE_INTERVAL, True)
        
    def cb(self, *args, **kwds):
        self.assertEquals(len(args), 3)
        self.assertEquals(len(kwds), 3)
        self.execCount += 1
        
        if self.execCount == 2:
            args[0].stop()
            self.assertEquals(time.time()-self.starttm < 60*3, True)
            self.deferred.callback("DONE")

    def xxxtestStart(self):
        self.execCount = 0
        self.starttm = time.time()
        self.deferred = defer.Deferred()
        obj = cron.Cron()
        obj.add("* * * * *".split(), self.cb, obj, 1,2, a=3, b=4, c=5)
        self.assertEquals(len(obj.crontab), 1)
        obj.start()
        return self.deferred
        
