from datetime import datetime, timedelta
from twisted.web import resource, static, server
from habu.habuutils import dumpRSS
import habu.log as log
import StringIO

def createStaticResource(config):
    path = config.get("path", None)
    if not path:
        return None
    return static.File(path)

_SEARCH_HTML = """<html>
  <head>
    <meta content="text/html; charset=utf-8" http-equiv="content-type" />
  </head>
  <body><h1>Search</h1>
    <form method="GET">Search Phrase : <input type="text" name="q" /></form>
  </body>
</html>
"""

def createHEResource(config):
    try:
        from habu.subscription.hesearcher import Searcher
        url = config.get("node")
        userid = config.get("userid", "admin")
        password = config.get("password", "admin")
        searcher = Searcher(url, userid, password)
        
        def gotResult(result, request):
            log.debug("HyperEstraier Search : got result")
            log.debug(result)
            dumpRSS(result, request, "atom1.0")
            request.finish()

        def gotErr(failure, request):
            request.write(str(failure))
            request.finish()

        class HyperEstraierResource(resource.Resource):
            def render_GET(self, request):
                queryStr = request.args.get("q", None)
                if not queryStr:
                    return _SEARCH_HTML
                queryStr = " ".join(queryStr)
                
                deferred = searcher.search(queryStr)
                deferred.addCallback(gotResult, request)
                deferred.addErrback(gotErr, request)
                
                return server.NOT_DONE_YET        

        return HyperEstraierResource()
    except Exception, e:
        log.error(e)

    return None

def createPlanetResource(config):
    try:
        import habu.habudb as dbstore
        # maxRange: days to show
        # maxSize: 0 meeans unlimited
        # if fetched size of maxRange is greater than maxSize,
        # truncated to maxSize.
        maxRange = config.get("range", 7)
        maxSize = config.get("size", 0)

        title = config.get("title", "Habu Planet")
        url = config.get("url", "http://localhost/")
        
        class PlanetResource(resource.Resource):
            def toDict(self, entry):
                if entry.author:
                    author = "%s (%s)" % (entry.author, entry.feed.name)
                else:
                    author = entry.feed.name
                data = {
                    "link": entry.url,
                    "title": entry.title,
                    "author": author,
                    "updated": str(entry.updated),
                    "summary": entry.summary,
                    "summary_detail": {"type": entry.format,
                                       "value": entry.summary}
                    }
                return data
                
            def render_GET(self, request):
                # TODO: cgi parameter must be filter condition

                feeds = dbstore.Feed.select(orderBy = "name")
                dtFrom = datetime.now() - timedelta(days=maxRange)
                entries = dbstore.Entry.select(
                    dbstore.Entry.q.updated > dtFrom,
                    orderBy = "-updated")
                if maxSize:
                    entries = entries[:maxSize]

                content = {"feed": {"title": title,
                                    "link": url}}
                content["entries"] = [ self.toDict(e) for e in entries ]
                io = StringIO.StringIO()
                dumpRSS(content, io, "atom1.0")
                
                return io.getvalue()

        return PlanetResource()
    except Execption, e:
        log.error(e)

    return None
                


_RESOURCE_FACTORY_TABLE = {
    "static": createStaticResource,
    "hyperestraier": createHEResource,
    "planet": createPlanetResource,
}


class HabuRootResource(resource.Resource):
    def render_GET(self, request):
        content = ["""<html><body>
        <h1>Habu Web Menu</h1>
        <ul>"""]
        for entry in self.listStaticNames():
            content.append("<li><a href='%(name)s'>%(name)s</a></li>"
                           % {"name": entry})
        content.append("</ul></body></html>")
        return "\n".join(content)

    def getChild(self, name, request):
        if not name:
            return self
        return resource.Resource.getChild(self, name, request)

def createServerFactory(config):
    root = HabuRootResource()
    resourceConfigs = config.get("resource", [])
    for cfg in resourceConfigs:
        module = cfg.get("module", None)
        name = cfg.get("name", None)
        if module and name and _RESOURCE_FACTORY_TABLE.has_key(module):
            func = _RESOURCE_FACTORY_TABLE[module]
            res = func(cfg)
            if res:
                root.putChild(name, res)
                
    factory = server.Site(root)
    
    return factory
    

