from twisted.internet.defer import Deferred
from xml.sax.saxutils import escape
import datetime
import re
import habu.log as log


##################
# utility filters
#

class Join(object):
    def __init__(self):
        self.contents = []
        self.deferred = None

    def mergeContents(self):
        """
        merge all contents to one content.
        If you want to merge with your custom way,
        override this method.

        This method provide very simple merge.
        """
        if len(self.contents) == 1:
            return self.contents[0]
        
        content = self.contents.pop()
        for c in self.contents:
            content["entries"].extend(c["entries"])
            
        return content


    def eventFired(self, manager):
        if len(manager.executeContexts) == 1 and self.deferred:
            log.debug("fired next")
            manager.removeHook(manager.HOOK_TYPE_GOT_ERROR, self.eventFired)
            manager.removeHook(manager.HOOK_TYPE_SUCCESS, self.eventFired)
            self.deferred.callback(self.mergeContents())
        else:
            log.debug("habuutils.Join : event firerd(ignored)")


    def execute(self, content):
        manager = self.executeManager
        self.contents.append(content)
        log.debug("habuutils.Join : enter")
        if len(manager.executeContexts) == 1 and not self.deferred:
            log.debug("habuutils.Join : No joined threads. Return contents.")
            return self.mergeContents()
        elif not self.deferred:
            log.debug("habuutils.Join : Start to join thread.")
            self.deferred = Deferred()
            manager.addHook(manager.HOOK_TYPE_GOT_ERROR, self.eventFired)
            manager.addHook(manager.HOOK_TYPE_SUCCESS, self.eventFired)
            return self.deferred

        log.info("Thread Joined")
        return None


#####################
# utility methods
#

def mergeResult(content, subModules, arg = None):
    if not isinstance(content, list):
        content = [content]

    for module in subModules:
        result = module.execute(arg)
        if not result:
            pass
        elif not isinstance(result, list):
            content.append(result)
        else:
            content.extend(result)

    return content


def configGet(config, environ, key, value=None):
    if config.has_key(key):
        return config[key]

    return environ.get(key, value)

def _check(l, e):
    if e not in l:
        l.append(e)
        return True
    return False

def uniq(mylist):
    a = []
    return [e for e in mylist if _check(a, e)]


_RSS_HEADER = """<?xml version="1.0" encoding="utf-8"?>
<rss version="2.0">
<channel>
  <title>%(title)s</title>
  <link>%(link)s</link>
"""

_RSS_FOOTER = """</channel>
</rss>"""

_ITEM = """
  <item>
    <title>%(title)s</title>
    <link>%(link)s</link>
    <pubDate>%(updated)s</pubDate>
    <description>%(description)s</description>
  </item>
"""

def toUTF8(s):
    if isinstance(s, unicode):
        s = s.encode("utf-8")
    return s

def _enc(s, no_escape = False):
    if type(s) == unicode:
        s = s.encode("utf-8")
    if no_escape:
        return s
    return escape(s)

_TAG_STRIPPER = re.compile("<.*?>", re.M | re.S | re.U)
def _escapeEntry(entry, otherContent = None):
    d = {"title": _enc(entry["title"]),
         "link":  _enc(entry["link"])}
    if entry.has_key("updated"):
        d["updated"] = _enc(entry["updated"])
    else:
        d["updated"] = str(datetime.datetime.now())
    if entry.has_key("summary_detail") \
            and entry["summary_detail"]["type"] == "text/html":
        summary = "<![CDATA[%s]]>" % _enc(entry["summary_detail"]["value"], True)
    else:
        summary = _enc(entry["summary"])
    d["description"] = summary
    d["summary"] = _TAG_STRIPPER.sub("", _enc(entry["summary"], True))
    d["author"] = _enc(entry.get("author", ""))
    if otherContent:
        d["other"] = otherContent(entry)
    else:
        d["other"] = ""
    
    return d

def _dumpAsRSS2(content, output):
    feed = content.get("feed", {})
    output.write(_RSS_HEADER %
                {"title": _enc(feed.get("title", "Habu RSS")),
                 "link": _enc(feed.get("link", "http://localhost/"))
                 }
                )
    for entry in content.get("entries", []):
        output.write(_ITEM % _escapeEntry(entry))
    output.write(_RSS_FOOTER)

_ATOM1_0_HEADER = """<?xml version="1.0" encoding="utf-8"?>
<feed xmlns="http://www.w3.org/2005/Atom" xmlns:py="http://purl.org/kid/ns#">	
  <id>%(link)s</id>
  <title>%(title)s</title>
  <link rel="self" href="%(link)s" />
"""
_ATOM1_0_ITEM = """
<entry>
  <title>%(title)s</title>
  <link href="%(link)s" rel="alternate" />
  <updated>%(updated)s</updated>
  <summary>%(summary)s</summary>
  <author><name>%(author)s</name></author>
  %(other)s
</entry>

"""
_ATOM1_0_FOOTER = "</feed>"


_Dummy = """ "#
"""

def _atomContent(entry):
    if entry.has_key("summary_detail") \
            and entry["summary_detail"]["type"] == "text/html":
        summary = "<![CDATA[%s]]>" % _enc(entry["summary_detail"]["value"], True)
        return '<content type="text/html">' + summary + "</content>"
    
    return ""

def _dumpAsAtom1_0(content, output):
    feed = content.get("feed", {})
    output.write(_ATOM1_0_HEADER %
                {"title": _enc(feed.get("title", "Habu RSS")),
                 "link": _enc(feed.get("link", "http://localhost/"))
                 }
                )
    for entry in content.get("entries", []):
        output.write(_ATOM1_0_ITEM % _escapeEntry(entry, _atomContent))
    output.write(_ATOM1_0_FOOTER)


def dumpRSS(content, output, format = "rss2.0"):
    dumper = _RSS_FORMATTER_TABLE[format]
    dumper(content, output)

_RSS_FORMATTER_TABLE = {
    "rss2.0": _dumpAsRSS2,
    "atom1.0": _dumpAsAtom1_0,
}



#####################
# decorators
#


class HabuDecoratorBase(object):
    def __init__(self, *args, **kwds):
        self.args = args
        self.kwds = kwds

    def before(self, content):
        return content

    def after(self, content):
        return content

    def __call__(self, func):
        def modifiedFunc(obj, content):
            content = self.before(content)
            result = func(obj, content)
            return self.after(result)

        return modifiedFunc

class Sort(HabuDecoratorBase):
    def __init__(self, *args, **kwds):
        super(Sort, self).__init__(*args, **kwds)

    def before(self, content):
        key = self.kwds.get("key", "updated_parsed")
        reverse = self.kwds.get("reverse", False)
        entries = content.get("entries", [])
        
        if callable(key):
            getkey = key
        else:
            def getkey(a):
                return a[key]
        
        entries.sort(key=getkey, reverse=reverse)
        return content


def evaluate(item, expr):
    if isinstance(expr, str) or isinstance(expr, unicode):
        return item.find(expr) > -1
    
    if callable(expr):
        return expr(item)
    
    if hasattr(expr, "search") and  expr.search(item):
        return True

    return False

class Grep(HabuDecoratorBase):
    """
    one of conditions match, filter it
    """

    def __init__(self, exclusive = False, *args, **kwds):
        super(Grep, self).__init__(*args, **kwds)
        self.exclusive = exclusive

    def iterEntries(self, content):
        for entry in content.get("entries", []):
            found = 0
            total = 0
            for key, value in self.kwds.iteritems():
                total += 1
                item = entry.get(key, None)
                if item == None:
                    if self.exclusive:
                        found += 1
                elif evaluate(item, value) != self.exclusive:
                    found +=1
                else:
                    found = 0
                    break
            if found == total:
                yield entry

    def before(self, content):
        content["entries"] = [ entry for entry in self.iterEntries(content) ]
        
        return content

