from datetime import datetime
import time
from sqlobject import *
import log


class Feed(SQLObject):
    url = StringCol(alternateID = True)
    name = UnicodeCol(unique = True)
    entries = MultipleJoin('Entry')

class Entry(SQLObject):
    url = StringCol(alternateID = True)
    title = UnicodeCol(default = "")
    updated = DateTimeCol(default = datetime.now)
    format = StringCol(default = "text/plain")
    summary = UnicodeCol(default = "")
    author = UnicodeCol(default = "")
    feed = ForeignKey('Feed')

    updatedIndex = DatabaseIndex(updated)


def toUnicode(s):
    if isinstance(s, unicode):
        return s
    else:
        return s.decode("utf-8")

def updateEntries(content):
    conn = sqlhub.getConnection()
    transaction = conn.transaction()
    try:
        start = time.time()
        _updateEntryInTransaction(content)
        end = time.time()
        log.debug("Database Update Time : " + str(end-start))
        transaction.commit()
    except Exception, e:
        log.error(str(e))
        transaction.rollback()
        raise e

def _updateEntryInTransaction(content):
    feed = content["feed"]
    sResult = Feed.selectBy(url=feed["link"])
    if sResult.count() == 0:
        feedCol = Feed(url = feed["link"], name = toUnicode(feed["title"]))
    else:
        feedCol = sResult[0]
        feedCol.set(name=toUnicode(feed["title"]))

    for entry in content["entries"]:
        sResult = Entry.selectBy(url = entry["link"])
        if sResult.count() == 0:
            entryCol = Entry(url = entry["link"], feed = feedCol)
        else:
            entryCol = sResult[0]
        
        title = toUnicode(entry["title"])
        author = toUnicode(entry.get("author", ""))
        if not author:
            author = toUnicode(feed.get("author", ""))
        detail = entry.get("summary_detail", None)
        if detail:
            format = detail["type"]
            summary = toUnicode(detail["value"])
        else:
            format = "text/plain"
            summary = toUnicode(entry["summary"])

        if entry.has_key("updated_parsed"):
            tm = time.mktime(entry["updated_parsed"])
            updated = datetime.fromtimestamp(tm)
        else:
            updated = datetime.now()
        entryCol.set(updated = updated, format = format, summary = summary,
                     title = title, author = author, feed = feedCol)


def selectEntryByUrl(url):
    sResult = Entry.selectBy(url = url)
    count = sResult.count()
    if sResult.count() == 0:
        return None
    assert count == 1
    return sResult[0]
    
def clearAll():
    for tbl in (Feed, Entry):
        tbl.dropTable()
        tbl.createTable()


_available = False

class HabuDatabaseError(Exception):
    """base exception class for habudb"""

def initialize(db_uri):
    if not db_uri:
        return False
    if isAvailable():
        raise HabuDatabaseError("Database can be initalize only once.")

    connection = connectionForURI(db_uri)
    sqlhub.processConnection = connection
    
    for tbl in (Feed, Entry):
        tbl.createTable(ifNotExists = True)
    
    global _available
    _available = True
    return _available



def isAvailable():
    return _available
