from twisted.trial.unittest import TestCase
from datetime import datetime
from habu.habuutils import dumpRSS

from habu.filter import grep
from habu.filter import first
from habu.filter import head
from habu.filter import reverse
from habu.filter import sort
from habu.filter import subs
from habu.filter import unread
from habu.filter import adjust_date
from habu.filter import delete_dup
from habu.filter import moblogformatter
from habu.filter import rssload
from habu.filter import format_title


# not tested
# unread (0.3)
## join

class GrepTest(TestCase):
    def testGrepStr(self):
        g = grep.Grep({"str": {"title": "ab"}}, None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123abc")
    

    def testGrepRe(self):
        g = grep.Grep({"re": {"title": "ab"}}, None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123abc")
    

    def testGrepEval(self):
        g = grep.Grep({"eval": {"title": "entry.find('ab') != -1"}}, None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123abc")

    def testGrepExclusiveStr(self):
        g = grep.Grep({"str": {"title": "ab"}, "exclusive": True}, None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123bac")
    

    def testGrepExclusiveRe(self):
        g = grep.Grep({"re": {"title": "ab"}, "exclusive": True}, None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123bac")
    

    def testGrepExclusiveEval(self):
        g = grep.Grep({"eval": {"title": "entry.find('ab') != -1"}, 
                       "exclusive": True},
                      None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123bac")

    
    def testGrepAll(self):
        g = grep.Grep({"str": {"title": "ab", "summary":"hoge"},
                       "re": {"title": "[1-9]*ab"},
                       "eval": {"summary": "entry.find('hoge') == 0"}
                       }, None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123abc")

    def testGrepExclusiveAll(self):
        g = grep.Grep({"str": {"title": "ab", "summary":"hoge"},
                       "re": {"title": "[1-9]*ab"},
                       "eval": {"summary": "entry.find('hoge') == 0"},
                       "exclusive": True
                       }, None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123bac")

    def testGrepNotFoundField(self):
        g = grep.Grep({"str": {"title1": "entry.find('ab') != -1"}, 
                       },
                      None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 0)

    def testGrepExclusiveNotFoundField(self):
        g = grep.Grep({"str": {"title1": "entry.find('ab') != -1"}, 
                       "exclusive": True},
                      None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 2)
        
    def testGrepOneof(self):
        g = grep.Grep({"str": {"title": "ab", "summary":"hoged"},
                       }, None)
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 0)

    def testGrepExclusiveOneof(self):
        g = grep.Grep({"str": {"title": "ab", "summary":"hoged"},
                       "exclusive": True
                       }, None)
        
        content = g.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)



class FirstTest(TestCase):
    def testNormal(self):
        f = first.First(None, None)
        content = f.execute({"entries": [
                    {"title": "123abc",
                     "summary": "hoge"},
                    {"title": "123bac",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "123abc")

    def testNoEntry(self):
        f = first.First(None, None)
        content = f.execute({"entries": []})
        self.assertEquals(len(content["entries"]), 0)



class HeadTest(TestCase):
    def testNormal(self):
        f = head.Head({"start": 1, "end": 3}, None)
        content = f.execute({"entries": [
                    {"title": "0",
                     "summary": "hoge"},
                    {"title": "1",
                     "summary": "123abc"},
                    {"title": "2",
                     "summary": "hoge"},
                    {"title": "3",
                     "summary": "123abc"},
                    {"title": "4",
                     "summary": "hoge"},
                    {"title": "5",
                     "summary": "123abc"},
                    {"title": "6",
                     "summary": "hoge"},
                    {"title": "7",
                     "summary": "123abc"},
                    {"title": "8",
                     "summary": "hoge"},
                    {"title": "9",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 2)
        self.assertEquals(content["entries"][0]["title"], "1")
        self.assertEquals(content["entries"][1]["title"], "2")


    def testNoEnd(self):
        f = head.Head({"start": 1, }, None)
        content = f.execute({"entries": [
                    {"title": "0",
                     "summary": "hoge"},
                    {"title": "1",
                     "summary": "123abc"},
                    {"title": "2",
                     "summary": "hoge"},
                    {"title": "3",
                     "summary": "123abc"},
                    {"title": "4",
                     "summary": "hoge"},
                    {"title": "5",
                     "summary": "123abc"},
                    {"title": "6",
                     "summary": "hoge"},
                    {"title": "7",
                     "summary": "123abc"},
                    {"title": "8",
                     "summary": "hoge"},
                    {"title": "9",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 9)
        self.assertEquals(content["entries"][0]["title"], "1")
        self.assertEquals(content["entries"][1]["title"], "2")
        self.assertEquals(content["entries"][-1]["title"], "9")


    def testNoStart(self):
        f = head.Head({"end": 2, }, None)
        content = f.execute({"entries": [
                    {"title": "0",
                     "summary": "hoge"},
                    {"title": "1",
                     "summary": "123abc"},
                    {"title": "2",
                     "summary": "hoge"},
                    {"title": "3",
                     "summary": "123abc"},
                    {"title": "4",
                     "summary": "hoge"},
                    {"title": "5",
                     "summary": "123abc"},
                    {"title": "6",
                     "summary": "hoge"},
                    {"title": "7",
                     "summary": "123abc"},
                    {"title": "8",
                     "summary": "hoge"},
                    {"title": "9",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 2)
        self.assertEquals(content["entries"][0]["title"], "0")
        self.assertEquals(content["entries"][1]["title"], "1")
        self.assertEquals(content["entries"][-1]["title"], "1")


    def testBadSeq(self):
        f = head.Head({"start": -1, "end": 0, }, None)
        content = f.execute({"entries": [
                    {"title": "0",
                     "summary": "hoge"},
                    {"title": "1",
                     "summary": "123abc"},
                    {"title": "2",
                     "summary": "hoge"},
                    {"title": "3",
                     "summary": "123abc"},
                    {"title": "4",
                     "summary": "hoge"},
                    {"title": "5",
                     "summary": "123abc"},
                    {"title": "6",
                     "summary": "hoge"},
                    {"title": "7",
                     "summary": "123abc"},
                    {"title": "8",
                     "summary": "hoge"},
                    {"title": "9",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 0)

    def testBigEnd(self):
        f = head.Head({"end": 100, }, None)
        content = f.execute({"entries": [
                    {"title": "0",
                     "summary": "hoge"},
                    {"title": "1",
                     "summary": "123abc"},
                    {"title": "2",
                     "summary": "hoge"},
                    {"title": "3",
                     "summary": "123abc"},
                    {"title": "4",
                     "summary": "hoge"},
                    {"title": "5",
                     "summary": "123abc"},
                    {"title": "6",
                     "summary": "hoge"},
                    {"title": "7",
                     "summary": "123abc"},
                    {"title": "8",
                     "summary": "hoge"},
                    {"title": "9",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 10)


class ReverseTest(TestCase):
    def testBigEnd(self):
        f = reverse.Reverse(None, None)
        content = f.execute({"entries": [
                    {"title": "0",
                     "summary": "hoge"},
                    {"title": "1",
                     "summary": "123abc"},
                    {"title": "2",
                     "summary": "hoge"},
                    {"title": "3",
                     "summary": "123abc"},
                    {"title": "4",
                     "summary": "hoge"},
                    {"title": "5",
                     "summary": "123abc"},
                    {"title": "6",
                     "summary": "hoge"},
                    {"title": "7",
                     "summary": "123abc"},
                    {"title": "8",
                     "summary": "hoge"},
                    {"title": "9",
                     "summary": "123abc"}
                    ]})
        self.assertEquals(len(content["entries"]), 10)
        self.assertEquals(content["entries"][0]["title"], "9")
        self.assertEquals(content["entries"][-1]["title"], "0")


class SubsTest(TestCase):
    def testNormal(self):
        f = subs.Subs([{"key": "title",
                        "re": "([0-9]+)",
                        "to": " b "},
                       {"key": "summary",
                        "re": "([0-9]+)",
                        "to": ""},], None)
        content = f.execute({"entries": [
                    {"title": "a0c",
                     "summary": "hoge"},
                    {"title": "a1c",
                     "summary": "123abc"}]})
        self.assertEquals(content["entries"][0]["title"], "a b c")
        self.assertEquals(content["entries"][0]["summary"], "hoge")
        self.assertEquals(content["entries"][1]["title"], "a b c")
        self.assertEquals(content["entries"][1]["summary"], "abc")
        
    def testKey(self):
        f = subs.Subs([{"key": "title2",
                        "re": "([0-9]+)",
                        "to": " b "},], None)
        content = f.execute({"entries": [
                    {"title": "a0c",
                     "summary": "hoge"},
                    {"title": "a1c",
                     "summary": "123abc"}]})
        self.assertEquals(content["entries"][0]["title2"], "")
        

class SortTest(TestCase):
    def testNormal(self):
        f = sort.Sort({"key": "title"}, None)
        content = f.execute({"entries": [
                    {"title": "acb",
                     "summary": "hoge"},
                    {"title": "cab",
                     "summary": "hoge"},
                    {"title": "abc",
                     "summary": "123abc"}]})
        self.assertEquals(content["entries"][0]["title"], "abc")
        self.assertEquals(content["entries"][1]["title"], "acb")
        self.assertEquals(content["entries"][2]["title"], "cab")

        f = sort.Sort({"key": "title", "reverse": True}, None)
        content = f.execute({"entries": [
                    {"title": "acb",
                     "summary": "hoge"},
                    {"title": "cab",
                     "summary": "hoge"},
                    {"title": "abc",
                     "summary": "123abc"}]})
        self.assertEquals(content["entries"][0]["title"], "cab")
        self.assertEquals(content["entries"][1]["title"], "acb")
        self.assertEquals(content["entries"][2]["title"], "abc")

    def testNoKey(self):
        f = sort.Sort({"key": "title2"}, None)
        self.assertRaises(KeyError, f.execute, {"entries": [
                    {"title": "acb",
                     "summary": "hoge"},
                    {"title": "cab",
                     "summary": "hoge"},
                    {"title": "abc",
                     "summary": "123abc"}]})
    

class RssLoadTest(TestCase):
    def setUp(self):
        content = {"feed": {"title": "my feed",
                            "link": "http://localhost/"},
                   "entries": [{"title": "abc",
                                "summary": "123abc",
                                "link": "http://localhost/abc"
                                }]}
        fp = open("testrssload.xml", "w")
        dumpRSS(content, fp)
        fp.close()
        
    def tearDown(self):
        import os
        os.unlink("testrssload.xml")

    def testNormal(self):
        f = rssload.RssLoader({"path": "testrssload.xml"}, None)
        content = f.execute({"entries": [
                    {"title": "acb",
                     "summary": "hoge"},
                    ]})
        self.assertEquals(len(content["entries"]), 2)
        self.assertEquals(content["entries"][0]["title"], "abc")
        self.assertEquals(content["entries"][1]["title"], "acb")

    def testNoFile(self):
        f = rssload.RssLoader({"path": "testrssload123.xml"}, None)
        content = f.execute({"entries": [
                    {"title": "acb",
                     "summary": "hoge"},
                    ]})
        self.assertEquals(len(content["entries"]), 1)
        self.assertEquals(content["entries"][0]["title"], "acb")

class AdjustDate(TestCase):
    def testNormal(self):
        f = adjust_date.AdjustDate({}, None)
        content = {
            "entries": [
                {"link": "http://localhost/1",
                 "updated": "2007/09/15 01:23:00",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/1",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/1",
                 "updated": "2007-09-15T01:23:00Z",},
                {"link": "http://localhost/1",
                 }
                ]
            }
        content = f.execute(content)
        entries = content["entries"]
        for i in range(0, 3):
            self.assertEquals(entries[i]["updated"], "2007-09-15T01:23:00Z")
            self.assertEquals(entries[i]["updated_parsed"][:6], (2007, 9, 15, 1, 23, 0))
        self.assertEquals(not entries[3]["updated"], False)
        self.assertEquals(not entries[3]["updated_parsed"], False)

    def testDb(self):
        f = adjust_date.AdjustDate({"use_db": True}, None)
        content = {
            "entries": [
                {"link": "http://localhost/1",
                 }
                ]
            }
        from habu import habudb
        if not habudb.isAvailable():
            habudb.initialize("sqlite:///memory")
        habudb.clearAll()
        now = datetime.now()
        feed = habudb.Feed(url="http://localhost", name="test feed")
        habudb.Entry(url="http://localhost/1",
                     title="title1", updated=now, feed=feed)
        content = f.execute(content)
        entries = content["entries"]
        timeTuple = now.timetuple()
        nowStr = "%d-%02d-%02dT%02d:%02d:%02dZ" % timeTuple[:6]
        self.assertEquals(entries[0]["updated"], nowStr)
        self.assertEquals(entries[0]["updated_parsed"], timeTuple)
        habudb.clearAll()

class UnreadTest(TestCase):
    def testMemory(self):
        f = unread.UnreadFilter({}, None)
        content = {
            "entries": [
                {"link": "http://localhost/1",
                 "updated": "2007/09/15 01:23:00",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/2",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/3",
                 }
                ]
            }
        content = f.execute(content)
        entries = content["entries"]
        self.assertEquals(len(entries), 3)
        content = {
            "entries": [
                {"link": "http://localhost/1",
                 "updated": "2007/09/15 01:23:01",
                 "updated_parsed": (2007, 9, 15, 1, 23, 1),},
                {"link": "http://localhost/2",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/3",
                 }
                ]
            }
        content = f.execute(content)
        entries = content["entries"]
        self.assertEquals(len(entries), 1)

    def testDb(self):
        from habu import habudb
        if not habudb.isAvailable():
            habudb.initialize("sqlite:///memory")
        habudb.clearAll()
        
        f = unread.UnreadFilter({"use_db": True}, None)
        content = {
            "entries": [
                {"link": "http://localhost/1",
                 "updated": "2007/09/15 01:23:00",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/2",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/3",
                 }
                ]
            }
        content = f.execute(content)
        entries = content["entries"]
        self.assertEquals(len(entries), 3)

        now = datetime.now()
        feed = habudb.Feed(url="http://localhost", name="test feed")
        habudb.Entry(url="http://localhost/1",
                     title="title1", updated=now, feed=feed)
        
        content = {
            "entries": [
                {"link": "http://localhost/1",
                 "updated": "2007/09/15 01:23:00",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/2",
                 "updated_parsed": (2007, 9, 15, 1, 23, 0),},
                {"link": "http://localhost/3",
                 }
                ]
            }
        content = f.execute(content)
        entries = content["entries"]
        self.assertEquals(len(entries), 2)

        habudb.clearAll()


class DuplicateFilterTest(TestCase):
    def testNormarl(self):
        f = delete_dup.DeleteDuplicatedLink(None, None)
        content = {
            "entries": [
                {"link": "http://localhost/#1"},
                {"link": "http://localhost/#12"},
                {"link": "http://localhost/#1",},
                ]
            }
        content = f.execute(content)
        entries = content["entries"]
        self.assertEquals(len(entries), 2)
        self.assertEquals(entries[0]["link"], "http://localhost/#1")
        self.assertEquals(entries[1]["link"], "http://localhost/#12")


class MoblogFormatter(TestCase):
    def testNormal(self):
        f = moblogformatter.MoblogFormatter({"secretkey": "himitsu"}, None)
        content = {
            "entries": [
                {"summary": "himitsu\n\nmy body", "file":["dummy"]},
                {"summary": "himitsu\n\n", "file":["dummy"]},
                {"summary": "himitsu\n\nmy body", "file":[]},
                {"summary": "himitsu\n\nmy body"},
                {"summary": "himitsu\nmy body", "file":["dummy"]},
                {"summary": "himitsujanai\nmy body", "file":["dummy"]},
                {"summary": "", "file": ["dummy"]},
                {"summary": "himitsu\n\nmy body", "file":["dummy"], 
                 "summary_detail": {}},
                ]
            }
        content = f.execute(content)
        entries = content["entries"]
        self.assertEquals(len(entries), 2)
        self.assertEquals(entries[0]["summary"], "my body")
        self.assertEquals(entries[1]["summary"], "my body")
        self.assertEquals(entries[1]["summary_detail"]["value"], "my body")
        self.assertEquals(entries[1]["summary_detail"]["type"], "text/plain")

class TitleFormatter(TestCase):
    def testNormal(self):
        f = format_title.TitleFormatter({"format": "%(title)s %(feed.title)s"}, None)
        content = {
            "feed": {"title": "feeder",
                     "link": "http://localhost"},
            "entries": [
                {"title": "title2", "file":["dummy"]},
                ]
            }
        content = f.execute(content)
        entries = content["entries"]
        self.assertEquals(entries[0]["title"], "title2 feeder")

    def testNoKey(self):
        f = format_title.TitleFormatter({"format": "%(title)s %(feed.titlse)s"}, None)
        content = {
            "feed": {"title": "feeder",
                     "link": "http://localhost"},
            "entries": [
                {"title": "title2", "file":["dummy"]},
                ]
            }
        self.assertRaises(KeyError, f.execute, content)

