import imp
import os
import os.path
from twisted.internet import reactor
from twisted.internet.defer import Deferred
from twisted.web.client import getPage
from twisted.python import logfile
import  habu.log as log
import feedparser
import yaml
import sys

import habu.cron
import habuutils
from habu.moduleloader import loadModule, loadModuleFromNet
from habu.server import HABU_SERVER_SETUP_FACTORY_TAB

CURRENT_PATH  = os.path.dirname(__file__)


class ExecuteContext(object):
    def __init__(self, executeManager, name, restInstances):
        self.executeManager = executeManager
        self.name = name
        self.restInstances = restInstances

    def _copy(self, refAdd=True):
        return ExecuteContext(
            self.executeManager, 
            self.name,
            self.restInstances[:])

    def hasNext(self):
        num = len(self.restInstances)
        log.debug("Remaining Context Num : %d" % num)
        return num > 0

    def next(self):
        return self.restInstances.pop(0)
    
        

class ExecuteManager(object):
    HOOK_TYPE_START = 1
    HOOK_TYPE_FOLK = 2
    # We need add end event
    HOOK_TYPE_SUCCESS = 3
    HOOK_TYPE_GOT_ERROR = 4

    
    def __init__(self, name, lineInstances):
        self.lineInstances = lineInstances
        for instance in lineInstances:
            setattr(instance, "executeManager", self)
        self.name = name
        self.executeContexts = []
        self.hooks = []

    def deferredWrapper(self, content):
        deferred = Deferred()
        reactor.callLater(0, deferred.callback, content)
        return deferred

    def addHook(self, hook_type, hook_func):
        self.hooks.append((hook_type, hook_func))
        return self

    def removeHook(self, hook_type, hook_func):
        hook = (hook_type, hook_func)
        if hook in self.hooks:
            self.hooks.remove(hook)

    def _fireHook(self, hook_type):
        myhooks = [ hook for hook in self.hooks
                    if hook[0] == hook_type]
        for hook in myhooks:
            log.debug("Event Fired : " + str(hook_type))
            hook[1](self)
    
    def execute(self):
        log.debug("Start Line : " + self.name)
        if not self.lineInstances:
            log.debug("End Line(no line instance) : " + self.name)
            return
        
        executeContext = ExecuteContext(
            self,
            self.name,
            self.lineInstances)

        self.executeContexts = [executeContext]
        self._fireHook(self.HOOK_TYPE_START)
        self.runNext(None, executeContext)

    def executeErrback(self, failure, executeContext):
        log.debug("End Line(%s) : %s" % (str(failure), self.name))
        self.executeContexts.remove(executeContext)
        self._fireHook(self.HOOK_TYPE_GOT_ERROR)

    def executeCallback(self, result, executeContext):
        log.debug("Get executeCallback : " + self.name)
        self._fireHook(self.HOOK_TYPE_SUCCESS)
        if result == None:
            log.debug("End Line(no callback data) : " + self.name)
            self.executeContexts.remove(executeContext)
            self._fireHook(self.HOOK_TYPE_SUCCESS)
            return

        if not executeContext or not executeContext.hasNext():
            log.debug("End Line : " + self.name)
            self.executeContexts.remove(executeContext)
            self._fireHook(self.HOOK_TYPE_SUCCESS)
            return

        self.runNext(result, executeContext)

        return result

    def runNext(self, result, executeContext):
        instance = executeContext.next()
        values = instance.execute(result)
        if not values:
            log.debug("End Line(no execute data) : " + self.name)
            self.executeContexts.remove(executeContext)
            self._fireHook(self.HOOK_TYPE_SUCCESS)
            return
        if not executeContext.hasNext():
            log.debug("End Line(no next instance) : " + self.name)
            self.executeContexts.remove(executeContext)
            self._fireHook(self.HOOK_TYPE_SUCCESS)
            return
        
        self.scheduleNext(values, executeContext)

    def scheduleNext(self, values, executeContext):
        if not isinstance(values, list):
            deferredList = [values]
        else:
            deferredList = values
        
        deferred = deferredList.pop(0)
        self.callbackNext(deferred, executeContext)

        for deferred in deferredList:
            log.debug("Folk Line : " + self.name)
            copiedContext = executeContext._copy()
            self.executeContexts.append(copiedContext)
            self._fireHook(self.HOOK_TYPE_FOLK)
            self.callbackNext(deferred, copiedContext)

    def callbackNext(self, deferred, executeContext):
        if not isinstance(deferred, Deferred):
            deferred = self.deferredWrapper(deferred)
        
        deferred.addCallback(self.executeCallback, executeContext)
        deferred.addErrback(self.executeErrback, executeContext)



def setupNullServer(config):
    return None


class Habu(object):
    def __init__(self):
        self.lineConfigs = None
        self.environ = None
        self.scheduler = None
        self.servers = []

    def load(self, yamlStream):
        config = yaml.load(yamlStream)
        self.environ = config.setdefault("global", None)
        self.lineConfigs = config.setdefault("pipeline", [])
        self._loadAfter(config)
        

    def _loadAfter(self, config):
        # plugin loading path sequence order is
        # 1. current directory
        # 2. plugin path in config file.
        # 3. this module directory path for the standard plugins.
        self.environ.setdefault("plugin_path", ["."])
        if "." not in self.environ["plugin_path"]:
            self.environ["plugin_path"].insert(0, ".")
        if CURRENT_PATH not in self.environ["plugin_path"]:
            self.environ["plugin_path"].append(CURRENT_PATH)
        
        scheduleConfig = config.get("scheduler", None)
        self.setupCron(scheduleConfig)

        dburi = self.environ.get("dburi", None)
        if dburi:
            dburi = dburi % {"current_dir_uri": os.getcwd()}
            try:
                import habu.habudb
                habu.habudb.initialize(dburi)
            except Exception, e:
                log.warn("database is not available")
                log.debug(e)

        serverConfig = config.get("server", None)
        self.setupServer(serverConfig)

    def addPluginPath(self, path=None):
        if not path:
            return
        l = path + self.environ["plugin_path"]
        self.environ["plugin_path"] = habuutils.uniq(l)

    def setLogLevel(self, level):
        self.environ["loglevel"] = level
    
    def setLogFile(self, path):
        if path:
            self.environ["log"] = path
    
    def setProxy(self, proxy_host = None, proxy_port = 0):
        self.environ["proxy_host"] = proxy_host
        self.environ["proxy_port"] = proxy_port

    def startLogging(self):
        output = self.environ.setdefault("log", "null")
        level = self.environ.get("loglevel", log.INFO)
        if not output or output == "null":
            return
        if output == "stdout":
            logFile = sys.stdout
        elif output == "stderr":
            logFile = sys.stderr
        else:
            import os.path
            basename = os.path.basename(output)
            dirname = os.path.dirname(output)
            logFile = logfile.DailyLogFile(basename, dirname)
        log.startLogging(logFile, level)

    def setupServer(self, serverConfig):
        if not serverConfig:
            return
        for key, config in serverConfig.iteritems():
            func = HABU_SERVER_SETUP_FACTORY_TAB.get(key, setupNullServer)
            server = func(config)
            if server:
                self.servers.append(server)

    def setupCron(self, scheduleConfig):
        if not scheduleConfig:
            return
        if not self.scheduler:
            log.debug("cron created")
            self.scheduler = habu.cron.Cron()
        for crontab in scheduleConfig:
            try:
                name = crontab.get("pipeline", None)
                expr = crontab.get("expression")
                log.debug("con expression(%s) : %s" % (name, expr))
                expr = habu.cron.parseExpression(expr)
                if name:
                    self.scheduler.add(expr, self.runJob, name)
                else:
                    log.error("ERROR: remove cron tab(name is not defined")
            except Exception, e:
                log.error("ERROR: remove cron tab(%d)" % name)

    def runJob(self, name):
        log.info("Run Job : " + name)
        if self.lineConfigs.has_key(name):
            self.run(name)

    def getPipelines(self):
        return self.lineConfigs

    def getPipelineNames(self):
        return self.lineConfigs.keys()

    def startServer(self):
        if not self.servers:
            return
        
        for server in self.servers:
            server.start()

    def startScheduler(self):
        if self.scheduler:
            self.scheduler.start()

    def runAll(self):
        return [self.run(key) for key in self.lineConfigs.keys()]

    def run(self, pipeline):
        line = self.lineConfigs[pipeline]
        lineInstances = [self.instanciate(lineModule) for lineModule in line]
        executeManager = ExecuteManager(pipeline,
                                        lineInstances)
        reactor.callLater(0, executeManager.execute)

        return executeManager

    def error(self, *args, **kwds):
        log.error("ERROR: ", *args, **kwds)

    def instanciate(self, lineModule):
        mod = self.loadModule(lineModule["module"])
        config = lineModule.setdefault("config", {})
        obj = mod.create(config, self.environ)
        if hasattr(obj, "setSubModules"):
            subModules = lineModule.get("sub-modules", [])
            subModuleObjs = [self.instanciate(module) for module in subModules]
            obj.setSubModules(subModuleObjs)
        
        return obj        

    def loadModule(self, moduleName):
        return loadModule(moduleName, self.environ.get("plugin_path"))

    def preloadModules(self, enableNetInstall = True):
        deferredList = []
        for configs in self.lineConfigs.values():
            for lineModule in configs:
                url = enableNetInstall and lineModule.get("url", None) or None
                mod = loadModuleFromNet(lineModule["module"],
                                        self.environ.get("plugin_path"),
                                        url,
                                        self.environ.get("proxy_host", None),
                                        self.environ.get("proxy_port", 0))
                if not mod:
                    raise "Failed to load Module %s" % lineModule["module"]
                if isinstance(mod, Deferred):
                    deferredList.append(mod)
        return deferredList

