/*******************************************************************************
 * Copyright (c) 2003, Michael Bartl
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Created on Jun 27, 2003
 *******************************************************************************/

package viPlugin.commands.search;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;

import viPlugin.TextModificator;

/**
 * Search related functions.
 * @author Michael Bartl
 */
public abstract class Search implements ISearch {
	public IDocument _doc;
	public ITextSelection _selection;
	public TextModificator _tm;
	public String _lastSearch;
	public static final int NEW = 0;
	public static final int NEXT = 1;
	public static final int PREVIOUS = 2;

	public Search() {
		_lastSearch = "";
	}

	public boolean searchNext() throws Exception {
		if (_lastSearch != null) {
			return search(_lastSearch, NEXT);
		}
		return false;
	}

	public boolean searchPrevious() throws Exception {
		if (_lastSearch != null) {
			return search(_lastSearch, PREVIOUS);
		}
		return false;
	}

	public boolean search(String str) throws Exception {
		_lastSearch = str;
		return search(str, NEW);
	}

	public void searchCurrentWord() throws Exception {
		_tm = TextModificator.getInstance();
		_doc = _tm.getDocument();
		_selection = _tm.getSelection();
		String text = _doc.get();
		int cursorPos = _selection.getOffset();
		cursorPos = skipToken(text, cursorPos);
		String word =
			text.substring(
				getPreviousDelimiter(text, cursorPos) + 1,
				getNextDelimiter(text, cursorPos));
		search("\\b" + word + "\\b", NEXT);
	}

	public void searchCurrentWordBackwards() throws Exception {
		_tm = TextModificator.getInstance();
		_doc = _tm.getDocument();
		String text = _doc.get();
		_selection = _tm.getSelection();
		int cursorPos = _selection.getOffset();
		// should not skip backward
		cursorPos = skipToken(text, cursorPos);
		String word =
			text.substring(
				getPreviousDelimiter(text, cursorPos) + 1,
				getNextDelimiter(text, cursorPos));
		search("\\b" + word + "\\b", PREVIOUS);
	}

	public int getPreviousDelimiter(String text, int cursorPos) {
		int pos = cursorPos;
		for (int i = cursorPos - 1; i > 0; i--) {
			if (!isWordCharacter(text.charAt(i))) {
				pos = i;
				break;
			}

		}
		return pos;
	}

	public int getNextDelimiter(String text, int cursorPos) {
		int pos = cursorPos;
		for (int i = cursorPos + 1; i < text.length(); i++) {
			if (!isWordCharacter(text.charAt(i))) {
				pos = i;
				break;
			}

		}
		return pos;
	}

	public int skipToken(String text, int cursorPos) {
		int pos = cursorPos;
		for (int i = cursorPos; i < text.length(); i++) {
			if (isWordCharacter(text.charAt(i))) {
				pos = i;
				break;
			}

		}
		return pos;
	}

	public int skipTokenBackwards(String text, int cursorPos) {
		int pos = cursorPos;
		for (int i = cursorPos - 1; i > 0; i--) {
			if (isWordCharacter(text.charAt(i))) {
				pos = i;
				break;
			}

		}
		return pos;
	}

	/**
	 * @return true if ch is [a-zA-Z0-9_] or other Unicode Letter.
	 */
	public boolean isWordCharacter(char ch) {
		boolean isPunctuation =
			(Character.getType(ch) == Character.CONNECTOR_PUNCTUATION);
		return Character.isLetterOrDigit(ch) || isPunctuation;
	}

	public abstract boolean search(String str, int mode) throws Exception;
}