/*******************************************************************************
 * Copyright (c) 2003, Michael Bartl
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Created on Jun 25, 2003
 *******************************************************************************/

package viPlugin.commands.search;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.BadLocationException;

import viPlugin.TextModificator;
import viPlugin.ViPluginPlugin;
import viPlugin.preferences.ViPrefNames;
import viPlugin.preferences.ViPreferenceService;

/**
 * Search related functions supporting regular expressions. Only works with v1.4
 * or higher!
 * 
 * @author Michael Bartl, Hiroyuki Shigenaga
 */
public class RegExSearch extends Search {

    public static boolean isSupported(boolean showInformation) {
        String version = System.getProperty("java.version");
        if (version.startsWith("1.4") || version.startsWith("1.5")) {
            return true;
        } else {
            if (showInformation) {
                MessageDialog.openInformation(TextModificator.getInstance()
                        .getTextEditor().getSite().getWorkbenchWindow()
                        .getShell(), "Upgrade your java installation",
                        "Only supported with Java 1.4 or higher!");
            }
            return false;
        }
    }

    public boolean search(final String regex, final int mode) throws Exception {
        try {
            if (regex == null) { throw new Exception(
                    "Got an empty string for searching!"); }

            _tm = TextModificator.getInstance();
            _doc = _tm.getDocument();
            _selection = _tm.getSelection();
            _lastSearch = regex;

            String text = _doc.get();
            int cursorPos = _selection.getOffset();

            if (mode == NEXT) {
                cursorPos++;
            } else if (mode == PREVIOUS) {
                //				cursorPos--;
            }

            int cursor = -1;
            String foundStr = "";
            if (mode == NEW || mode == NEXT) {
                Pattern pat = Pattern.compile(regex, getFlags());
                Matcher matcher = pat.matcher(text);
                // try to wrapscan if failed.
                if (matcher.find(cursorPos) || matcher.find()) {
                    cursor = matcher.start();
                    foundStr = matcher.group();
                }
            } else {
                // TODO: Reimplement if it's too slow.
                Pattern pat = Pattern.compile(regex, getFlags());
                Matcher matcher = pat.matcher(text);
                while (matcher.find()) {
                    int newCursor = matcher.start();
                    String newFoundStr = matcher.group();
                    if (newCursor >= cursorPos) {
                        break;
                    }
                    cursor = newCursor;
                    foundStr = newFoundStr;
                }
                // try to wrapscan (find last) if failed.
                if (cursor == -1) {
                    while (matcher.find()) {
                        cursor = matcher.start();
                        foundStr = matcher.group();
                    }
                }
            }
            if (cursor != -1) {
                _tm.setCaretPosition(cursor);
                _tm.setVisualSelection(cursor, foundStr.length());
            }
            return foundStr.length() > 0;
        } catch (PatternSyntaxException e) {
            e.printStackTrace();
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
        return false;
    }

    private int getFlags() {
        // needed for unit-tests
        if (ViPluginPlugin.getDefault() != null) {
            boolean ignoreCase = ViPreferenceService.getInstance().getBoolean(
                    ViPrefNames.IGNORECASE);

            int flags = Pattern.MULTILINE;
            if (ignoreCase) flags |= Pattern.CASE_INSENSITIVE;

            return flags;
        }
        return 0;
    }
}