package viPlugin;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.texteditor.ITextEditor;

/**
 * <p>
 * Manages <code>ViLayer</code> instances. Be aware that the <code>init()</code>
 * method must be called before the first call of <code>getInstance()</code>
 * method.
 * </p>.
 * 
 * @author <a href="mailto:emdot@seznam.cz">Mairtin Krauskopf</a>
 * @version $Revision: 1.2 $ $Date: 2003/11/19 11:47:18 $
 */
public class ViLayerFactory {

    private static Map viLayers = new HashMap();
    private IWorkbenchWindow window;
    private static ViLayerFactory viLayerFactory;

    /**
	 * Return the class instance.
	 * 
	 * @param window
	 * @return
	 */
    public static void init(IWorkbenchWindow window) {
        if (viLayerFactory == null) {
            viLayerFactory = new ViLayerFactory();
            viLayerFactory.window = window;
        }
    }

    /**
     * DOCDO
     *
     * @return
     */
    public static ViLayerFactory getInstance() {
        if (viLayerFactory == null) {
            throw new NullPointerException("ViLayerFactory.init() must be called first.");
        }
        return viLayerFactory;
    }

    /**
     * Activates layer for given editor.
     *
     * @param textEditor
     */
    public void activateLayer(ITextEditor textEditor) {
        String currLayerId = getLayerId(textEditor);
        ViLayer currViLayer = getLayerById(currLayerId);
        if (currViLayer == null) {
            currViLayer = new ViLayer(textEditor);
            viLayers.put(currLayerId, currViLayer);
        } else {
            TextModificator.activateInstance(currViLayer.getTextModificator());
        }
    }

    /**
     * <p>
     * Unload all loaded <code>ViLayer</code> instances.
     * </p>
     */
    public void unLoadAllLayers() {
        for (Iterator iter = viLayers.values().iterator(); iter.hasNext();) {
            ViLayer viLayer = (ViLayer) iter.next();
            viLayer.switchToInsertMode();
            viLayer.unLoad();
            viLayers = new HashMap();
        }
    }

    /**
	 * <p>
	 * Unload <code>ViLayer</code> instance for a currently activated
	 * texteditor.
	 * </p>
	 * 
	 * @param switchToSingleMode should we switch to insert mode before
	 *                    unloading?
	 */
	public void unLoadCurrentLayer(boolean switchToInsertMode) {
		ITextEditor currEditor = getCurrTextEditor();
		if (currEditor != null) {
			ViLayer viLayer = getLayerById(getLayerId(currEditor));
			if (viLayer != null) {
				if (switchToInsertMode) {
					viLayer.switchToInsertMode();
				}
				viLayer.unLoad();
			}
		}
	}

	/**
	 * DOCDO
	 *
	 * @param switchToInsertMode
	 */
	public void removeViLayer(ITextEditor editor) {
		if (editor != null) {
			String layerId = getLayerId(editor);
			ViLayer viLayer = getLayerById(layerId);
			if (viLayer != null) {
				viLayer.unLoad();
				viLayers.remove(layerId);
			}
		}
	}

    /**
     * Returns currently activated editor.
     *
     * @return currently activated editor
     */
    private ITextEditor getCurrTextEditor() {
        IEditorPart editor = window.getActivePage().getActiveEditor();
        if (editor instanceof ITextEditor) {
            return (ITextEditor) editor;
        } else {
            return null;
        }
    }

    private ViLayer getLayerById(String currLayerId) {
        return (ViLayer) viLayers.get(currLayerId);
    }

    /**
	 * Returns uniq id on the base of given <code>ITextEditor</code> instance
	 * (which is exactly what we need - one layer per editor instance).
	 * 
	 * @param textEditor
	 * @return
	 */
    private String getLayerId(ITextEditor textEditor) {
        // TODO toolTipText is not ideal as uniq idenifier (e.g. IPath would be
        // better)
        return textEditor.getEditorInput().getToolTipText();
    }
}
