/*******************************************************************************
 * Copyright (c) 2003, Michael Bartl
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Created on 26.05.2003
 *******************************************************************************/

package viPlugin;

import java.lang.reflect.Method;

import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.text.DefaultUndoManager;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.ITextViewerExtension;
import org.eclipse.jface.text.IUndoManager;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.widgets.Caret;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.texteditor.AbstractTextEditor;
import org.eclipse.ui.texteditor.ITextEditor;

import viPlugin.preferences.ViPrefNames;
import viPlugin.preferences.ViPreferenceService;

/**
 * @author Michael Bartl
 */
public class ViLayer implements ILayerInterface {

	public CommandBuffer commandBuffer;
	public ViMode mode;
	//	private ITextEditor textEditor;
	private IStatusLineManager slm;
	private StyledText _textWidget;
	private ITextViewer _textViewer;
	//	private IKeyBindingService _keyBindingService;
	private ViVerifyKeyListener _keyListener;
	private Caret _originalCaret;
	private Caret _vimCaret;

	private static final int CARET_THIN = 1;

	/** TextModificator for this layer */
	private TextModificator textModificator;

	public ViLayer(ITextEditor textEditor) {
		this.mode = ViMode.COMMAND_MODE;
		IUndoManager undoManager = new DefaultUndoManager(100);

		// implemented for i18n (i.e. input Unicode character in search word.)
		// TODO: evil hack!
		// keyVector.add(257 to 65535) is too slow to input any key.
		// ViEditor can use getSourceViewer, but ViLayer cannot,
		// so I use java.lang.refletion reluctantly. -Shige-
		try {
			Method me =
				AbstractTextEditor.class.getDeclaredMethod(
					"getSourceViewer",
					null);
			me.setAccessible(true);
			Object viewer = me.invoke(textEditor, null);

			_keyListener = new ViVerifyKeyListener(this);
			((ITextViewerExtension) viewer).prependVerifyKeyListener(
				_keyListener);

			_textViewer = (ITextViewer) viewer;
			_textViewer.setUndoManager(undoManager);

			undoManager.connect(_textViewer);

			_textWidget = _textViewer.getTextWidget();
		} catch (Exception e1) {
			e1.printStackTrace();
		}

		IEditorSite editorSite = textEditor.getEditorSite();
		slm = editorSite.getActionBars().getStatusLineManager();

		// _keyBindingService = textEditor.getSite().getKeyBindingService();

		try {
			textModificator = new TextModificator(textEditor, _textViewer);
			TextModificator.activateInstance(textModificator);
		} catch (Exception e) {
			// TODO: do something more proper
			e.printStackTrace();
		}

		// save original caret
		_originalCaret = _textWidget.getCaret();
		// create VIM style caret
		_vimCaret =
			new Caret(_originalCaret.getParent(), _originalCaret.getStyle());
		_vimCaret.setImage(null);
		_textWidget.setCaret(_vimCaret);

		commandBuffer = new CommandBuffer(this);
		commandBuffer.setUndoManager(undoManager);

		switchToCommandMode();
		setStatusLine();
	}

	void setStatusLine() {
		String message = new String();
		if (mode == ViMode.COMMAND_MODE) {
			message =
				"CommandMode: CommandBuffer = " + commandBuffer.getCommand();
		} else if (mode == ViMode.INSERT_MODE) {
			message = "InsertMode";
		} else if (mode == ViMode.VISUAL_MODE) {
			message = "VisualMode";
		} else {
			message = "MODE NOT SUPPORTED - SOMETHING IS VERY WRONG!";
		}
		slm.setMessage(message);
	}

	public void switchToCommandMode() {
		mode = ViMode.COMMAND_MODE;
		commandBuffer.endCompoundChange();
		commandBuffer.clear();
		setStatusLine();
		setCaretThick();
		resetSelection();
	}

	public void switchToInsertMode() {
		mode = ViMode.INSERT_MODE;
		setStatusLine();
		commandBuffer.clear();
		commandBuffer.beginCompoundChange();
		setCaretThin();
	}

	public void switchToVisualMode() {
		mode = ViMode.VISUAL_MODE;
		TextModificator.getInstance().activateVisualMode();
		setStatusLine();
		commandBuffer.clear();
		commandBuffer.beginCompoundChange();
		setCaretThick();
	}

	public void leaveVisualMode() {
		switchToCommandMode();
		TextModificator.getInstance().deactivateVisualMode();
	}

	/**
	 * Completely unload the viLayer.
	 */
	public void unLoad() {
		commandBuffer.clear();
		((ITextViewerExtension) _textViewer).removeVerifyKeyListener(
			_keyListener);
		setStatusLine();
		// restore original cursor
		if (!_textWidget.isDisposed()) {
			_textWidget.setCaret(_originalCaret);
		}
	}

	private void setCaretWidth(int width) {
		_vimCaret.setSize(width, _vimCaret.getSize().y);
	}

	private void setCaretThin() {
		setCaretWidth(CARET_THIN);
	}

	private void setCaretThick() {
		boolean vimStyleCursor =
			ViPreferenceService.getInstance().getBoolean(ViPrefNames.VIMCURSOR);
		if (vimStyleCursor) {
			GC gc = new GC(_textWidget);
			setCaretWidth(gc.getFontMetrics().getAverageCharWidth());
			gc.dispose();
		}
	}

	private void resetSelection() {
		TextModificator.getInstance().setVisualSelection(
			_textWidget.getCaretOffset(),
			0);
	}

	/**
	 * @return Returns the textModificator.
	 */
	public TextModificator getTextModificator() {
		return textModificator;
	}

	public boolean isInInsertMode() {
		return mode == ViMode.INSERT_MODE;
	}

	public boolean isInVisualMode() {
		return mode == ViMode.VISUAL_MODE;
	}

	public boolean isInCommandMode() {
		return mode == ViMode.COMMAND_MODE;
	}

	public void switchVisualMode() {
		if (isInVisualMode()) {
			leaveVisualMode();
		} else {
			switchToVisualMode();
		}
	}
}
