/*******************************************************************************
 * Copyright (c) 2003, Michael Bartl
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 	
 * Created on 26.05.2003
 *******************************************************************************/

package viPlugin;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.ui.texteditor.ITextEditor;

/**
 * TextModificator is a utility class for all kinds of operations on text
 * strings. Sooner or later this will be split into the single Commands of the
 * viplugin.commands package, ViDocument class and a TextUtil class.
 * 
 * @author <a href="mailto:zeddicus@satokar.com">Michael Bartl </a>
 * @author <a href="mailto:emdot@seznam.cz">Martin Krauskopf </a>
 * @version $Revision: 1.47 $ $Date: 2004/04/13 22:17:34 $
 */
public class TextModificator {

    private static TextModificator _currTextModificator;

    private ITextEditor _textEditor;
    private ITextViewer _textViewer;
    private IDocument _document;
    private YankBuffer _yankBuffer;

    // Only one of those selections can be active at a time
    /** This selection is used in visual mode */
    private ITextSelection _visualSelection;
    /** This selection is not visible to the user */
    private ITextSelection _internalSelection;

    private boolean _deleteLastDelimiter;
    private char _lastSearchCh;
    private boolean _doesSearchChBackward;
    private boolean _isSearchChInclusive;
    private boolean _isInVisualMode;
    private int _visualAnchor;

    /**
     * Creates initialized instance.
     * 
     * @param textEditor
     *            text editor
     * @param textViewer
     *            text viewer
     */
    public TextModificator(ITextEditor textEditor, ITextViewer textViewer) {
        _textEditor = textEditor;
        _textViewer = textViewer;
        _yankBuffer = new YankBuffer("<default>");
    }

    public static void activateInstance(TextModificator textModificator) {
        _currTextModificator = textModificator;
    }

    public static TextModificator getInstance() {
        return _currTextModificator;
    }

    /**
     * This method servers <b>only </b> for testing purposes. Should be used
     * nowhere else.
     */
    public static void setEmptyInstance(IDocument doc) {
        activateInstance(new TextModificator(null, null));
        _currTextModificator.setDocument(doc);
        _currTextModificator.setSelection(new TextSelection(doc, 0, 0));
    }

    public ITextEditor getTextEditor() {
        return _textEditor;
    }

    public ITextSelection getSelection() {
        if (_isInVisualMode) {
            return _visualSelection;
        }
        else {
            return _internalSelection;
        }
    }

    public IDocument getDocument() {
        return _document;
    }

    public YankBuffer getYankBuffer() {
        return _yankBuffer;
    }

    /**
     * Returns the line of the current cursor position.
     * 
     * @return the line of the current cursor position
     */
    public IRegion getLine() {
        try {
            return _document.getLineInformation(getLinePos());
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Returns the line number of the current cursor position.
     * 
     * @return the line number of the current cursor position
     */
    public int getLinePos() {
        try {
            return _document.getLineOfOffset(getCaretPosition());
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    /**
     * Get the delimiter of the first line, or a legal delimiter of none is
     * present.
     * 
     * @return The delimiter used in this document or a legal delimiter if none
     *         is present.
     */

    public String getDelimiter() {
        String delim;
        try {
            delim = _document.getLineDelimiter(0);
        }
        catch (BadLocationException e) {
            delim = _document.getLegalLineDelimiters()[0];
        }
        if (delim == null) {
            delim = _document.getLegalLineDelimiters()[0];
        }

        return delim;
    }

    /**
     * Get the delimiter of the given line.
     * 
     * @param linePos
     *            The position of the line.
     * @return The delimiter of the current line or an emptry String if there is
     *         none.
     */

    public String getLineDelimiter(int linePos) {
        String delim;
        try {
            delim = _document.getLineDelimiter(linePos);
        }
        catch (BadLocationException e) {
            return "";
        }
        if (delim == null) { return ""; }

        return delim;
    }

    public void setSelection(ITextSelection selection) {
        _internalSelection = _visualSelection = selection;
    }

    /**
     * Set a new selection.
     * 
     * @param start
     *            start postion of the selection
     * @param length
     *            length of the selection
     */
    public void setVisualSelection(int start, int length) {
        _internalSelection = _visualSelection = new TextSelection(_document,
                start, length);
        // this check is needed for the junit tests
        if (_textEditor != null) {
            _textEditor.getSelectionProvider().setSelection(_visualSelection);
        }
    }

    /**
     * Set a new internal selection.
     * 
     * @param start
     *            start postion of the selection
     * @param length
     *            length of the selection
     */
    public void setInternalSelection(int start, int length) {
        _internalSelection = new TextSelection(_document, start, length);
    }

    /**
     * @param document
     */
    public void setDocument(IDocument document) {
        _document = document;
    }

    // helper functions start here
    public void setCaretPosition(int pos) throws BadLocationException {
        if (_isInVisualMode) {
            refreshVisualSelection(pos);
        }
        else {
            setInternalSelection(pos, 0);
            if (_textViewer != null) {
                _textViewer.getTextWidget().setCaretOffset(pos);
                _textViewer.getTextWidget().showSelection();
            }
        }
    }

    public int getCaretPosition() {
        // needed for JUnit Testsuite
        //		if (_visualAnchor != -1 || _textEditor == null) {
        //			return getSelection().getOffset();
        //		}
        if (_isInVisualMode && (_visualAnchor == getSelection().getOffset())) {
            return _visualAnchor + getSelection().getLength();
        }
        else {
            return getSelection().getOffset();
        }
    }

    private int getCursorInLinePosition() throws BadLocationException {
        int cursorInLinePos = getCaretPosition() - getLine().getOffset();
        return cursorInLinePos;
    }

    private int getPreviousWhiteSpace(String text, int startIndex) {
        int index = text.lastIndexOf(" ", startIndex - 1);
        if (index == -1) {
            index = 0;
        }
        return index;
    }

    private int getNextWhiteSpace(String text, int startIndex) {
        int index = text.indexOf(" ", startIndex + 1);
        if (index == -1) {
            index = text.length() - 1;
        }
        return index;
    }

    public int getNextNonBlankCharacter(String text, int from) {
        for (int i = from; i < text.length(); i++) {
            if (!Character.isWhitespace(text.charAt(i))) { return i; }
        }
        return -1;
    }

    private final char _wordEndings[] = { '.', '(', ')', ' ', '\t', '\n', ':',
            ';', '?', '+', '=', '>', '<', '*', '{', '}', '"', '|', ',', '-',
            '_', '/', '@'};

    /**
     * @param ch
     *            character to check
     * @param withSpace
     *            if true space is included in wordEndings
     * @return true if the character is a wordEnding character
     */
    private boolean isWordEnding(char ch, boolean withSpace) {
        if (!withSpace && ch == ' ') return false;
        for (int i = 0; i < _wordEndings.length; ++i) {
            if (_wordEndings[i] == ch) { return true; }
        }
        return false;
    }

    /**
     * This methods returns an integer representing the number of characters
     * represented by a logical word based on the startIndex.
     * 
     * Currently the private char[] wordEndings, contains characters that are
     * used as wordends.
     * 
     * @param textOriginal
     *            The String to look through.
     * @param startIndex
     *            Where to start looking for a word ending.
     * @return the number of chars between the startIndex, and the logical
     *         ending of the current word.
     * @see wordEndings for a list of chars that determine the end of a word.
     * @author Bryce Alcock, Michael Bartl
     * @since 0.0.6-bla
     *  
     */
    private int getEndOfCurrentWord(String text, int startIndex) {
        int endIndex = text.length();
        int newStartIndex = startIndex;

        // starting at whitespace characters - search for beginning of word
        int i = startIndex + 1;
        while (i < endIndex && Character.isWhitespace(text.charAt(i))) {
            i++;
        }

        // set startIndex to first character of word
        newStartIndex = i;

        int index = 0;
        for (i = 0; i < _wordEndings.length; ++i) {
            index = text.indexOf(_wordEndings[i], newStartIndex);
            if (index > 0 && index < endIndex) {
                endIndex = index;
            }
        }

        // current character is already an end character
        // now jump until we find the last end character!! (vim behaviour)
        if (newStartIndex == endIndex) {
            for (i = startIndex + 2; i < text.length(); ++i) {
                // a wordending, but no space!
                if (isWordEnding(text.charAt(i), false)) {
                    endIndex = i;
                }
                else
                    break;
            }
            return endIndex;
        }

        return endIndex - 1;
    }

    public int getNextWord(String text, int startIndex) {
        int endIndex = text.length();
        int newStartIndex = startIndex;

        if (text == null) return 0;

        if (text.length() == 0) return 0;

        // current character is already an end character
        // now jump until we find the last end character!! (vim behaviour)
        if (isWordEnding(text.charAt(startIndex), false)) {
            for (int i = startIndex + 1; i < text.length(); ++i) {
                // a wordending, but no space!
                if (isWordEnding(text.charAt(i), false)) {
                    newStartIndex = i;
                }
                else
                    break;
            }
        }

        int index = 0;
        for (int i = 0; i < _wordEndings.length; ++i) {
            index = text.indexOf(_wordEndings[i], newStartIndex);
            if (index > 0 && index < endIndex) {
                endIndex = index;
            }
        }

        // we are on a wordend already so search for the next character
        if (newStartIndex == endIndex) {
            for (int i = newStartIndex + 1; i < text.length(); ++i) {
                char c = text.charAt(i);
                if (isWordEnding(c, false) || Character.isLetter(c)) { return i; }
            }
        }

        // next word starts after all spaces
        if (endIndex < text.length()) {
            if (text.charAt(endIndex) == ' ') {
                for (int i = endIndex; i < text.length(); ++i) {
                    char c = text.charAt(i);
                    if (isWordEnding(c, false) || Character.isLetter(c)) { return i; }
                }
            }
        }
        else
            --endIndex;

        return endIndex;
    }

    // TODO: Reimplement!!!
    public int getPreviousWord(String text, int startIndex) {
        for (int i = startIndex - 1; i >= 0; i--) {
            if (getNextWord(text, i) < startIndex - 1) { return i; }
        }
        return -1;
    }

    // movement functions
    public int cursorLeft(int counter) {
        try {
            int start = getLine().getOffset();
            int newPosition = getCaretPosition() - counter;
            if (newPosition < start) {
                newPosition = start;
            }
            setCaretPosition(newPosition);
            return newPosition;
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    public int cursorRight(int counter) {
        try {
            int end = getLine().getOffset() + getLine().getLength();
            int newPosition = getCaretPosition() + counter;
            if (newPosition > end) {
                newPosition = end;
            }
            setCaretPosition(newPosition);
            return newPosition;
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    /**
     * Set the cursor to a given line in the current buffer. Checks if the given
     * line is within the boundaries.
     * 
     * @param linenr
     *            The line to set the cursor toe
     * @param toLineBegin
     *            sets the cursor to the linebegin if true, otherwise it uses
     *            the current cursorPosition
     */
    public int cursorToLine(int linenr, boolean toLineBegin) {
        try {
            linenr = Math.max(0, linenr);
            linenr = Math.min(linenr, _document.getNumberOfLines() - 1);
            IRegion line = _document.getLineInformation(linenr);
            if (toLineBegin) {
                setCaretPosition(line.getOffset());
                return getCaretPosition();
            }
            else {
                int cursorInLinePos = getCursorInLinePosition();
                if (line.getLength() < cursorInLinePos) {
                    cursorInLinePos = line.getLength();
                }
                setCaretPosition(line.getOffset() + cursorInLinePos);
                return getCaretPosition();
            }
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    public int cursorWordBack(int cursor) {
        try {
            String text = _document.get();
            int newCursor = getCaretPosition();
            for (int i = 0; i < cursor; i++) {
                newCursor = getPreviousWord(text, newCursor);
            }

            setCaretPosition(newCursor + 1);
            return newCursor + 1;
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    public int cursorToLineEnd() {
        try {
            IRegion line = getLine();
            setCaretPosition(line.getOffset() + line.getLength());
            return getCaretPosition();
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    public void cursorToLineBegin() {
        cursorToLine(getLinePos(), true);
    }

    public int cursorWordEnd(int counter) {
        try {
            String text = _document.get();
            int newCursor = getCaretPosition();
            for (int i = 0; i < counter; i++) {
                newCursor = getEndOfCurrentWord(text, newCursor);
            }
            setCaretPosition(newCursor);
            return newCursor;
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    public int cursorNextWhiteSpace(int counter) {
        try {
            String text = _document.get();
            int newCursor = getCaretPosition();
            for (int i = 0; i < counter; i++) {
                newCursor = getNextWhiteSpace(text, newCursor + 1);
            }
            setCaretPosition(newCursor + 1);
            return newCursor + 1;
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    public int cursorToLineFirstNonBlank() {
        try {
            IRegion line = _document.getLineInformation(getLinePos());
            if (line.getLength() == 0) {
                cursorToLine(getLinePos(), true);
            }
            else {
                setCaretPosition(getNextNonBlankCharacter(_document.get(), line
                        .getOffset()));
            }
            return getCaretPosition();
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
    }

    public void replaceCharacters(String text) {
        try {
            // build text
            if (text.length() > 0) {
                for (int i = 1; i < getSelection().getLength(); i++) {
                    text += text.charAt(0);
                }
            }

            _document.replace(getSelection().getOffset(), getSelection()
                    .getLength(), text);
        }
        catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    public boolean selectCharacters(int nr) {
        int cursorPos = getCaretPosition();
        IRegion line = getLine();
        int offset = line.getOffset();
        int length = line.getLength();
        String delim = getLineDelimiter(getLinePos());

        // cursor on delimiter or last character
        if ((delim != null && cursorPos == offset + length)
                || (cursorPos == _document.getLength())) { return false; }

        // only delete until the end of line, regardless of nr
        if (nr + cursorPos > offset + length) {
            nr = offset + length - cursorPos;
        }
        setInternalSelection(cursorPos, nr);
        return true;
    }

    public boolean selectCharactersBackwards(int nr) {
        int cursorPos = getCaretPosition();
        IRegion line = getLine();
        int offset = line.getOffset();

        // cursor on first character or first character of line
        if (cursorPos == 0 || cursorPos == offset) { return false; }

        // only delete until the beginning of line, regardless of nr
        if (cursorPos - nr < offset) {
            nr = cursorPos - offset;
        }
        setInternalSelection(cursorPos - nr, nr);
        return true;
    }

    public int selectToCharacter(int counter, char ch, boolean inclusive) {
        int pos = getCaretPosition();
        IRegion line = getLine();
        int offset = line.getOffset();
        int endLine = line.getLength();
        String text;
        try {
            text = _document.get(pos, offset + endLine - pos);
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }

        // shoud move, so starting at 1.
        for (int i = 1; i < text.length(); i++) {
            if (text.charAt(i) == ch) {
                counter -= 1;
                if (counter == 0) {
                    if (inclusive) {
                        i += 1;
                    }
                    setInternalSelection(pos, i);
                    return pos + i;
                }
            }
        }
        return -1;
    }

    public int selectToCharacterBackward(int counter, char ch, boolean inclusive) {
        int pos = getCaretPosition();
        IRegion line = getLine();
        int offset = line.getOffset();
        String text;
        try {
            text = _document.get(offset, pos - offset);
        }
        catch (BadLocationException e) {
            e.printStackTrace();
            return -1;
        }
        for (int i = text.length() - 1; i >= 0; i--) {
            if (text.charAt(i) == ch) {
                counter -= 1;
                if (counter == 0) {
                    if (!inclusive) {
                        i += 1;
                    }
                    setInternalSelection(pos - (text.length() - i), text
                            .length()
                            - i);
                    return pos - (text.length() - i);
                }
            }
        }
        return -1;
    }

    /**
     * @param words
     */
    public void selectWordsBack(int words) {
        int start = getCaretPosition();
        int end = start;
        String text = _document.get();
        for (int i = 0; i < words; i++) {
            end = getPreviousWhiteSpace(text, end + 1);
        }
        int length = end - start + 1;

        setInternalSelection(start, length);
    }

    public void selectNextWhiteSpace(int counter) {
        int start = getCaretPosition();
        int end = start;
        String text = _document.get();
        for (int i = 0; i < counter; i++) {
            end = getNextWhiteSpace(text, end + 1);
        }
        int length = end - start + 1;

        setInternalSelection(start, length);
    }

    /**
     * remove the delimiter of the current line
     */
    public void removeDelimiter() {
        try {
            String delim = getLineDelimiter(getLinePos());
            if (delim != null) {
                _document.replace(
                        getLine().getOffset() + getLine().getLength(), delim
                                .length(), "");
            }
        }
        catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Return the position of the first visible character.
     * 
     * @param string
     * @return
     */
    public int getFirstVisibleCharacter(String string) {
        for (int i = 0; i < string.length(); i++) {
            if (!Character.isWhitespace(string.charAt(i))) { return i; }
        }
        return -1;
    }

    public void searchCharacter(int c, char ch, boolean inclusive) {
        _lastSearchCh = ch;
        _doesSearchChBackward = false;
        _isSearchChInclusive = inclusive;
        cursorToCharacter(c, ch, inclusive);
    }

    public void searchCharacterBackward(int c, char ch, boolean inclusive) {
        _lastSearchCh = ch;
        _doesSearchChBackward = true;
        _isSearchChInclusive = inclusive;
        cursorToCharacterBackward(c, ch, inclusive);
    }

    public void searchToCharacterNext(int c) {
        if (_lastSearchCh != 0) {
            if (_doesSearchChBackward) {
                cursorToCharacterBackward(c, _lastSearchCh,
                        _isSearchChInclusive);
            }
            else {
                cursorToCharacter(c, _lastSearchCh, _isSearchChInclusive);
            }
        }
    }

    public void searchToCharacterNextBackward(int c) {
        if (_lastSearchCh != 0) {
            if (_doesSearchChBackward) {
                cursorToCharacter(c, _lastSearchCh, _isSearchChInclusive);
            }
            else {
                cursorToCharacterBackward(c, _lastSearchCh,
                        _isSearchChInclusive);
            }
        }
    }

    public void cursorToCharacter(int c, char ch, boolean inclusive) {
        try {
            int pos = selectToCharacter(c, ch, inclusive);
            if (pos != -1) {
                setCaretPosition(pos - 1);
            }
        }
        catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    public void cursorToCharacterBackward(int c, char ch, boolean inclusive) {
        try {
            int pos = selectToCharacterBackward(c, ch, inclusive);
            if (pos != -1) {
                setCaretPosition(pos);
            }
        }
        catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    public ITextViewer getTextViewer() {
        return _textViewer;
    }

    public void pageUp(int counter) {
        int lines = _textViewer.getBottomIndex() - _textViewer.getTopIndex();
        cursorToLine(getLinePos() - lines * counter - 1, true);
    }

    public void pageDown(int counter) {
        int lines = _textViewer.getBottomIndex() - _textViewer.getTopIndex();
        cursorToLine(getLinePos() + lines * counter + 1, true);
    }

    public void scrollUp(int counter) {
        int lines = _textViewer.getBottomIndex() - _textViewer.getTopIndex();
        lines /= 2;
        cursorToLine(getLinePos() - lines * counter - 1, true);
    }

    public void scrollDown(int counter) {
        int lines = _textViewer.getBottomIndex() - _textViewer.getTopIndex();
        lines /= 2;
        cursorToLine(getLinePos() + lines * counter + 1, true);
    }

    /**
     * Tells the TextModificator that the visual mode was activated.
     */
    public void activateVisualMode() {
        _visualAnchor = getCaretPosition();
        _isInVisualMode = true;
    }

    /**
     * Tells to TextModificator that visual mode was deactivated.
     */
    public void deactivateVisualMode() {
        _isInVisualMode = false;
    }

    void refreshVisualSelection(int newPosition) {
        setVisualSelection(_visualAnchor < newPosition ? _visualAnchor
                : newPosition, Math.abs(newPosition - _visualAnchor));
    }

    public void resetSelection() {
        setVisualSelection(getCaretPosition(), 0);
    }

    /**
     * Delegates <code>TextSelection.getText()</code> on the wrappered
     * document.
     * 
     * @param offset
     *            the offset of the selected range
     * @param length
     *            the length of the selected range
     * @return text text
     */
    public String getText(int offset, int length) {
        return new TextSelection(_document, offset, length).getText();
    }

    public void setDeleteLastDelimiter(boolean deleteLastDelimiter) {
        _deleteLastDelimiter = deleteLastDelimiter;
    }

    public boolean getDeleteLastDelimiter() {
        return _deleteLastDelimiter;
    }

    /**
     * @return Text of current line
     */
    public String getLineText() {
        IRegion line = getLine();
        return getText(line.getOffset(), line.getLength());
    }
}