/*******************************************************************************
 * Copyright (c) 2003, Michael Bartl
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Created on 26.05.2003
 *******************************************************************************/

package viPlugin;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.IUndoManager;
import org.eclipse.swt.SWT;
import org.eclipse.ui.texteditor.ITextEditor;

import viPlugin.commands.BookMark;
import viPlugin.commands.ChangeCaseSelection;
import viPlugin.commands.Command;
import viPlugin.commands.DeleteSelection;
import viPlugin.commands.ExtendSelectionToOffset;
import viPlugin.commands.GroupCommand;
import viPlugin.commands.InsertStringAbove;
import viPlugin.commands.InsertStringBelow;
import viPlugin.commands.LineJoin;
import viPlugin.commands.MatchBrace;
import viPlugin.commands.PasteAbove;
import viPlugin.commands.PasteBelow;
import viPlugin.commands.ReplaceString;
import viPlugin.commands.ShiftSelection;
import viPlugin.commands.SwitchToInsertModeCommand;
import viPlugin.commands.YankSelection;
import viPlugin.commands.delete.DeleteCommandFactory;
import viPlugin.commands.motion.MotionCommandFactory;
import viPlugin.commands.search.ISearch;
import viPlugin.commands.search.RegExSearch;
import viPlugin.commands.search.SearchDelegate;
import viPlugin.commands.search.SimpleSearch;
import viPlugin.commands.select.SelectLines;
import viPlugin.commands.select.SelectToLineBegin;
import viPlugin.commands.select.SelectToLineBeginNonBlank;
import viPlugin.commands.select.SelectToLineEnd;
import viPlugin.commands.select.SelectionCommandFactory;
import viPlugin.preferences.ViPreferenceService;

/**
 * @author Michael Bartl
 */
public class CommandBuffer {

    private String _cmdString;
    private String _prevCmdString;
    private TextModificator _tm = TextModificator.getInstance();
    private ViLayer _viLayer;
    private ISearch _search;
    private SearchDelegate _searchDelegate;
    private IUndoManager _undoManager;
    private CommandHistory _extendedCommandHistory;
    private CommandHistory _searchCommandHistory;

    public CommandBuffer(ViLayer viLayer) {
        _viLayer = viLayer;
        _cmdString = new String();
        _extendedCommandHistory = new CommandHistory();
        // TODO: implement it
        _searchCommandHistory = new CommandHistory();

        // get the correct Search implementation depending on the JRE version
        if (RegExSearch.isSupported(true)) {
            _search = new RegExSearch();
        }
        else {
            _search = new SimpleSearch();
        }
    }
    
    public String getPreviousCommand() {
        return _prevCmdString;
    }

    public String getCommand() {
        if (_cmdString.equals("")) {
            return "Empty";
        }
        else {
            return _cmdString;
        }
    }

    public void append(char command) {
        _cmdString += command;
    }

    public void append(String command) {
        _cmdString += command;
    }

    public void backspace() {
        // if in search or colon command remove last character
        if (_cmdString.length() > 0
                && (_cmdString.startsWith("/") || _cmdString.startsWith(":"))) {
            _cmdString = _cmdString.substring(0, _cmdString.length() - 1);
        }
        // else move cursor left
        else {
            _cmdString = "h";
        }
    }

    public void clear() {
        _cmdString = "";
        _searchDelegate = null;
    }

    /**
     * Parse the command and do the appropriate action.
     */
    public void eval() {
        try {
            CommandParser commandParser = new CommandParser();
            if (!commandParser.parse(_cmdString)) {
                clear();
                return;
            }

            // get variables with short names
            boolean uc = commandParser._useCounter;
            int c = commandParser._counter;
            String o = commandParser._operation;
            String m = commandParser._modifier;         
            
            IDocument document = _tm.getTextEditor().getDocumentProvider()
                    .getDocument(_tm.getTextEditor().getEditorInput());
            _tm.setDocument(document);

            ITextSelection selection = (ITextSelection) _tm.getTextEditor()
                    .getSelectionProvider().getSelection();
            _tm.setSelection(selection);

            // no operation given yet
            if (o.length() == 0) { return; }

            if (_searchDelegate != null) {
                _searchDelegate.search(o);
                if (_searchDelegate.searchFinished()) {
                    clear();
                }
            }
            else if (o.equals("ZZ")) {
                o = "wq!";
                evalColonCommand(o);
            }
            else if (isColonCommandComplete(o)) {
                evalColonCommand(o.substring(1, o.length() - 1));
            }
            else if (o.startsWith(":") && o.endsWith("<CU>")) {
                clear();
                append(_extendedCommandHistory.up());
            }
            else if (o.startsWith(":") && o.endsWith("<CD>")) {
                clear();
                append(_extendedCommandHistory.down());
            }
            else if (evalMotionCommand(o, c, uc)) {
            }
            else if (preprocessInsertModeCommand(o, c)) {
                _viLayer.switchToInsertMode();
            }
            else if (o.equals("v")) {
                _viLayer.switchVisualMode();
            }
            else if (evalSelectionCommands(o, c)) {
                return;
            }
            else if (o.startsWith("y") || o.startsWith("Y")) {
                evalYankCommand(o, m, c);
            }
            else if (isDeleteCommand(o)) {
                evalDeleteCommand(o, m, c);
            }
            else if (o.equals(".")) {
                if (_prevCmdString.length() != 0) {
                    _cmdString = _prevCmdString;
                    eval();
                }
                else {
                    clear();
                }
            }
            else if (o.equals("/")) {
                _searchDelegate = new SearchDelegate(_search);
            }
            else if (o.equals("P")) {
                _prevCmdString = _cmdString;
                evalMultiple(new PasteAbove(), c);
                clear();
            }
            else if (o.equals("p")) {
                _prevCmdString = _cmdString;
                evalMultiple(new PasteBelow(), c);
                clear();
            }
            else if (o.equals("x")) {
                _prevCmdString = _cmdString;
                clear();
                if (_tm.selectCharacters(c)) {
                    YankSelection yankSelection = new YankSelection();
                    yankSelection.execute();
                    DeleteSelection deleteSelection = new DeleteSelection(true);
                    deleteSelection.execute();
                }
            }
            else if (o.equals("X")) {
                _prevCmdString = _cmdString;
                clear();
                if (_tm.selectCharactersBackwards(c)) {
                    YankSelection yankSelection = new YankSelection();
                    yankSelection.execute();
                    DeleteSelection deleteSelection = new DeleteSelection(true);
                    deleteSelection.execute();
                }
            }
            else if (o.startsWith("r") && o.length() > 1) {
                _prevCmdString = _cmdString;
                _tm.selectCharacters(c);
                _tm.replaceCharacters(m);
                clear();
            }
            else if (o.equals("~")) {
                _prevCmdString = _cmdString;
                _tm.selectCharacters(c);
                ChangeCaseSelection command = new ChangeCaseSelection();
                command.execute();
                clear();
            }
            else if (o.equals("<<")) {
                _prevCmdString = _cmdString;
                ShiftSelection shiftSelection = new ShiftSelection(c,
                        ShiftSelection.LEFT);
                shiftSelection.execute();
                clear();
            }
            else if (o.equals(">>")) {
                _prevCmdString = _cmdString;
                ShiftSelection shiftSelection = new ShiftSelection(c,
                        ShiftSelection.RIGHT);
                shiftSelection.execute();
                clear();
            }
            else if (o.equals("J")) {
                _prevCmdString = _cmdString;
                Command lineJoin = new LineJoin();
                lineJoin.execute();
                clear();
            }
            else if (o.startsWith("m") && m.length() == 1) {
                BookMark.add(m);
                clear();
            }
            else if (o.startsWith("'") && m.length() == 1) {
                BookMark.goTo(m);
                clear();
            }
            else if (o.equals("u")) {
                _undoManager.undo();
                clear();
            }
            else if (o.equals("<REDO>")) {
                _undoManager.redo();
                clear();
            }
            else if (o.charAt(o.length() - 1) == SWT.ESC) {
                clear();
            }
            else if (o.endsWith(String.valueOf((char) 13))) {
                clear();
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    private boolean isColonCommand(String o) {
        return o.startsWith(":");
    }

    private boolean isColonCommandComplete(String o) {
        return o.startsWith(":")
                && o.endsWith(ViVerifyKeyListener.ENTER_STRING);
    }

    private void evalMultiple(Command command, int counter) {
        for (int i = 0; i < counter; i++) {
            command.execute();
        }
    }

    private boolean isDeleteCommand(String o) {
        return o.startsWith("d") || o.startsWith("D") || o.startsWith("c")
                || o.startsWith("S") || o.startsWith("C");
    }

    /**
     * Helps simplify <code>eval()</code> method.
     * 
     * @param o
     *            operation
     * @param c
     *            counter
     * @return true if <code>o</code> isinsertMode operation
     */
    private boolean preprocessInsertModeCommand(String o, int c) {
        boolean isInsertCommand = true;
        if (o.equals("i")) { // do nothing
        }
        else if (o.equals("a")) {
            _tm.cursorRight(1);
        }
        else if (o.equals("I")) {
            _tm.cursorToLineFirstNonBlank();
        }
        else if (o.equals("A")) {
            _tm.cursorToLineEnd();
        }
        else if (o.equals("s")) {
            _tm.selectCharacters(c);
            DeleteSelection deleteSelection = new DeleteSelection(false);
            deleteSelection.execute();
        }
        else if (o.equals("o")) {
            InsertStringBelow insertStringBelow = new InsertStringBelow("");
            insertStringBelow.execute();
        }
        else if (o.equals("O")) {
            InsertStringAbove insertStringAbove = new InsertStringAbove("");
            insertStringAbove.execute();
        }
        else {
            isInsertCommand = false;
        }
        return isInsertCommand;
    }

    /**
     * Delete commands - all commands starting with d, c, D c commands are the
     * same like d commands they only switch to insert mode
     * 
     * @param o
     *            operation
     * @param m
     *            modifier
     * @param c
     *            counter
     */
    private void evalDeleteCommand(String o, String m, int c) {
        boolean checkLastDelimiter = o.startsWith("d");
        Command delete = DeleteCommandFactory.createCommand(o, m, c);
        if (delete != null) {
            delete.execute();
        }
        else { // TODO replace code in the following condition by
            // individual commands
            if (m.equals("d") || o.equals("cc") || o.equals("S")) {
                SelectLines selectLines = new SelectLines(c, SelectLines.DOWN,
                        checkLastDelimiter);
                selectLines.execute();
            }
            else if (m.equals("<CD>") || m.equals("j")) {
                SelectLines selectLines = new SelectLines(c + 1,
                        SelectLines.DOWN, checkLastDelimiter);
                selectLines.execute();
            }
            else if (m.equals("<CU>") || m.equals("k")) {
                SelectLines selectLines = new SelectLines(c, SelectLines.UP,
                        checkLastDelimiter);
                selectLines.execute();
            }
            else if (m.equals("<CR>") || m.equals("h")) {
                _tm.selectCharactersBackwards(c);
            }
            else if (m.equals("<CL>") || m.equals("l")) {
                _tm.selectCharacters(c);
            }
            else if (m.equals("$") || o.equals("D") || o.equals("C")) {
                SelectToLineEnd selectToLineEnd = new SelectToLineEnd();
                selectToLineEnd.execute();
            }
            else if (m.equals("0")) {
                SelectToLineBegin selectToLineBegin = new SelectToLineBegin();
                selectToLineBegin.execute();
            }
            else if (m.equals("^")) {
                SelectToLineBeginNonBlank slctToLineBeginNB = new SelectToLineBeginNonBlank();
                slctToLineBeginNB.execute();
            }
            else if (m.equals("b")) {
                _tm.selectWordsBack(c);
            }
            else if (m.equals("E")) {
                _tm.selectNextWhiteSpace(c);
            }
            else if (m.startsWith("t") && m.length() > 1) {
                _tm.selectToCharacter(c, m.charAt(1), false);
            }
            else if (m.startsWith("f") && m.length() > 1) {
                _tm.selectToCharacter(c, m.charAt(1), true);
            }
            else if (m.startsWith("T") && m.length() > 1) {
                _tm.selectToCharacterBackward(c, m.charAt(1), false);
            }
            else if (m.startsWith("F") && m.length() > 1) {
                _tm.selectToCharacterBackward(c, m.charAt(1), true);
            }
            else if (m.equals(";")) {
                _tm.searchToCharacterNext(c);
            }
            else if (m.equals(",")) {
                _tm.searchToCharacterNextBackward(c);
            }
            else if ((o.startsWith("c") || o.startsWith("d"))
                    && m.startsWith("/")) {
                List commands = new ArrayList();
                commands
                        .add(new ExtendSelectionToOffset(_tm.getCaretPosition()));
                commands.add(new YankSelection());
                commands.add(new DeleteSelection(checkLastDelimiter));
                if (o.startsWith("c")) {
                    commands.add(new SwitchToInsertModeCommand(_viLayer));
                }
                _searchDelegate = new SearchDelegate(_search, new GroupCommand(
                        commands));
                _cmdString = m;
                eval();
                return;
            }
            else {
                return;
            }
            YankSelection yankSelection = new YankSelection();
            yankSelection.execute();
            DeleteSelection deleteSelection = new DeleteSelection(
                    checkLastDelimiter);
            deleteSelection.execute();
        }

        if (o.startsWith("c") || o.startsWith("S") || o.startsWith("C")) {
            _viLayer.switchToInsertMode();
        }

        _prevCmdString = _cmdString;
        clear();
    }

    /**
     * Commands that are only available with a selection
     * 
     * @param o
     *            operation
     * @param c
     *            command
     */
    private boolean evalSelectionCommands(String o, int c) {
        if (_tm.getSelection().getLength() > 0) {
            if ((o.equals("d")) || o.equals("x")) {
                YankSelection yankSelection = new YankSelection();
                yankSelection.execute();
                DeleteSelection deleteSelection = new DeleteSelection(true);
                deleteSelection.execute();
            }
            else if (o.equals("c") || o.equals("s")) {
                DeleteSelection deleteSelection = new DeleteSelection(false);
                deleteSelection.execute();
                _viLayer.leaveVisualMode();
                _viLayer.switchToInsertMode();
            }
            else if (o.equals("y")) {
                YankSelection yankSelection = new YankSelection();
                yankSelection.execute();
                _tm.setVisualSelection(_tm.getCaretPosition(), 0);
            }
            else if (o.equals("~")) {
                ChangeCaseSelection command = new ChangeCaseSelection();
                command.execute();
            }
            else if (o.equals("<")) {
                ShiftSelection shiftSelection = new ShiftSelection(c,
                        ShiftSelection.LEFT);
                shiftSelection.execute();
            }
            else if (o.equals(">")) {
                ShiftSelection shiftSelection = new ShiftSelection(c,
                        ShiftSelection.RIGHT);
                shiftSelection.execute();
            }
            else {
                return false;
            }
            clear();
            // reapplying selection _cmdString isn't clear
            _prevCmdString = "";
            if (_viLayer.isInVisualMode()) {
                _viLayer.leaveVisualMode();
            }

            return true;
        }
        else {
            return false;
        }
    }

    /**
     * Yank commands
     * 
     * @param o
     *            operation
     * @param m
     *            modifier
     * @param c
     *            counter
     */
    private void evalYankCommand(String o, String m, int c) {
        Command select = SelectionCommandFactory.createCommand(o, m, c);
        if (select != null) {
            select.execute();
        }
        else { // TODO replace code in the folowing condition by
            // individual commands
            if (m.equals("y") || o.equals("Y")) {
                SelectLines selectLines = new SelectLines(c, SelectLines.DOWN,
                        false);
                selectLines.execute();
            }
            else if (m.equals("<CD>") || m.equals("j")) {
                SelectLines selectLines = new SelectLines(c + 1,
                        SelectLines.DOWN, false);
                selectLines.execute();
            }
            else if (m.equals("<CU>") || m.equals("k")) {
                SelectLines selectLines = new SelectLines(c, SelectLines.UP,
                        false);
                selectLines.execute();
            }
            else if (m.equals("<CR>") || m.equals("h")) {
                _tm.selectCharactersBackwards(c);
            }
            else if (m.equals("<CL>") || m.equals("l")) {
                _tm.selectCharacters(c);
            }
            else if (m.equals("$")) {
                SelectToLineEnd selectToLineEnd = new SelectToLineEnd();
                selectToLineEnd.execute();
            }
            else if (m.equals("0")) {
                SelectToLineBegin selectToLineBegin = new SelectToLineBegin();
                selectToLineBegin.execute();
                //_tm.selectToLineBegin();
            }
            else if (m.equals("^")) {
                SelectToLineBeginNonBlank slctToLineBeginNB = new SelectToLineBeginNonBlank();
                slctToLineBeginNB.execute();
            }
            else if (m.equals("b")) {
                _tm.selectWordsBack(c);
            }
            else if (m.equals("E")) {
                _tm.selectNextWhiteSpace(c);
            }
            else if (m.startsWith("t") && m.length() > 1) {
                _tm.searchCharacter(c, m.charAt(1), false);
            }
            else if (m.startsWith("f") && m.length() > 1) {
                _tm.searchCharacter(c, m.charAt(1), true);
            }
            else if (m.startsWith("T") && m.length() > 1) {
                _tm.searchCharacterBackward(c, m.charAt(1), false);
            }
            else if (m.startsWith("F") && m.length() > 1) {
                _tm.searchCharacterBackward(c, m.charAt(1), true);
            }
            else if (m.equals(";")) {
                _tm.searchToCharacterNext(c);
            }
            else if (m.equals(",")) {
                _tm.searchToCharacterNextBackward(c);
            }
            else {
                return;
            }
            YankSelection yankSelection = new YankSelection();
            yankSelection.execute();
        }

        clear();
    }

    /**
     * Evals motion command string.
     * 
     * @param o
     *            operation
     * @param c
     *            counter
     * @param uc
     *            useCounter
     * @return if <code>o</code> is motion command
     */
    private boolean evalMotionCommand(String o, int c, boolean uc) {
        try {
            Command command = MotionCommandFactory.createCommand(o, c, uc);
            if (command != null) {
                command.execute();
            }
            else if (o.equals("n")) {
                _search.searchNext();
            }
            else if (o.equals("N")) {
                _search.searchPrevious();
            }
            else if (o.equals("*")) {
                _search.searchCurrentWord();
            }
            else if (o.equals("#")) {
                _search.searchCurrentWordBackwards();
            }
            else if (o.equals("$")) {
                _tm.cursorToLineEnd();
            }
            else if (o.equals("%")) {
                MatchBrace mb = new MatchBrace();
                mb.execute();
            }
            else if (o.equals("^")) {
                _tm.cursorToLineFirstNonBlank();
            }
            else if (o.equals("e")) {
                _tm.cursorWordEnd(c);
            }
            else if (o.equals("E")) {
                _tm.cursorNextWhiteSpace(c);
            }
            else if (o.startsWith("f") && o.length() > 1) {
                _tm.searchCharacter(c, o.charAt(1), true);
            }
            else if (o.startsWith("T") && o.length() > 1) {
                _tm.searchCharacterBackward(c, o.charAt(1), false);
            }
            else if (o.startsWith("F") && o.length() > 1) {
                _tm.searchCharacterBackward(c, o.charAt(1), true);
            }
            else if (o.equals(";")) {
                _tm.searchToCharacterNext(c);
            }
            else if (o.equals(",")) {
                _tm.searchToCharacterNextBackward(c);
            }
            else if (o.equals("H")) {
                _tm.cursorToLine(_tm.getTextViewer().getTopIndex() + c - 1,
                        true);
            }
            else if (o.equals("L")) {
                _tm.cursorToLine(_tm.getTextViewer().getBottomIndex() - c + 1,
                        true);
            }
            else if (o.equals("M")) {
                int middle = _tm.getTextViewer().getTopIndex()
                        + _tm.getTextViewer().getBottomIndex();
                middle /= 2;
                _tm.cursorToLine(middle, true);
            }
            else if (o.equals("<PU>")) {
                _tm.pageUp(c);
            }
            else if (o.equals("<PD>")) {
                _tm.pageDown(c);
            }
            else if (o.equals("<SU>")) {
                _tm.scrollUp(c);
            }
            else if (o.equals("<SD>")) {
                _tm.scrollDown(c);
            }
            else {
                // no movement _cmdString found
                return false;
            } //			if (_viLayer.mode == ViMode.VISUAL_MODE) {
            //				_tm.refreshVisualSelection();
            //			}
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        clear();
        return true;
    }

    private boolean evalColonCommand(String command) {
        // save colon command
        _extendedCommandHistory.add(command);
        // get the line number (if any)
        int line;
        try {
            line = Integer.parseInt(command);
        }
        catch (NumberFormatException e1) {
            line = -1;
        }

        if (command.equals("q")) {
            try {
                if (!_tm.getTextEditor().isSaveOnCloseNeeded()) {
                    _tm.getTextEditor().close(false);
                }
                else {
                    MessageDialog.openInformation(_tm.getTextEditor().getSite()
                            .getWorkbenchWindow().getShell(),
                            "Can't close buffer",
                            "There are unsaved changes. To override use :q!");
                }
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        else if (command.equals("q!")) {
            try {
                _tm.getTextEditor().close(false);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        else if (command.equals("w")) {
            ITextEditor editor;
            try {
                editor = _tm.getTextEditor();
                IProgressMonitor pm = editor.getEditorSite().getActionBars()
                        .getStatusLineManager().getProgressMonitor();
                editor.doSave(pm);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        else if (command.equals("wq") || command.equals("x")) {
            try {
                _tm.getTextEditor().close(true);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        else if (command.equals("wq!")) {
            ITextEditor editor;
            try {
                editor = _tm.getTextEditor();
                IProgressMonitor pm = editor.getEditorSite().getActionBars()
                        .getStatusLineManager().getProgressMonitor();
                editor.doSave(pm);
                editor.close(true);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        }
        else if (line != -1) {
            _tm.cursorToLine(line - 1, true);
        }
        else if (command.startsWith("set")) {
            evalPreferenceSetting(command);
        }
        else if (command.startsWith("%s/")) {
            try {
                ReplaceString repstr = new ReplaceString(command);
                repstr.execute();
                _prevCmdString = ":" + command + String.valueOf((char) 13);
            }
            catch (Exception e) {
                MessageDialog
                        .openInformation(_tm.getTextEditor().getSite()
                                .getWorkbenchWindow().getShell(), "Error",
                                "Invalid _cmdString: usage: :%s/<search_for>/<replace_with>/g");
                return false;
            }
        }
        else {
            clear();
            return false;
        }
        clear();
        return true;
    }

    private void evalPreferenceSetting(String command) {
        StringTokenizer st = new StringTokenizer(command);
        st.nextToken();
        if (st.hasMoreTokens()) {
            String name = st.nextToken();
            String value = null;
            if (st.hasMoreTokens()) value = st.nextToken();
            ViPreferenceService.getInstance().eval(name, value);
        }
    }

    public void setUndoManager(IUndoManager undoManager) {
        _undoManager = undoManager;
    }

    public void beginCompoundChange() {
        _undoManager.beginCompoundChange();
    }

    public void endCompoundChange() {
        _undoManager.endCompoundChange();
    }
}