using System;
using System.Reflection;
using System.Text;

namespace PrivateNUnitCore
{
	/// <summary>
	/// TestCaseGenerator ̊Tv̐łB
	/// </summary>
	public class TestCaseGenerator
	{
		private string fqdnClassName;
		private MethodInfo[] methodsToTest;

		private const string CLASS_HEADER = "using System;\r\n"
											+ "using NUnit.Framework;\r\n"
											+ "using {originalNamespace};\r\n"
											+ "using System.Reflection;\r\n"
			                                + "namespace {namespace}\r\n"
											+ "{\r\n"
											+ "    [TestFixture]\r\n"
											+ "    public class {classname}\r\n"
											+ "    {\r\n";
											
		private const string CLASS_FOOTER = "    }\r\n"
											+ "}\r\n";
									

		public string FqdnClassName
		{
			get { return fqdnClassName; }
			set { fqdnClassName = value; }
		}

		public string OriginalNamespace
		{
			get
			{
				if(fqdnClassName == null || fqdnClassName.Length ==0)
				{
					return string.Empty;
				}
				return fqdnClassName.Substring(0,fqdnClassName.LastIndexOf("."));
			}
		}

		public string OriginalClassName
		{
			get
			{
				if(fqdnClassName == null || fqdnClassName.Length ==0)
				{
					return string.Empty;
				}
				return fqdnClassName.Substring(fqdnClassName.LastIndexOf(".") +1);
			}
		}

		public MethodInfo[] MethodsToTest
		{
			get { return methodsToTest; }
			set { methodsToTest = value; }
		}

		public TestCaseGenerator(string fqdnClassName, MethodInfo[] methodsToTest)
		{
			this.fqdnClassName = fqdnClassName;
			this.methodsToTest = methodsToTest;
		}

		public string Generate()
		{
			StringBuilder sb = new StringBuilder();
			
			sb.Append(GenerateMethodHeader()).Append("\r\n");
			foreach(MethodInfo method in methodsToTest)
			{
				if(method.IsStatic)
				{
					if(method.IsPublic)
					{
						sb.Append(GeneratePublicStatic(method));
						sb.Append("\r\n");
					}
					else
					{
						sb.Append(GenerateNonPublicStatic(method));
						sb.Append("\r\n");
					}
				}
				else
				{
					if(method.IsPublic)
					{
						sb.Append(GeneratePublicInstance(method));
						sb.Append("\r\n");
					}
					else
					{
						sb.Append(GenerateNonPublicInstance(method));
						sb.Append("\r\n");
					}
				}
			}
			sb.Append(GenerateMethodFooter()).Append("\r\n");
			return sb.ToString();
		}
		
		private string GenerateMethodHeader()
		{
			string testNamespace = "Test." + OriginalNamespace;
			string testClassName = OriginalClassName + "Test";
			
			return CLASS_HEADER.Replace("{originalNamespace}",OriginalNamespace).Replace("{namespace}",testNamespace).Replace("{classname}",testClassName);
		}

		private string GenerateMethodFooter()
		{
			return CLASS_FOOTER;
		}
		private string GeneratePublicStatic(MethodInfo method)
		{
			StringBuilder sb = new StringBuilder();
			sb.Append("        [Test]\r\n");
			sb.Append("        public void ").Append(method.Name).Append("Test()\r\n");
			sb.Append("        {\r\n");
			
			ParameterInfo[] args = method.GetParameters();
			
			foreach(ParameterInfo arg in args)
			{
				sb.Append("            ");
				sb.Append(arg.ParameterType.FullName).Append(" ");
				sb.Append(arg.Name).Append(" = null;\r\n");				
			}

			sb.Append("            ");
			if(method.ReturnType != typeof(void))
			{
				sb.Append(method.ReturnType.FullName).Append(" result = ");
			}

			sb.Append(OriginalClassName).Append(".").Append(method.Name).Append("(");
			bool needComma = false;
			foreach(ParameterInfo arg in args)
			{
				if(needComma)
				{
					sb.Append(",");
				}
				else
				{
					needComma = true;
				}
				sb.Append(arg.Name);
			}
			sb.Append(");\r\n");

			sb.Append("        }\r\n");
			return sb.ToString();
		}
		
		private string GenerateNonPublicStatic(MethodInfo method)
		{
			StringBuilder sb = new StringBuilder();
			sb.Append("        [Test]\r\n");
			sb.Append("        public void ").Append(method.Name).Append("Test()\r\n");
			sb.Append("        {\r\n");

			sb.Append("            MethodInfo method = typeof(").Append(OriginalClassName).Append(").GetMethod(\"").Append(method.Name).Append("\",BindingFlags.Instance | BindingFlags.Static | BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.DeclaredOnly);\r\n");
			
			ParameterInfo[] args = method.GetParameters();
			foreach(ParameterInfo arg in args)
			{
				sb.Append("            ");
				sb.Append(arg.ParameterType.FullName).Append(" ");
				sb.Append(arg.Name).Append(" = null;\r\n");
			}

			if(args !=null || args.Length > 0)
			{
				sb.Append("    object[] args = {");
				bool needComma = false;
				foreach(ParameterInfo arg in args)
				{
					if(needComma)
					{
						sb.Append(",");
					}
					else
					{
						needComma = true;
					}
					sb.Append(arg.Name);
				}
				sb.Append("};\r\n");
			}
			else
			{
				sb.Append("            object[] args = null\r\n");
			}
			
			sb.Append("            object result = method.Invoke(null,args);\r\n");

			sb.Append("        }\r\n");

			return sb.ToString();
		}

		private string GeneratePublicInstance(MethodInfo method)
		{
			StringBuilder sb = new StringBuilder();
			sb.Append("        [Test]\r\n");
			sb.Append("        public void ").Append(method.Name).Append("Test()\r\n");
			sb.Append("        {\r\n");
			sb.Append("            ").Append(OriginalClassName).Append(" objToTest = new ").Append(OriginalClassName).Append("();\r\n");
			
			ParameterInfo[] args = method.GetParameters();
			foreach(ParameterInfo arg in args)
			{
				sb.Append("            ");
				sb.Append(arg.ParameterType.FullName).Append(" ");
				sb.Append(arg.Name).Append(" = null;\r\n");
			}

			sb.Append("            ");
			if(method.ReturnType != typeof(void))
			{
				sb.Append(method.ReturnType.FullName).Append(" result = ");
			}

			sb.Append("objToTest.").Append(method.Name).Append("(");
			bool needComma = false;
			foreach(ParameterInfo arg in args)
			{
				if(needComma)
				{
					sb.Append(",");
				}
				else
				{
					needComma = true;
				}
				sb.Append(arg.Name);
			}
			sb.Append(");\r\n");

			sb.Append("        }\r\n");
			return sb.ToString();
		}
		
		private string GenerateNonPublicInstance(MethodInfo method)
		{

			StringBuilder sb = new StringBuilder();
			sb.Append("        [Test]\r\n");
			sb.Append("        public void ").Append(method.Name).Append("Test()\r\n");
			sb.Append("        {\r\n");

			sb.Append("            ").Append(OriginalClassName).Append(" objToTest = new ").Append(OriginalClassName).Append("();\r\n");
			sb.Append("            MethodInfo method = objToTest").Append(".GetType().GetMethod(\"").Append(method.Name).Append("\",BindingFlags.Instance | BindingFlags.Static | BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.DeclaredOnly);\r\n");
			
			ParameterInfo[] args = method.GetParameters();
			foreach(ParameterInfo arg in args)
			{
				sb.Append("            ");
				sb.Append(arg.ParameterType.FullName).Append(" ");
				sb.Append(arg.Name).Append(" = null;\r\n");
			}

			if(args !=null || args.Length > 0)
			{
				sb.Append("            object[] args = {");
				bool needComma = false;
				foreach(ParameterInfo arg in args)
				{
					if(needComma)
					{
						sb.Append(",");
					}
					else
					{
						needComma = true;
					}
					sb.Append(arg.Name);
				}
				sb.Append("};\r\n");
			}
			else
			{
				sb.Append("            object[] args = null\r\n");
			}
			
			sb.Append("            object result = method.Invoke(objToTest,args);\r\n");

			sb.Append("        }\r\n");

			return sb.ToString();
		}

	}
}
