package jp.co.app.ynomoto.pmft;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;

import android.app.PendingIntent;
import android.appwidget.AppWidgetManager;
import android.appwidget.AppWidgetProvider;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.AsyncTask;
import android.widget.RemoteViews;
import android.widget.Toast;

import android.util.Log;

/*
 *  ToDo
 *  1. メニューアクティビティとプレファレンスアクティビティを作成する。
 *		- メニューアクティビティはランチャーから起動できる。
 *		- メニューアクティビティはプレファレンスアクティビティを設定項目より起動する。
 *		- メニューアクティビティは、コピーライト、設定項目を表示する。
 *		- プレファレンスアクティビティは、以下の項目がある。
 *			- CSV の取得先の変更 (ミラーから本家への変更)
 *			- フレームの変更
 *			- テキストカラーの変更
 *			- 顔文字の編集
 */


//2011/04/19 (火): onUpdate() の処理を powerMeterMain() へ分割。onUpdate() は powerMeterMain() の実行に専念する。
//2011/04/19 (火): onReceive() をオーバーライドし、jp.co.app.ynomoto.pmft.UPDATE によるネットワークエラーの対処を追加
//2011/04/19 (火): ネットワークエラーが発生した場合、特定のウィジェットに対してのみインテントによるデータ更新ができる。
//2011/04/20 (水): ネットワーク障害時に WebClient インスタンス生成中に、DNS エラーが発生すると ANR が発生する問題への対処として、AsyncTask のサブクラスとして、powerMeterMain() を PowerMeterMain クラスへ変更
//2011/04/22 (金): PowerMeterMain の doInBackground で UI の更新処理をしていたので、頻繁に UI が正常に更新されない問題があった。 onPostExecute で更新を行うように修正。

public class PowerMeter extends AppWidgetProvider {
	private static final String updateIntent = "jp.co.app.ynomoto.pmft.UPDATE";
	// awm と c は、PowerMeterMain クラスで参照したいので、static 宣言する。
	private static AppWidgetManager awm;
	private static Context context;
	private boolean networkError=true;
	
	@Override
	public void onUpdate(Context context, AppWidgetManager awm, int[] appWidgetIds) {
		PowerMeter.awm = awm; 
		PowerMeter.context = context;	
		// ウィジェットがホームに配置されている分実行するつもりだが、appWidgetIds[0] にしか値は無い...。
		// つまり、このループは一回しか実行されない。
		for (int i = 0; i < appWidgetIds.length; i++) {
			(new PowerMeterMain(appWidgetIds[i], "android.appwidget.action.APPWIDGET_UPDATE")).execute();
		}
	}
	
	@Override
	public void onReceive(Context c, Intent intent){
		// onUpdate() を実行するために、スーパークラスの onReceive を呼ぶ。
		super.onReceive(c, intent);
		// ネットワークエラー復帰後を想定して、updateIntent によるデータ更新を行う。
		if(intent.getAction().equals(updateIntent)){
			(new PowerMeterMain(intent.getIntExtra("appWidgetId", -1), intent.getAction())).execute();
		}
	}
	
	public class PowerMeterMain extends AsyncTask<Void, Void, RemoteViews>{
		private int appWidgetId;
		private String intentAction;
		
		public PowerMeterMain(int appWidgetId, String intentAction){
			this.appWidgetId = appWidgetId;
			this.intentAction = intentAction;
		}

		@Override
		protected RemoteViews doInBackground(Void... params) {
			Log.v("jp.co.app.ynomoto.pmft.PowerMeter", "update appWidgetIds: " + appWidgetId);
			// リモートビューに main.xml をセットする。
			RemoteViews rv = new RemoteViews(context.getPackageName(), R.layout.main);

			try {
				// csv ファイルをダウンロードする。
				String csvUrl = "http://powermeter.sourceforge.jp/juyo-j.csv";
				// String csvUrl =
				// "http://www.tepco.co.jp/forecast/html/images/juyo-j.csv";
				Log.v("jp.co.app.ynomoto.pmft.PowerMeter", "WebClient("
						+ csvUrl + ")");
				WebClient wc = new WebClient(csvUrl);

				// 取得した csv の解析
				TempcoDataParser tdp = new TempcoDataParser(wc.getCsv());

				// ブラウザを呼び出し、Google チャート API にグラフのデータを渡す。
				int[] maximumkW = new int[24];
				for (int j = 0; j < tdp.getMaximumTime(); j++) {
					maximumkW[j] = 0;
				}
				maximumkW[tdp.getMaximumTime()] = tdp.getMaximumKw();
				GChart gc = new GChart(maximumkW, tdp.getTodayKw(), tdp
						.getYesterdayKw());
				PendingIntent pendingIntent = PendingIntent.getActivity(context, 0,
						new Intent(Intent.ACTION_VIEW, gc.getChart()), 0);
				rv.setOnClickPendingIntent(R.id.powerMeter, pendingIntent);

				// ピーク時供給力供給量に対する電力の消費率を表示
				int usage = new PowerUsageCalc().calculate(tdp.getCurrentKw(),
						tdp.getMaximumKw());
				Log.v("jp.co.app.ynomoto.pmft.PowerMeter", "usage=" + usage);
				rv.setTextViewText(R.id.textView1, Integer.toString(usage)
						+ "%");

				// 電力消費率による顔文字の変化
				if (usage >= 90) {
					rv.setTextViewText(R.id.textView2, "(´；ω；`)");
				} else if (usage >= 70) {
					rv.setTextViewText(R.id.textView2, "(´・ω・`)");
				} else {
					rv.setTextViewText(R.id.textView2, "(`・ω・´)");
				}
				
				networkError=false;
								
			} catch (MalformedURLException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();

				rv.setTextViewText(R.id.textView1, "Err");
				// ネットワークエラー解決後にデータを更新できるようにインテントの設定
				Intent retryIntent = new Intent(updateIntent);
				// インテントに appWidgetId を渡し、更新するべきウィジェットを区別する。
				retryIntent.putExtra("appWidgetId", appWidgetId);
				// getBroadcast() の第2引数には appWidgetId が必要らしいが??
				PendingIntent pendingIntent = PendingIntent.getBroadcast(context, appWidgetId, retryIntent, 0);
				rv.setOnClickPendingIntent(R.id.powerMeter, pendingIntent);
				
				networkError=true;		
			}
			return rv;		
		}
		
		 @Override  
		 protected void onPostExecute(RemoteViews rv) {
			// AppWidgetManager に appWidgetId のウィジェットを更新することを伝える。
			awm.updateAppWidget(appWidgetId, rv);
			if(intentAction.equals(updateIntent)){
				if(networkError){
					Toast.makeText(context, "電力情報の取得に失敗しました。", Toast.LENGTH_SHORT).show();
				} else {
					Toast.makeText(context, "電力情報を更新しました。", Toast.LENGTH_LONG).show();
				}
			}
		 }  
	}
}
/*
 * 取得した csv は、String[]に格納して getPage() で情報の取得ができる。
 */

// 2011/04/08 (金): webPage を ArrayList にして、csv の長さが変更されても例外が発生しないように修正
// 2011/04/08 (金): User-Agent として pmft/1.0 と名乗るように修正
// 2011/04/14 (木): HTTP Response Code の取得を getHeaderField(0) から getResponseCode() に変更

class WebClient {
	private String[] csv;
	private ArrayList<String> webPage = new ArrayList<String>();

	WebClient(String url) throws MalformedURLException, IOException {
		String host[] = url.split("/");
		
		HttpURLConnection urlconn = (HttpURLConnection)(new URL(url).openConnection());
		urlconn.setRequestMethod("GET");
		urlconn.setInstanceFollowRedirects(false);
		urlconn.setRequestProperty("Host", host[2]);
		urlconn.setRequestProperty("Accept", "*/*");
		urlconn.setRequestProperty("Referer", "http://" + host[2] + "/");
		urlconn.setRequestProperty("Accept-Language", "ja");
		urlconn.setRequestProperty("User-Agent", "pmft/1.0");
		urlconn.setRequestProperty("Connection:", "close");
		urlconn.connect();

		Log.d("PowerMeter", "HTTP Response Code:" + urlconn.getResponseCode());
		
		if (urlconn.getResponseCode() == HttpURLConnection.HTTP_OK) {
			BufferedReader reader = new BufferedReader(new InputStreamReader(
					urlconn.getInputStream(), "Shift_JIS"));
			while (true) {
				webPage.add(reader.readLine());
				if(webPage.get(webPage.size()-1) == null) break;
			}
			csv = new String[webPage.size()];
			for (int i = 0; i < webPage.size(); i++) {
				csv[i] = webPage.get(i);
			}
			reader.close();
		}

		urlconn.disconnect();
	}

	String[] getCsv() {
		return csv;
	}
}

/*
 * juyo-j.csv を解析して、各フィールドにデータを格納する。この csv のフォーマット は度々変更される可能性があるので注意すること。
 */
// 2011/3/28 時点での csv のフォーマット
// -----------------------------------------------------------------
// 2011/3/28 7:05 UPDATE
// ピーク時供給力(万kW),時台,供給力情報更新日,供給力情報更新時刻
// 3850,18:00,3/27,17:30
//
// DATE,TIME,当日実績(万kW),前日実績(万kW)
// 2011/3/28,0:00,2710,2904
// 2011/3/28,1:00,2576,2759
// 2011/3/28,2:00,2506,2661
// 2011/3/28,3:00,2457,2578
// 2011/3/28,4:00,2454,2520
// 2011/3/28,5:00,2589,2529
// 2011/3/28,6:00,2863,2598
// 2011/3/28,7:00,0,2760
// 2011/3/28,8:00,0,2908
// 2011/3/28,9:00,0,2946
// 2011/3/28,10:00,0,2913
// 2011/3/28,11:00,0,2857
// 2011/3/28,12:00,0,2826
// 2011/3/28,13:00,0,2786
// 2011/3/28,14:00,0,2772
// 2011/3/28,15:00,0,2786
// 2011/3/28,16:00,0,2860
// 2011/3/28,17:00,0,3073
// 2011/3/28,18:00,0,3371
// 2011/3/28,19:00,0,3392
// 2011/3/28,20:00,0,3345
// 2011/3/28,21:00,0,3251
// 2011/3/28,22:00,0,3103
// 2011/3/28,23:00,0,2939
// -----------------------------------------------------------------

//2011/4/8 時点での csv のフォーマット
//-----------------------------------------------------------------
//2011/4/8 5:30 UPDATE
//ピーク時供給力(万kW),時台,供給力情報更新日,供給力情報更新時刻
//4050,18:00,4/8,1:05
//
//予想最大電力(万kW),時間帯,予想最大電力情報更新日,予想最大電力情報更新時刻
//3350,18:00～19:00,4/8,1:05
//
//DATE,TIME,当日実績(万kW),前日実績(万kW)
//2011/4/8,0:00,2677,2659
//2011/4/8,1:00,2546,2551
//2011/4/8,2:00,2484,2492
//2011/4/8,3:00,2438,2442
//2011/4/8,4:00,2424,2438
//2011/4/8,5:00,0,2499
//2011/4/8,6:00,0,2713
//2011/4/8,7:00,0,2909
//2011/4/8,8:00,0,3153
//2011/4/8,9:00,0,3301
//2011/4/8,10:00,0,3300
//2011/4/8,11:00,0,3287
//2011/4/8,12:00,0,3051
//2011/4/8,13:00,0,3183
//2011/4/8,14:00,0,3184
//2011/4/8,15:00,0,3124
//2011/4/8,16:00,0,3133
//2011/4/8,17:00,0,3114
//2011/4/8,18:00,0,3295
//2011/4/8,19:00,0,3281
//2011/4/8,20:00,0,3177
//2011/4/8,21:00,0,3071
//2011/4/8,22:00,0,3018
//2011/4/8,23:00,0,2860
//-----------------------------------------------------------------

// 2011/04/08 (金): ピーク時供給力の時間にバグがったので、maximumTime を追加
// 2011/04/08 (金): tmp 変数を削除して、コードを最適化
// 2011/04/08 (金): csv フォーマットの変更に柔軟に対応できるように timeIndex を追加

class TempcoDataParser {
	private int maximumKw; // ピーク時供給力
	private int maximumTime; //ピーク時供給力時間
	private int currentKw; // 当日実績の最新
	private int[] yesterdayKw = new int[24]; // 前日実績の0時～23時
	private int[] todayKw = new int[24]; // 当日実績の0時～23時
	private int timeIndex = 8; // 0 時が CSV の9行目にあるので配列の index は 8

	TempcoDataParser(String[] csv) {
		// ピーク時供給力(万kW)を求める
		maximumKw = Integer.parseInt(csv[2].split(",")[0]);
		
		// ピーク時供給力時間を求める
		maximumTime = Integer.parseInt(csv[2].split(",")[1].split(":")[0]);
		
		// CSV の更新時間を求める
		String updateTime = csv[0].split(" ")[1].split(":")[0]; 
				
		// currentKw を求める
		for(int i = timeIndex; csv[i] != null; i++){
			if(csv[i].matches(".*" + updateTime + ":.*")){
				if(i == timeIndex){
					// updateTime が0時ならば、前日の23時の情報を取得する。CSV
					// の現在のフォーマットでは、0時の場合は、前日の情報全てを取得で
					// きる。
					currentKw = Integer.parseInt(csv[timeIndex+23].split(",")[2]);
					break;
				} else {
					// updateTime が 0時でないならば、その前の時間の消費電力量を取得。
					currentKw = Integer.parseInt(csv[i-1].split(",")[2]);
					break;
				}
			}
		}
		
		// yesterdayKw と todayKw を求める。
		for(int i = timeIndex; csv[i] != null; i++){
			yesterdayKw[i-timeIndex] = Integer.parseInt(csv[i].split(",")[3]);
			todayKw[i-timeIndex] = Integer.parseInt(csv[i].split(",")[2]);
		}
	}

	int getMaximumKw() {
		return maximumKw;
	}
	
	int getMaximumTime(){
		return maximumTime;
	}

	int getCurrentKw() {
		return currentKw;
	}
	
	public int[] getYesterdayKw() {
		return yesterdayKw;
	}

	public int[] getTodayKw() {
		return todayKw;
	}

}

/*
 * TempcoDataParser から得られた結果に基づいてピーク時供給力供給量に対する消費率を計算する。
 */
class PowerUsageCalc {
	public int calculate(int currentKw, int maximumKw){
		return (int)((float)currentKw / maximumKw * 100);
	}
}

/*
 * Google Chart API 用の URL を生成する。
 */
	
/*	URL=http://chart.apis.google.com/chart
	chart に渡すパラメータの説明
	chs=300x150 // グラフの幅(px)、グラフの高さ(px)
	chd=t:60,80,83,85,87,88|85,80,90,80,70,90 //本日のデータ, 前日のデータ
	cht=lc // グラフの種類は折れ線グラフ
	chco=0000ff,00ff00// 本日のデータは青、前日のデータは緑
	chtt=TEPCO+Power+Usage+%28million%20kW%29 //グラフタイトル
	chdl=Today|Yesterday //本日のデータ、前日のデータ、
	chxt=x,y // x 軸下方(index0)と y軸左辺(index1)にラベルを表示する
	chxl=0:|0|12|23|1:|0|2000|4000 // index0, index1
 	chg=25,-1// グリッドを表示

	chd パラメータは 0 ～ 100 までの範囲でなければならない。従って、最大値を100%
	として、それに対する比率をデータとして用意する必要がある。
 */

class GChart {
	private String URL = "http://chart.apis.google.com/chart";
	private String chs,chd, cht, chco, chtt, chdl, chxt, chxl, chg;
	
	// Google チャートへ渡すパラメータの作成
	GChart(int[] maximumKw, int[] todayKw, int[] yesterdayKw){
		int graphMax = 6000;
		chs = "800x300";
		chd = "t:"+ toPercentage(maximumKw, graphMax) + "|" + toPercentage(todayKw, graphMax) + "|" + toPercentage(yesterdayKw, graphMax);
		cht = "bvo";
		chco = "11210D,3D7930,A2C180";
		chtt = "「本日の TEPCO 管轄内電力の使用状況グラフ(単位:万Kw)」";
		chdl = "ピーク時供給力|本日実績|前日実績";
		chxt = "x,y"; 
		chxl = "0:|0|1|2|3|4|5|6|7|8|9|10|11|12|13|14|15|16|17|18|19|20|21|22|23|1:|0|" + (graphMax / 2) + "|" + graphMax;
		chg = "25,-1";
	}
	
	private String toPercentage(int[] kWs, int graphMax){
		String percentages = String.valueOf((int)((float)kWs[0] / graphMax * 100));
		for(int i = 1; i < kWs.length; i++){
			percentages += "," + (int)((float)kWs[i] / graphMax * 100);
		}
		return percentages;
	}
	
	public Uri getChart() {
		String s = URL 
					+ "?" + "chs=" + chs 
					+ "&" + "chd=" + chd 
					+ "&" + "cht=" + cht 
					+ "&" + "chco=" + chco 
					+ "&" + "chtt=" + chtt 
					+ "&" + "chdl=" + chdl 
					+ "&" + "chxt=" + chxt
					+ "&" + "chxl=" + chxl
					+ "&" + "chg=" + chg;
		Log.v("jp.co.app.ynomoto.pmft.PowerMeter.GChart.getChart()", s);
		return Uri.parse(s);
	}	
}