/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.util;

import java.util.*;
import java.util.concurrent.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.postgresforest.constant.ConstStr;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.NonForestUrlException;
import org.postgresforest.exception.ForestInvalidUrlException;

import net.jcip.annotations.*;

/**
 * Forest用URLの各要素（IP:PORT、DBNAME、OPTION）を格納するImmutableクラス。
 * クラスメソッドのparseUrlを使用してインスタンスを作成する。
 * 接続先が等しいかどうかは、本クラスのインスタンス同士を比較（equals()）
 * することで判断できる。
 */
@Immutable
public final class ForestUrl {
    private final String ipports_dbname;
    
    private final String url;
    
    private final List<PgUrl> mngDbs = new CopyOnWriteArrayList<PgUrl>();
    /**
     * 管理情報DBへのPgUrlオブジェクトの、serveridをインデックスとした
     * 並行アクセス可能・変更不可能なリストを返す
     */
    public List<PgUrl> getMngDbUrls() {
        return Collections.unmodifiableList(mngDbs);
    }
    
    /** このForestUrlが保持するIP:PORT,IP:PORT/DBの文字列表現 */
    @Override public String toString() {
        return ipports_dbname;
    }
    
    /**
     * このインスタンス同士の論理的な同値性を比較をする。<br>
     * "IP1:PORT1,IP2:PORT2/DBNAME" の文字列が等しいことを、
     * ForestUrlが等しいと定義する（設計書参照）
     */
    @Override public boolean equals(Object obj) {
        if (obj instanceof ForestUrl) {
            return ipports_dbname.equals(((ForestUrl) obj).ipports_dbname);
        } else {
            return false;
        }
    }
    
    /** 
     * ハッシュコードを返却する<br>
     * equlasで等しいと判断されるインスタンス同士では、必ず同じハッシュ
     * コードが返る。<br>
     * （EffectiveJava等、Javaの情報参照）equalsをオーバーライドしている
     * ため、hashCodeのオーバーライドは必須
     */
    @Override public int hashCode() {
        return ipports_dbname.hashCode();
    }
    
    /** パース用パターン （IP/PORT/DB名/オプション分割用）*/
    private static final Pattern patternForestUrl = Pattern.compile("^(.+),(.+)/([^\\?]+)(\\?(.+))?$");
    /** パース用パターン （DB名確認用。先頭はアルファベット、２文字目以降アルファベット＋数字＋＿＋＄）*/
    private static final Pattern patternDb = Pattern.compile("[A-Za-z][A-Za-z0-9\\$_]*");
    
    /**
     * 接続文字列のパースを行い、ForestUrlオブジェクトと、接続文字列の後ろ側に
     * オプションで付いているkey=valueのペアをPropertiesとして返却する
     * @param url このJDBCドライバの接続文字列（外部仕様定義書参照）
     * @return 生成したForestUrlインスタンスと、オプションのペア
     * @throws NonForestUrlException 接続文字列がForestのものではなかった場合
     * @throws ForestInvalidUrlException 与えられた接続文字列が適切でなかった場合
     */
    public static Pair2<ForestUrl,Map<String, String>> parseUrl(final String url) throws ForestInvalidUrlException, NonForestUrlException {
        if (!url.startsWith(ConstStr.FOREST_URL_PREFIX.toString())) {
            throw new NonForestUrlException(ErrorStr.INVALID_FORESTURL.toString() + " : " + url);
        }
        final String ipports_db_opt = url.substring(ConstStr.FOREST_URL_PREFIX.toString().length());
        // IP0PORT0,IP1PORT1,DBNAME,OPTIONへの分解
        final Matcher matcherForest = patternForestUrl.matcher(ipports_db_opt);
        if (matcherForest.matches() == false) {
            throw new ForestInvalidUrlException(ErrorStr.INVALID_FORESTURL.toString() + " : " + url);
        }
        final List<String> ipports = new ArrayList<String>(2);
        ipports.add(matcherForest.group(1));
        ipports.add(matcherForest.group(2));
        // IP:PORTを分割して検査する
        for (final String ipport : ipports) {
            final int colonIndex = ipport.lastIndexOf(":");
            if (colonIndex < 1 || ipport.length() - 1 <=  colonIndex) {
                throw new ForestInvalidUrlException(ErrorStr.INVALID_FORESTURL.toString() + " : " + url);
            }
            final String ip = ipport.substring(0, colonIndex);
            final String port = ipport.substring(colonIndex + 1);
            
            // IPの中には「,」と「/」が存在しないことを確認
            if (ip.indexOf(",") >= 0 || ip.indexOf("/") >= 0) {
                throw new ForestInvalidUrlException(ErrorStr.INVALID_IPADDR.toString() + " : " + ip);
            }
            
            // PORTは0～65535であることを確認
            try {
                int portInt = Integer.valueOf(port);
                if (portInt < 0 || 65535 < portInt) {
                    throw new NumberFormatException();
                }
            } catch (NumberFormatException e) {
                throw new ForestInvalidUrlException(ErrorStr.INVALID_PORT.toString() + " : " + port);
            }
        }
        
        // DBNameには「,」、「/」、「:」がどれも1個も含まれていないことを確認
        final String dbname = matcherForest.group(3);
        if (dbname.indexOf(",") != -1 || dbname.indexOf("/") != -1 || dbname.indexOf(":") != -1) {
            throw new ForestInvalidUrlException(ErrorStr.INVALID_DBNAME.toString() + " : " + dbname);
        }
        
        // OPTIONは存在していれば頭の？を除いた部分、存在していなければ空文字列とする
        final String options = (matcherForest.group(5) != null) ? matcherForest.group(5) : "";
        
        // dbnameのチェック
        final Matcher matcherDb = patternDb.matcher(dbname);
        if (matcherDb.matches() == false) {
            throw new ForestInvalidUrlException(ErrorStr.INVALID_DBNAME.toString() + " : " + dbname);
        }
        
        // IP,PORTの重複チェック（但し名前解決などは含まないため、完全な重複チェックではない）
        if (ipports.get(0).equals(ipports.get(1))) {
            throw new ForestInvalidUrlException(ErrorStr.SAME_ADDRESS.toString());
        }
        
        // ForestUrlを生成する
        final ForestUrl forestUrl = new ForestUrl(ipports, dbname, url);
        
        // optionを分解してPropertyを構成する
        final Map<String, String> propMap = new HashMap<String, String>();
        if (options.length() != 0) {
            for (final String opt : options.split("&")) {
                final String[] elem = opt.split("=");
                if (elem.length == 2 && propMap.containsKey(elem[0]) == false && elem[1].startsWith(" ") == false) {
                    propMap.put(elem[0], elem[1]);
                } else {
                    // xxx=yyyの形ではない場合、あるいは既にキーがプロパティに存在している場合は例外
                    throw new ForestInvalidUrlException(ErrorStr.INVALID_OPTION.toString() + " : " + opt);
                }
            }
        }
        
        // 生成したForestUrlとPropertiesを返却する
        return new Pair2<ForestUrl, Map<String, String>>(forestUrl, propMap);
    }
    
    /** コンストラクタ。このコンストラクタはparseUrlからのみ呼ばれる */
    private ForestUrl(final List<String> ipport, final String dbname, final String url) {
        this.ipports_dbname = ipport.get(0) + "," + ipport.get(1) + "/" + dbname;
        this.url = url;
        // 管理データベースへ接続するためのURLオブジェクトを作成
        // 管理データベースに接続する際のUSER・PASSWORDは、管理DB名と同じ（外部設計書参照）
        final String mngdbname = dbname + ConstStr.MNGDB_SUFFIX.toString();
        mngDbs.add(new PgUrl(ipport.get(0), mngdbname));
        mngDbs.add(new PgUrl(ipport.get(1), mngdbname));
    }
    
    /**
     * 接続時に指定されたURLを返却する。
     * @return 接続時に指定されたURL（パラメータも含む）
     */
    public String getUrl() {
        return this.url;
    }
}
