/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.mng;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicIntegerArray;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.postgresforest.constant.ConstInt;
import org.postgresforest.constant.ConstStr;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.constant.LogStr;
import org.postgresforest.constant.UdbValidity;
import org.postgresforest.exception.ForestException;
import org.postgresforest.exception.ForestInitFailedException;
import org.postgresforest.exception.ForestInvalidMngInfoException;
import org.postgresforest.mng.MngInfo.EnumValidity;

import org.postgresforest.util.*;

import net.jcip.annotations.*;

/**
 * 各管理データベースから情報を読み出し、管理を行うためのクラス<br>
 * <br>
 * このクラスのインスタンスは、各ユーザデータベースの管理情報と１：１
 * に対応している。どのユーザデータベースに対する管理情報なのかは、
 * ForestUrlオブジェクト（本JDBCドライバへの接続文字列）で特定できる。<br>
 * <br>
 * 本クラスのクラスメソッドは、複数ある管理情報の全てを管理し、
 * インスタンスの生存期間などを決定するためのもの
 * <br>
 * このリソースマネージャの内部に持たせる各機能ごとのリソースは、
 * リソースマネージャインスタンスの解放時に同時に解放されなくてはならない。
 * そのため、「ResourceManager#destroy()」関数内に各機能クラスごとの
 * 後処理を登録する必要がある。
 */ @ThreadSafe
public final class ResourceManager {
    
    /**
     * staticフィールド。全リソースマネージャを共通管理するためのマップと、
     * リソースマネージャ取得のための関数で構成される
     */
    
    /** リソースマネージャのインスタンスを格納しておくマップ。接続文字列クラスと１：１に対応する */
    private static final ConcurrentHashMap<ForestUrl, ResourceManager> resourceManagerMap = new ConcurrentHashMap<ForestUrl, ResourceManager>();
    
    /** リソースマネージャの状態を表す列挙型 */
    private static enum EnumManagerState {
        /** 生成直後～初期化最中 */
        INITIAL, 
        /** 初期化完了（正常完了） */
        RUNNING,
        /** 初期化完了（異常完了・即座にマップから取り除くべき） */
        INVALID, 
        /** 被参照数が0で今後動作しない（即座にマップから取り除くべき） */
        DISPOSED; 
    }
    
    /**
     * リソースマネージャのインスタンスを取得する関数<br>
     * <b>注：インスタンスの取得に成功した場合、取得したインスタンスごとに必ず1回、
     * 対になるResourceManager#releaseResourceManager()を呼び出す必要がある</b><br>
     * <br>
     * この関数を呼び出すと、引数で与えられた接続文字列に該当するリソースマネージャが
     * 既に存在する場合には、そのインスタンスを返却する。存在しない場合には、リソース
     * マネージャを新規に作成・登録する。<br>
     * リソースマネージャが初期化中の場合、リソースマネージャの初期化が完了するまで
     * 全ての呼び出しスレッドは停止し、初期化が成功した段階で全ての呼び出しスレッドに
     * リソースマネージャのインスタンスが返る。<br>
     * <br>
     * リソースマネージャの初期化は、内部的にMngInfoManager#init()を呼び出す。
     * MngInfoManager#init()の呼び出しが成功すると、初期化処理は正常に完了したことになり、
     * 失敗した場合にはリソースマネージャの初期化に失敗し、全てのResourceManager#
     * getResourceManagerを呼び出しているスレッドに例外が返る。
     * 
     * @param targetUrl
     * @param user 管理DBにアクセスするユーザ
     * @param pass 管理DBにアクセスするために必要なパスワード
     * @return 取得したリソースマネージャ
     * @throws ForestException 管理情報の初期ロードに失敗した場合
     */
    public static ResourceManager getResourceManager(final ForestUrl targetUrl, final String user, final String pass) throws ForestException {
        while (true) {
            ResourceManager resourceManager = resourceManagerMap.get(targetUrl);
            if (resourceManager != null) {
                // インスタンスがマップ上にあるなら、そのステータスをチェック。
                synchronized (resourceManager.lockState) {
                    
                    // 状態がDISPOSEDの場合は、マップからインスタンスを取得してから
                    // このsynchronizedブロックに入るまでの間に、他の参照中スレッドが
                    // 参照カウントを減らしたことによってDISPOSEDになっている。
                    // この場合は再度インスタンスの取得を試みる
                    if (resourceManager.status == EnumManagerState.DISPOSED) {
                        continue;
                    }
                    // それ以外の場合は同じsynchronizedブロックの中で参照カウントを
                    // インクリメントしてしまうことで、ステータスのチェックと参照
                    // カウントの増加をアトミックに実行する
                    resourceManager.refCount++;
                }
                // 続いて初期化完了のラッチを待つ
                // 初期化が完了するまでラッチのawaitはスレッドを停止させる。
                // 既に初期化が完了してラッチが開放されていれば、ラッチのawaitは
                // 即座に返ってくる。
                try {
                    resourceManager.initCompleteLatch.await();
                } catch (InterruptedException e) {
                    // この関数の呼び出しはユーザスレッドなので、interruptされた場合
                    // ユーザ側にinterruptの処理を任せるため、interruptを再セットした
                    // うえで、取得失敗の例外を投げる
                    Thread.currentThread().interrupt();
                    ForestInitFailedException newException =
                        new ForestInitFailedException(ErrorStr.MNGINIT_INTERRUPT.toString());
                    newException.setMultipleCause(Collections.<Exception>singletonList(e));
                    throw newException;
                }
                // この時点でのステータスを確認し、RUNNINGでないならば初期化に失敗して
                // いるため、初期化スレッドが発行したExceptionを読み取り、投げる。
                synchronized (resourceManager.lockState) {
                    if (resourceManager.status != EnumManagerState.RUNNING) {
                        throw resourceManager.initException;
                    }
                }
                return resourceManager;
            } else {
                // インスタンスがマップ上に存在しない場合、新規に生成して登録する
                // ConcurrentHashMapのputIfAbsentはnullが返ればputできたことを示す
                resourceManager = new ResourceManager(targetUrl, user, pass);
                if (resourceManagerMap.putIfAbsent(targetUrl, resourceManager) == null) {
                    // 登録成功 = ここで新規に生成した管理情報を初期化し、外部から読めるようにする
                    synchronized (resourceManager.lockState) {
                        resourceManager.refCount++;
                    }
                    // 管理情報の初期化をし、ステータスをRUNNING/INVALIDに設定する
                    try {
                        resourceManager.initMngInfo();
                        synchronized(resourceManager.lockState) {
                            resourceManager.setStatus(EnumManagerState.RUNNING);
                        }
                        return resourceManager;
                    } catch (ForestInitFailedException e) {
                        // この例外を受けた場合は、他の初期化完了を待っている
                        // スレッドにも例外を伝える必要がある
                        synchronized(resourceManager.lockState) {
                            resourceManager.setStatus(EnumManagerState.INVALID);
                        }
                        resourceManager.initException = e;
                        throw e;
                    } catch (InterruptedException e) {
                        // interruptされた場合、このスレッドはアプリケーションスレッド
                        // なので、interruptをセットし直したうえで、結論としては初期化完了に
                        // 失敗したということで例外を生成し、他のスレッドに例外を伝える
                        synchronized(resourceManager.lockState) {
                            resourceManager.setStatus(EnumManagerState.INVALID);
                        }
                        ForestInitFailedException newException =
                            new ForestInitFailedException(ErrorStr.MNGINIT_INTERRUPT.toString());
                        newException.setMultipleCause(Collections.<Exception>singletonList(e));
                        resourceManager.initException = newException;
                        throw newException;
                    } finally {
                        // ラッチを開放して初期化完了を待つ他のスレッドに知らせる
                        resourceManager.initCompleteLatch.countDown();
                    }
                } else {
                    // 登録失敗 = 他のスレッドによってほぼ同時刻に別のマップを登録された
                    // リトライして再取得を試みる
                    continue;
                }
            }
        }
    }
    
    
    /**
     * 以降、インスタンスメソッド（各接続文字列ごとのリソースを管理するためのメソッド群）
     */
    
    /** statusと参照カウントを変更する場合に取得するロック このロック取得無しに変更してはならない */
    private final Object lockState = new Object();
    
    /** 
     * このインスタンスの状態を表す変数<br>
     * <b>値の変更をする際は必ず lockState のロックを握った上で、setStatus() 関数を用いる</b>
     */
    @GuardedBy("lockState") private volatile EnumManagerState status = EnumManagerState.INITIAL;
    
    /**
     * ステータスを変更するための関数<br>
     * 注：この関数は、lockStateのロックを取得して呼び出さなくてはならない<br>
     * <br>
     * INITIAL -> RUNNING -> DISPOSED<br>
     * または<br>
     * INITIAL -> INVALID<br>
     * 以外の状態遷移はあってはならない<br>
     * <br>
     * 状態変化のタイミングは以下の通り<br>
     * INITIAL -> RUNNING  : 初期化が完了した段階<br>
     *                       （initCompleteラッチが開放される）<br>
     * INITIAL -> INVALID  : 初期化が完了したが失敗している<br>
     *                       （ラッチは開放されるがその後getMngInfoはnullが返る）<br>
     * RUNNING -> DISPOSED : 参照カウントが0となった段階<br>
     *                       （リソースを解放する）<br>
     * @param newStatus 新規に反映させるステータス値
     * @throws IllegalStateException 正常でないステータス遷移の場合（プログラム的なミス）
     */
    @GuardedBy("lockState")
    private void setStatus(final EnumManagerState newStatus) throws IllegalStateException {
        switch (newStatus) {
            case INITIAL:
                // INITIALへの変更（インスタンス初期化時を除き）はいかなる状況でもNG
                break;
                
            case RUNNING:
                // INITIAL -> RUNNING はOK
                if (status == EnumManagerState.INITIAL) {
                    status = newStatus;
                    return;
                }
                break;
                
            case DISPOSED:
                // RUNNING -> DISPOSED はOK
                if (status == EnumManagerState.RUNNING) {
                    status = newStatus;
                    // マップからの登録解除とリソースの解放を行う
                    resourceManagerMap.remove(this.targetUrl, this);
                    destroy();
                    return;
                }
                break;
                
            case INVALID:
                // INITIAL -> INVALID はOK
                if (status == EnumManagerState.INITIAL) {
                    status = newStatus;
                    // マップからの登録解除とリソースの解放を行う
                    resourceManagerMap.remove(this.targetUrl, this);
                    destroy();
                    return;
                }
                break;
                
            default:
                break;
            
        }
        // 変更ができなかった場合はプログラム的なミス
        // 変更できる状況で呼び出すべき
        throw new IllegalStateException(ErrorStr.ILLEGAL_STATEMENT.toString() + " status change " + status.toString() + " -> " + newStatus.toString());
    }
    
    /** このインスタンスの参照カウント 必ずlockStateのロックを取得すること */
    @GuardedBy("lockState") private int refCount = 0;
    
    /** 管理情報の初期生成が完了すると開放されるラッチ（initの成功・失敗は問わず開放される）*/
    private final CountDownLatch initCompleteLatch = new CountDownLatch(1);
    
    /** 管理情報の初期生成に失敗した場合に、その例外がセットされる */
    private volatile ForestInitFailedException initException = null;
    
    /** このリソースマネージャに紐づく接続文字列（リソースマネージャ解放の際に使用する） */
    private final ForestUrl targetUrl;
    
    /** 管理情報にアクセスする際のユーザ名 */
    private final String mngdbUsername;
    
    /** 管理情報にアクセスする際のパスワード */
    private final String mngdbPassword;
    
    
    // ThreadPool
    
    /** 管理情報の定期読み込み実行のスケジューリングスレッド */
    private final ScheduledExecutorService mnginfoRefreshScheduler;
    
    /** 管理情報の読み込みやコネクション作成を行うスレッドプール */
    private final ExecutorService mngdbApiExecutor;
    
    /**
     * 各ユーザコネクションのAPIを実行するためのExecutor
     * （全てのEntrypointCommonResourceで共用する）
     */
    private final ExecutorService udbApiExecutor;
    
    /**
     * 各ユーザコネクションのAPIを縮退時に即時停止可能状態で実行するための中継用Executor
     */
    private final AtomicReferenceArray<ExecutorService> udbApiProxyExecutor;
    
    /**
     * ResourceManagerのコンストラクタ
     * @param targetUrl リソースマネージャを構築するための接続先URL
     * @param username 管理DBにアクセスする際に必要なユーザ名
     * @param password 管理DBにアクセスする際に必要なパスワード
     */
    private ResourceManager(final ForestUrl targetUrl, final String username, final String password) {
       this.targetUrl = targetUrl;
       this.mngdbUsername = username;
       this.mngdbPassword = password;
       this.burstRefreshError.set(0, 0);
       this.burstRefreshError.set(1, 0);
       
       // ユーザDB実行用スレッドプール。最低4スレッド常備、5以上のスレッドは30分のアイドル時間で消滅
       this.udbApiExecutor = Executors.newCachedThreadPool(new ForestThreadFactory(targetUrl, "JdbcApiExecThread"));
       {
           final ThreadPoolExecutor udbApiThreadPool = (ThreadPoolExecutor) udbApiExecutor;
           udbApiThreadPool.setCorePoolSize(4);
           udbApiThreadPool.setKeepAliveTime(1800, TimeUnit.SECONDS);
       }
       
       // ユーザDBProxy実行用スレッドプール。
       this.udbApiProxyExecutor = new AtomicReferenceArray<ExecutorService>(2);
       for (int i = 0; i < 2; i++) {
           createProxyApiExecutor(i);
       }
       
       // 管理DBアクセススケジューリングスレッドプール。最低1スレッド常備、2以上のスレッドは10分のアイドル時間で消滅
       this.mnginfoRefreshScheduler = Executors.newScheduledThreadPool(1, new ForestThreadFactory(targetUrl, "MngInfoRefreshScheduleThread"));
       {
           final ThreadPoolExecutor mnginfoRefreshThreadPool = (ThreadPoolExecutor) mnginfoRefreshScheduler;
           mnginfoRefreshThreadPool.setKeepAliveTime(600, TimeUnit.SECONDS);
       }
       
       // 管理DBアクセス用スレッドプール。最低2スレッド常備、2以上のスレッドは10分のアイドル時間で消滅
       this.mngdbApiExecutor = Executors.newCachedThreadPool(new ForestThreadFactory(targetUrl, "MngInfoQueryExecThread"));
       {
           final ThreadPoolExecutor mngdbApiThreadPool = (ThreadPoolExecutor) mngdbApiExecutor;
           mngdbApiThreadPool.setCorePoolSize(2);
           mngdbApiThreadPool.setKeepAliveTime(600, TimeUnit.SECONDS);
       }
    }
    
    /** 各ユーザコネクションAPIの即時停止可能用中継Executorを構築する */
    private void createProxyApiExecutor(final int serverId) {
        final ExecutorService newProxyExecutor = Executors.newCachedThreadPool(new ForestThreadFactory(targetUrl, "JdbcApiHookedThreadDB" + serverId));
        udbApiProxyExecutor.set(serverId, newProxyExecutor);
        final ThreadPoolExecutor newProxyThreadPool = (ThreadPoolExecutor) newProxyExecutor;
        // 各系最低２スレッド常備、３以上のスレッドは30分のアイドル時間で消滅
        newProxyThreadPool.setCorePoolSize(2);
        newProxyThreadPool.setKeepAliveTime(1800, TimeUnit.SECONDS);
    }
    
    /**
     * ResourceManagerのデストラクタ（のようなもの）<br>
     * このインスタンスを破棄する際（ステータスがINVALID/DISPOSEDになる場合）
     * に呼ばれ、各種リソースの後処理を行うための関数<br>
     * <b>注：本関数は１つのインスタンスに対して複数呼ばれる可能性もあり、
     * 各種リソースはそれを考慮した作りにする必要がある</b>
     */
    private void destroy() {
        mngdbApiExecutor.shutdownNow();
        mnginfoRefreshScheduler.shutdownNow();
        for (int i = 0; i < 2; i++) {
            if (mngInfoReadCons.get(i) != null) {
                try { mngInfoReadCons.get(i).close(); } catch (SQLException ignore) { }
            }
        }
        
        udbApiExecutor.shutdownNow();
        for (int i = 0; i < 2; i++) {
            final ExecutorService es = udbApiProxyExecutor.get(i);
            if (es != null) {
                es.shutdownNow();
            }
        }
    }
    
    /**
     * このリソースマネージャを参照しなくなったことを宣言するために呼び出す関数<br>
     * 本関数を呼ぶことでリソースマネージャインスタンスの参照カウントが減り、
     * 0となった時点でそのリソースマネージャのリソースを解放する。<br>
     * <br>
     * <b>注：getResourceManagerを呼び出した側が、必ず1度だけ本関数を呼ぶ必要がある。
     * 複数回呼び出した場合の動作は未定義。逆に１度も呼ばないと、リソースリークする</b>
     */
    public void releaseResourceManager() {
        synchronized (lockState) {
            refCount--;
            if (refCount < 1 && status == EnumManagerState.RUNNING) {
                setStatus(EnumManagerState.DISPOSED);
            }
        }
    }
    
    // ユーザのJDBC-API実行用メソッド
    
    /**
     * ユーザに公開するJDBC-APIの各タスクを実行するために使用する。内部的には
     * ExecutorService.invokeAll(Callable<T>, long, TimeUnit) を呼び出しているため、
     * 関数の詳細な仕様はExecutorService.invokeAllを参照。
     * （但し、タイムアウトの単位は「秒」で固定としている）<br>
     * <br>
     * ここで使用するExecutorServiceはリソースマネージャに管理されており、
     * リソースマネージャが不要になった（そのリソースマネージャにアクセスする
     * ForestConnectionが存在しなくなった）段階で、executorは自動的に開放される。
     * @param <T>
     * @param apiTasks
     * @param timeout 実行制限時間（秒）
     * @return
     * @throws InterruptedException
     */
    public <T> List<Future<T>> execJdbcApiTask(final List<Callable<T>> apiTasks, final int timeout)
    throws InterruptedException {
        return udbApiExecutor.invokeAll(apiTasks, timeout, TimeUnit.SECONDS);
    }
    
    /**
     * execJdbcApiTaskと同様、ユーザに公開するJDBC-APIの各タスクを実行するために使用する。
     * execJdbcApiTaskの場合は、DBからの応答（結果の返却・例外の返却）があるか、
     * 指定したタイムアウトが経過しない限りユーザアプリケーションに応答を返せないが、
     * この関数の場合は上記に加えて縮退が発生した際に即時その系のDBへのアクセスを
     * キャンセル（厳密にはキャンセルではなく、実行が継続しているスレッドを放置する）
     * して、ユーザアプリケーションに応答を返すことができる。
     * それ以外のAPI仕様はexecJdbcApiTaskと同じなのでそちらを参考のこと。
     */
    public <T> List<Future<T>> cancellableExecJdbcApiTask(final List<ForestTask<T>> apiTasks, final int timeout) {
        final List<Future<T>> returnFutureList = new ArrayList<Future<T>>(2);
        for (final ForestTask<T> task : apiTasks) {
            // rejectedExecutionExceptionが出た場合、すでに他のコネクションなどによって
            // 縮退を検出されてExecutorが閉じられている。そのため処理を行わなかったことと
            // して、futureはnullを返却する（返却されたあとに上位層で呼び出される
            // getResultAndClassifyはnullのfutureは何も実行しなかったとして処理をする仕様）
            final ExecutorService es = udbApiProxyExecutor.get(task.getServerId());
            Future<T> future = null; 
            if (es != null) {
                try {
                    future = es.submit(new CancelHookedForestTask<T>(task, timeout));
                } catch (RejectedExecutionException e) {
                    future = null;
                }
            } else {
                future = null;
            }
            returnFutureList.add(future);
        }
        return returnFutureList;
    }
    
    /** currentPrefServerIdをガードするためのロック */
    private final Object lockPrefServerId = new Object();
    /** 現在払いだされた優先実行系のサーバID。0または1のみを取る */
    private int currentPrefServerId = 1;
    /**
     * 優先実行サーバのIDを払いだす。払いだすIDは0または1。
     * 試験容易化のため、サーバIDは各リソースマネージャごとに
     * 0 -> 1 -> 0 -> 1 -> 0 ...
     * の順で払いだされることとする。
     * @return
     */
    @GuardedBy("lockPrefServerId")
    public int getNextPreferentialServerId() {
        synchronized (lockPrefServerId) {
            currentPrefServerId = (currentPrefServerId == 0) ? 1 : 0;
            return currentPrefServerId;
        }
    }
    
    /**
     * 引数で与えたタスクを実行するだけのタスククラス。このクラスをExecutorによって
     * 非同期実行すると、callメソッドの中でさらに別のスレッドプールに処理を委譲して
     * 二重の非同期実行をおこなう。
     * 
     * このクラスの使用用途は、ユーザDBへのJDBC-APIの実行において、クエリ実行の
     * タイムアウトを待たずに（あるいはクエリ実行のタイムアウトを無限大として）
     * 縮退になった瞬間に即座にユーザアプリケーションに制御を戻すために使用する。
     * 
     * このタスクは単に別のExecutorServiceに実行依頼をしてその結果を待っているだけ
     * なので、このタスクそのものを実行しているExecutorServiceのshutdownに即時
     * 応じることができるため、これを利用して縮退時に即時応答を返すことができる。
     * @param <T>
     */
    @Immutable private final class CancelHookedForestTask<T> implements Callable<T> {
        private final int timeout;
        private final ForestTask<T> forestTask;
        public CancelHookedForestTask(final ForestTask<T> forestTask, final int timeout) {
            this.timeout = timeout;
            this.forestTask = forestTask;
        }
        public T call() throws Exception {
            final Future<T> future = udbApiExecutor.submit(forestTask);
            try {
                return future.get(timeout, TimeUnit.SECONDS);
            } catch (ExecutionException e) {
                // タスクで例外が起きた場合にはExecutionExceptionでラップされて
                // いるため、中身を取り出して呼び出し元に投げる
                final Throwable causeThrowable = e.getCause();
                if (causeThrowable instanceof Error) {
                    throw (Error) causeThrowable;
                } else {
                    throw (Exception) causeThrowable;
                }
            }
        }
    }
    
    // 情報出力（ロギング）に関する処理
    
    /**
     * java.util.logging.Loggerのインスタンス。MngInfoManagerのinit()関数内で
     * ロガーを初期化し、この変数にインスタンスを割り当てる。<br>
     * この変数を直接使わず、必ず logWithException か logWithoutException を使ってロギングを行う
     */
    private volatile Logger logger;
    
    /**
     * 通常運転時のログを出力するためのログ関数
     * @param level メッセージを出力するログレベル
     * @param message
     */
    private void logging(final Level level, final String message) {
        final Logger logger = this.logger;
        if (logger == null || logger.getLevel().intValue() > level.intValue()) {
            return;
        }
        final StackTraceElement[] elements = new Exception().getStackTrace();
        final String className = elements[1].getClassName();
        final String methodName = elements[1].getMethodName();
        logger.logp(level, className, methodName, message);
    }
    
    /**
     * 例外を含めたログを出力するためのログ関数
     * @param messageLevel メッセージを出力するログレベル
     * @param message
     * @param exceptionLevel メッセージの後ろに例外（とそのスタックトレース）を含めるログレベル
     * @param throwable
     */
    private void logging(final Level messageLevel, final String message, final Level exceptionLevel, final Throwable throwable) {
        final Logger logger = this.logger;
        if (logger == null || logger.getLevel().intValue() > messageLevel.intValue()) {
            return;
        }
        final StackTraceElement[] elements = new Exception().getStackTrace();
        final String className = elements[1].getClassName();
        final String methodName = elements[1].getMethodName();
        if (logger.getLevel().intValue() > exceptionLevel.intValue()) {
            logger.logp(messageLevel, className, methodName, message);
        } else {
            logger.logp(messageLevel, className, methodName, message, throwable);
        }
    }
    
    /**
     * 現在の管理情報のglobal_configの値に従って、ロガーの設定を適切なものに
     * 変更する関数。この関数を呼んでいる間に他のスレッドがロガーを使用した場合、
     * ログロストする可能性がある<br>
     * この関数は複数スレッドから同時に呼ばれることを想定していない。またこの
     * 関数の中でロガーを初期化・再構築する中において、何らかの問題が発生した場合、
     * 特に外部に通知することなくログ出力ができなくなる。（エラーは握りつぶしている）
     */
    private void initLogger() {
        final MngInfo.LogConfig logConfig = mngInfo.get().getGlobalConfig().getLogConfig();
        // ForestUrl.toStringの名前でロガーを作成
        // JVM起動後初めて呼ばれた場合であればインスタンスが新規に生成され、
        // そうでなければ既にあるロガーが呼び出される
        final Logger logger = Logger.getLogger(targetUrl.toString());
        // 親ロガー（rootロガー）のハンドラにメッセージが送られるのを抑止する
        logger.setUseParentHandlers(false);
        // 既存で登録されているハンドラを全消去する（以前に同一URLでForestが動作していた場合を考慮）
        for (final Handler handler : logger.getHandlers()) {
            handler.close();
            logger.removeHandler(handler);
        }
        // Logger側でログレベルをセットする（レベルを参照する際はロガーのレベルを参照する）
        logger.setLevel(logConfig.getLevel());
        
        // Formatterのインスタンスを生成する
        
        // ユーザに指定されたフォーマッタの生成に成功した場合はそれを使う。
        // 失敗した場合は全てForestでデフォルトで用意しているフォーマッタを使う
        Formatter logFormatter;
        try {
            logFormatter = (Formatter) logConfig.getFormatter().newInstance();
        } catch (Exception e) {
            logFormatter = new ForestLogFormatter();
        }
        
        // Handlerを生成
        final List<Handler> handlerList = new ArrayList<Handler>(2);
        // 標準エラー出力のハンドラ
        if (logConfig.getConsoleEnable() == true) {
            handlerList.add(new ConsoleHandler());
        }
        // ファイル出力のハンドラ
        if (logConfig.getFileEnable() == true) {
            // 指定されたファイル名、既存ファイルへの追記モードでファイルハンドラを開く
            try {
                handlerList.add(new FileHandler(logConfig.getFileName(), logConfig.getFileSizeMb() * 1024 * 1024, logConfig.getFileRotateCount(), true));
            } catch (SecurityException ignore) {
            } catch (IOException ignore) {
            } catch (IllegalArgumentException ignore) {
                // 設定のサイズ上限値やカウント値が無効な場合
            }
        }
        
        // Logger・Handler・Formatterを結合させる。
        // また、HandlerのログレベルはALLとする（Loggerのほうでフィルタしているため）
        for (final Handler handler : handlerList) {
            logger.addHandler(handler);
            handler.setFormatter(logFormatter);
            handler.setLevel(Level.ALL);
        }
        
        this.logger = logger;
    }
    
    
    // 管理情報の公開・定期読み込み処理
    
    /** 現在の管理情報のスナップショット */
    private final AtomicReference<MngInfo> mngInfo = new AtomicReference<MngInfo>();
    
    /** 現在の管理情報のスナップショットを返す */
    public MngInfo getMngInfo() {
        return mngInfo.get();
    }
    
    /**
     * 管理情報読み込みに使用する両系へのコネクションを格納する配列
     * （管理情報アクセスのためのコネクションプール）<br>
     * この配列は複数スレッドが操作する可能性があるため、AtomicReferenceを使用する。
     */
    private final AtomicReferenceArray<Connection> mngInfoReadCons = new AtomicReferenceArray<Connection>(2);
    
    /** 両系の管理情報読み込みの連続エラー数 */
    private final AtomicIntegerArray burstRefreshError = new AtomicIntegerArray(2);
    
    /**
     * 管理DBにアクセスし、初回アクセス時の管理情報を作成する。
     * また、生成できた管理情報をもとにLoggerを生成する。<br>
     * <br>
     * この関数は以下の処理を行う<br>
     * 1. このリソースマネージャがアクセスすべきForestUrlを基に、コネクションを生成<br>
     * 2. （１個以上コネクションができたなら）各コネクションからMngInfoを生成<br>
     * 3. （MngInfoが２個生成できたなら）両方のMngInfoを合成<br>
     * 4. コネクションを作成したForestUrlに指定されたIP:PORTが、管理情報DBから<br>
     *    取得したIP:PORTと等しいかを検証<br>
     * 5. 全て通ったなら3（ないしは2）で生成したMngInfoを公開する<br>
     * 6. ForestUrl.toString()の名前でjava.util.logger.Loggerを生成、初期化する<br>
     * 
     * @throws ForestInitFailedException （インタラプト以外の理由で）初期化に失敗した場合
     * @throws InterruptedException このスレッドがinterruptされた場合
     */
    public void initMngInfo() throws ForestInitFailedException, InterruptedException {
        
        // コネクションプールを補充する
        final List<Exception> conCreateExceptionList = supplyConnection();
        // プールのコネクションを使ってデータベースからMngInfoを読み込む
        final Pair2<List<MngInfo>, List<Exception>> mngInfoAndExceptionList = readMngInfoFromDb();
        final List<MngInfo> mngInfoAllList = mngInfoAndExceptionList.getFirst();
        final List<Exception> readInfoExceptionList = mngInfoAndExceptionList.getSecond();
        
        // 有効なmngInfoのみを取得(nullのものを除外する)
        final List<MngInfo> mngInfoList = new ArrayList<MngInfo>();
        for (final MngInfo info : mngInfoAllList) {
            if (info != null) {
                mngInfoList.add(info);
            }
        }
        // コネクションの補充とDBからの読み込みの両方で発生した例外を集約する
        final List<Exception> exceptionList = new ArrayList<Exception>(conCreateExceptionList);
        for (int i = 0; i < readInfoExceptionList.size(); i++) {
            final Exception readInfoException = readInfoExceptionList.get(i);
            if (readInfoException != null) {
                exceptionList.set(i, readInfoException);
            }
        }
        // データベースから読み込んだMngInfoのリストを1つに集約する
        final MngInfo newMngInfo;
        switch (mngInfoList.size()) {
            case 1: {
                // 片系から取得できたなら、それをデータベースから読んだMngInfoとして扱う
                newMngInfo = mngInfoList.get(0);
                break;
            }
            case 2: {
                // 両系から取得できたなら、MngInfoの合成を行う
                newMngInfo  = MngInfo.synthMngInfoFromDatabase(mngInfoList.get(0), mngInfoList.get(1));
                if (newMngInfo == null) {
                    // 合成に失敗した場合はinitに失敗
                    throw new ForestInitFailedException(ErrorStr.MNGINIT_SYNTH_INFO_FAILED.toString());
                }
                break;
            }
            default: {
                // 1個もとれなかった場合はinitに失敗
                final ForestInitFailedException e = new ForestInitFailedException(ErrorStr.MNGINIT_LOAD_INFO_FAILED.toString());
                e.setMultipleCause(exceptionList);
                throw e;
            }
        }
        // ここまで例外なしで通ったなら、作ったMngInfoを公開する
        mngInfo.set(newMngInfo);
        
        initLogger();
        
        logging(Level.INFO, LogStr.INFO_MNGINFO_INIT_SUCCESS.toString());
        logging(Level.CONFIG, getMngInfo().toString());
        
        // 管理情報更新の間隔が0以上なら、管理情報更新のスレッドを開始する
        if (newMngInfo.getGlobalConfig().getMngdbReadDuration() > 0) {
            // 初期実行は10秒後、それ以降はタスクが終わり次第1秒後に再度タスクを再開する。
            // （タスク自身が内部的にsleepを発行する。このsleepがMngInfo中のリフレッシュ間隔に従って
            // 動作しているため、ここで間隔の調整をする必要はない。また、1秒の間隔を設定しているのは
            // 万一Sleepが短時間で戻るようになってしまった場合に高負荷状態となるのを防ぐため）
            mnginfoRefreshScheduler.scheduleWithFixedDelay(new MngInfoRefreshTask(), 10, 1, TimeUnit.SECONDS);
            logging(Level.INFO, LogStr.INFO_MNGINFO_REFRESH_TASK_START.toString());
        }
    }
    
    /**
     * 管理情報読み込みのための両系へのコネクションプール（mngConnections）を補給する関数<br>
     * この関数を呼ぶと、コネクションがプールに無い場合には補給し、コネクションがプールに
     * ある場合には何もしない
     * @throws InterruptedException コネクション生成最中に割り込みが発生した場合
     * @return serveridをインデックスとして発生したエラーのリスト
     */
    private List<Exception> supplyConnection() throws InterruptedException {
        // コネクションが無い系について、コネクション補充をするタスクを作る
        // コネクションが既にある系については、何も実行しないタスクを作る
        final List<Callable<Connection>> conCreateTasks = new ArrayList<Callable<Connection>>(2);
        for (int i = 0; i < 2; i++) {
            if (mngInfoReadCons.get(i) == null) {
                conCreateTasks.add(new CreateConnectionTask(targetUrl.getMngDbUrls().get(i), mngdbUsername, mngdbPassword));
            } else {
                conCreateTasks.add(new CreateConnectionTask(null, null, null));
            }
        }
        // コネクション生成のタスクを実行する
        final List<Future<Connection>> conCreateFutureList;
        final int conCreateTimeout;
        if (getMngInfo() == null) {
            conCreateTimeout = ConstInt.MNGINIT_CONNECT_TIMEOUT.getInt();
        } else {
            conCreateTimeout = getMngInfo().getGlobalConfig().getMngConCreateTimeout();
        }
        try {
            conCreateFutureList = mngdbApiExecutor.invokeAll(conCreateTasks, conCreateTimeout, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            // 割り込みが発生した場合、スレッドに割り込みフラグを再度立てた上で即座に呼び出し元に戻る
            Thread.currentThread().interrupt();
            throw e;
        }
        // 返却する例外のリスト
        final List<Exception> exceptionList = new ArrayList<Exception>(2);
        
        // コネクション生成タスクの実行結果を取り出す
        for (int i = 0; i < 2; i++) {
            try {
                // コネクションが新規に生成できたならコネクションプールにセットする。
                // 出来ていない場合は何もしない。できていない＝nullタスクをセットした場合
                // なので、そもそも作る必要がないため例外を上げる必要がない。
                final Connection newCon = conCreateFutureList.get(i).get();
                if (newCon != null) {
                    mngInfoReadCons.set(i, newCon);
                }
                exceptionList.add(null);
            } catch (CancellationException e) {
                exceptionList.add(e);
                logging(Level.WARNING, LogStr.WARN_MNGINFO_SUPPLYCON_CANCEL.toString(i));
            } catch (ExecutionException e) {
                if (e.getCause() instanceof Exception) {
                    exceptionList.add((Exception) e.getCause());
                    logging(Level.WARNING, LogStr.WARN_MNGINFO_SUPPLYCON_EXCEPTION.toString(i), Level.FINE,  e.getCause());
                } else {
                    logging(Level.SEVERE, LogStr.SEVERE_MNGINFO_SUPPLYCON_ERROR.toString(i), Level.FINE, e.getCause());
                    throw (Error) e.getCause();
                }
            } catch (InterruptedException e) {
                // 割り込みが発生した場合、スレッドに割り込みフラグを再度立てた上で即座に呼び出し元に戻る
                Thread.currentThread().interrupt();
                throw e;
            }
        }
        return exceptionList;
    }
    
    /**
     * コネクションプール（mngConnections）に存在するコネクションを使い、MngInfoを読み出す。
     * Pairの1番目の要素には、管理DBから読みだした管理情報（読みだせなかった場合にはnull）
     * がサーバIDをインデックスとして格納されている。
     * Pairの2番目の要素には、管理DBから読みだす際に発生したエラーがサーバIDをインデックス
     * として格納されている。但し、コネクションプールが存在しない状態で呼び出された場合、
     * エラーは特に格納されない。
     * また、このペアに格納される管理情報は、MngInfoManagerが保持しているForestUrlとサーバ
     * 接続先情報が等しいことが保障され、接続先情報が異なる場合もnullが格納される。
     * nullが格納された場合、その系のコネクションは解放され、コネクションプールから取り除かれる。
     * @throws InterruptedException 割り込みが発生した場合
     */
    private Pair2<List<MngInfo>, List<Exception>> readMngInfoFromDb() throws InterruptedException {
        // MngInfoを各系から読み込むためのタスクを作る
        final List<Callable<MngInfo>> mnginfoReadTasks = new ArrayList<Callable<MngInfo>>(2);
        for (int i = 0; i < 2; i++) {
            mnginfoReadTasks.add( new ReadMngInfoTask(mngInfoReadCons.get(i)) );
        }
        // MngInfoを各系から読み込むタスクを実行する
        final int queryExecTimeout;
        if (getMngInfo() == null) {
            queryExecTimeout = ConstInt.MNGINIT_QUERYEXEC_TIMEOUT.getInt();
        } else {
            queryExecTimeout = getMngInfo().getGlobalConfig().getMngQueryExecTimeout();
        }
        final List<Future<MngInfo>> readMngInfoFutureList;
        try {
            readMngInfoFutureList = mngdbApiExecutor.invokeAll(mnginfoReadTasks, queryExecTimeout, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            // 割り込みが発生した場合、スレッドに割り込みフラグを再度立てそのまま例外を投げる
            Thread.currentThread().interrupt();
            throw e;
        }
        // MngInfo読み込みタスクの実行結果を取り出す
        final List<MngInfo> mngInfoList = new ArrayList<MngInfo>(2);
        final List<Exception> exceptionList = new ArrayList<Exception>(2);
        for (int i = 0; i < 2; i++) {
            final MngInfo tmpMngInfo;
            try {
                // 正常にMngInfoが取れたならリストに追加し、それ以外の場合はnullとする
                tmpMngInfo = readMngInfoFutureList.get(i).get();
                if (tmpMngInfo != null) {
                    if (tmpMngInfo.equalServerInfo(targetUrl)) {
                        mngInfoList.add(tmpMngInfo);
                        exceptionList.add(null);
                    } else {
                        // MngInfoは取得できたが本来取得できるサーバの情報でない場合はエラーとする
                        mngInfoList.add(null);
                        exceptionList.add(new ForestException(ErrorStr.MNGINIT_URL_DIFFERENT.toString()));
                    }
                } else {
                    // 例外は返らずにnullだけが返った場合は、コネクションプールがそもそも
                    // 確保できずに実行をしなかった場合なので、この場合は管理情報・読み込み時の
                    // エラー共にnullとする
                    mngInfoList.add(null);
                    exceptionList.add(null);
                }
            } catch (InterruptedException e) {
                // 割り込みが発生した場合、スレッドに割り込みフラグを再度立てた上でそのまま例外を投げる
                Thread.currentThread().interrupt();
                throw e;
            } catch (ExecutionException e) {
                // 実行中にエラーが起きた場合、そのコネクションは解放する
                final Connection closeTarget = mngInfoReadCons.getAndSet(i, null);
                if (closeTarget != null) {
                    try {
                        // クエリが実行中の場合、バックエンドが止まってくれないためキャンセルする
                        ((org.postgresql.jdbc2.AbstractJdbc2Connection) closeTarget).cancelQuery();
                    } catch (SQLException ignore) {
                    } finally {
                        try { closeTarget.close(); } catch (SQLException ignore) { }
                    }
                }
                if (e.getCause() instanceof Error) {
                    logging(Level.SEVERE, LogStr.SEVERE_MNGINFO_READDB_ERROR.toString(i), Level.FINE, e.getCause());
                    throw (Error) e.getCause();
                } else {
                    mngInfoList.add(null);
                    exceptionList.add((Exception) e.getCause());
                    logging(Level.WARNING, LogStr.WARN_MNGINFO_READDB_EXCEPTION.toString(i), Level.FINE,  e.getCause());
                }
            } catch (CancellationException e) {
                // 実行が指定時間内に終わらなかった場合、そのコネクションは解放する
                final Connection closeTarget = mngInfoReadCons.getAndSet(i, null);
                mngInfoList.add(null);
                exceptionList.add(e);
                logging(Level.WARNING, LogStr.WARN_MNGINFO_READDB_CANCEL.toString(i));
                if (closeTarget != null) {
                    try {
                        // クエリが実行中の場合、バックエンドが止まってくれないためキャンセルする
                        ((org.postgresql.jdbc2.AbstractJdbc2Connection) closeTarget).cancelQuery();
                    } catch (SQLException ignore) {
                    } finally {
                        try { closeTarget.close(); } catch (SQLException ignore) { }
                    }
                }
            }
        }
        
        return new Pair2<List<MngInfo>, List<Exception>>(mngInfoList, exceptionList);
    }
    
    private final class MngInfoRefreshTask implements Runnable {
        public MngInfoRefreshTask() {}
        public void run() {
            try {
                logging(Level.INFO, LogStr.INFO_MNGINFO_REFRESH_TASK_START.toString());
                
                final List<Exception> conCreateExceptionList;
                final Pair2<List<MngInfo>, List<Exception>> readMngInfoPair;
                try {
                    // まずコネクションプールのコネクションを補充し（DBアクセス）
                    conCreateExceptionList = supplyConnection();
                    // プールのコネクションを使ってデータベースからMngInfoを読み込む（DBアクセス）
                    readMngInfoPair = readMngInfoFromDb();
                } catch (InterruptedException e) {
                    // MngInfo読み込み中に割り込まれた場合、割りこみを呼び出し元にそのまま広める
                    // （終了処理に応答するため）
                    Thread.currentThread().interrupt();
                    return;
                }
                
                // このタスク中で比較対象とする現在のMngInfoのスナップショットを取得しておく
                final MngInfo currentMngInfo = getMngInfo();
                
                // 取得できた（nullでない）mngInfoのみを集める。
                // あわせてその系が縮退ではなかったにも関わらず情報取得に失敗した場合、
                // 後で縮退対象かどうか決定のために連続エラーのカウントをインクリメントする
                final List<MngInfo> mngInfoWithNullList = readMngInfoPair.getFirst();
                final List<MngInfo> mngInfoList = new ArrayList<MngInfo>(2);
                for (int i = 0; i < mngInfoWithNullList.size(); i++) {
                    final MngInfo targetMngInfo = mngInfoWithNullList.get(i);
                    if (targetMngInfo != null) {
                        mngInfoList.add(targetMngInfo);
                    } else {
                        if (currentMngInfo.getValidityList().get(0) == UdbValidity.VALID) {
                            burstRefreshError.incrementAndGet(i);
                        }
                    }
                }
                
                // コネクションプール補充時と管理情報読み込み時に発生したエラーを合わせる
                final List<Exception> exceptionList = new ArrayList<Exception>(2);
                exceptionList.addAll(conCreateExceptionList);
                for(int i = 0; i < readMngInfoPair.getSecond().size(); i++) {
                    final Exception readMngInfoException = readMngInfoPair.getSecond().get(i);
                    if (readMngInfoException != null) {
                        exceptionList.set(i, readMngInfoException);
                    }
                }
                
                // 管理情報をどちらの系からも取得できなかった場合、現在のリフレッシュ間隔に
                // 従ってタスクを一時停止し、次のタスクを実行する
                if (mngInfoList.size() == 0) {
                    // TODO (情報出力) logWithExceptionを使ってログに例外のスタックトレースを出すべき？
                    logging(Level.SEVERE, LogStr.SEVERE_MNGINFO_REFRESH_FAIL.toString());
                    try {
                        final int duration = getMngInfo().getGlobalConfig().getMngdbReadDuration();
                        Thread.sleep(duration * 1000);
                    } catch (InterruptedException ee) {
                        // 一時停止中に割り込まれた場合、interruptをセットした上で
                        // すみやかに終了する
                        Thread.currentThread().interrupt();
                    }
                    return;
                }
                
                // データベースから読み込んだMngInfoのリストを1つに集約する
                final MngInfo tmpMngInfo;
                switch (mngInfoList.size()) {
                    case 1: {
                        // 片系から取得できたなら、それをデータベースから読んだMngInfoとして扱う
                        tmpMngInfo = mngInfoList.get(0);
                        break;
                    }
                    case 2: {
                        // 両系から取得できたならMngInfoの合成をする。
                        // 合成に失敗した場合には現在JVM中に存在するMngInfoをそのまま使う
                        MngInfo synthesizedMngInfo = MngInfo.synthMngInfoFromDatabase(mngInfoList.get(0), mngInfoList.get(1));
                        tmpMngInfo = (synthesizedMngInfo == null) ? currentMngInfo : synthesizedMngInfo;
                        break;
                    }
                    default: {
                        // 1個もとれなかった場合、現在JVM中に存在するMngInfoをそのまま使う
                        tmpMngInfo = currentMngInfo;
                        break;
                    }
                }
                
                // JVM中のMngInfoのうち、ServerInfoの情報に関して状態遷移を行う。
                // 許可された状態遷移であれば、それに伴い発生する操作を実施し、
                // 異常な状態遷移でないことをフラグにセットする。
                // 許可されない状態遷移なら、何もせずにswitchを抜ける
                final boolean isIllegalStatusChange;
                switch (currentMngInfo.getEnumValidity()) {
                    case VALID_VALID: {
                        switch (tmpMngInfo.getEnumValidity()) {
                            case VALID_INVALID:
                            case INVALID_VALID:
                            case INVALID_INVALID:
                            case VALID_VALID:
                                isIllegalStatusChange = false;
                                break;
                            default:
                                isIllegalStatusChange = true;
                        }
                        break;
                    }
                    case VALID_INVALID: {
                        switch (tmpMngInfo.getEnumValidity()) {
                            case RECOVER_INVALID:
                                // リカバリのための抑制状態とする
                                waitRecoveryLatch.set(new CountDownLatch(1));
                            case INVALID_INVALID:
                            case VALID_INVALID:
                                isIllegalStatusChange = false;
                                break;
                            default:
                                isIllegalStatusChange = true;
                        }
                        break;
                    }
                    case INVALID_VALID: {
                        switch (tmpMngInfo.getEnumValidity()) {
                            case INVALID_RECOVER:
                                // リカバリのための抑制状態とする
                                waitRecoveryLatch.set(new CountDownLatch(1));
                            case INVALID_INVALID:
                            case INVALID_VALID:
                                isIllegalStatusChange = false;
                                break;
                            default:
                                isIllegalStatusChange = true;
                        }
                        break;
                    }
                    case RECOVER_INVALID: {
                        switch (tmpMngInfo.getEnumValidity()) {
                            case VALID_VALID:
                                // 2系のリカバリ処理が完了したため、中継用Executorの復旧と
                                // リカバリ待ちオブジェクトへの通知をする
                                createProxyApiExecutor(1);
                                notifyRecoveryCompleted(1);
                            case VALID_INVALID:
                            case INVALID_INVALID:
                                // リカバリ状態解除のため、ラッチを解放・削除する
                                final CountDownLatch latch = waitRecoveryLatch.getAndSet(null);
                                if (latch != null) {
                                    latch.countDown();
                                }
                            case RECOVER_INVALID:
                                isIllegalStatusChange = false;
                                break;
                            default:
                                isIllegalStatusChange = true;
                        }
                        break;
                    }
                    case INVALID_RECOVER: {
                        switch (tmpMngInfo.getEnumValidity()) {
                            case VALID_VALID:
                                // 1系のリカバリ処理が完了したため、中継用Executorの復旧と
                                // リカバリ待ちオブジェクトへの通知をする
                                createProxyApiExecutor(0);
                                notifyRecoveryCompleted(0);
                            case INVALID_VALID:
                            case INVALID_INVALID:
                                // リカバリが完了したためラッチを解放・削除する
                                final CountDownLatch latch = waitRecoveryLatch.getAndSet(null);
                                if (latch != null) {
                                    latch.countDown();
                                }
                            case INVALID_RECOVER:
                                isIllegalStatusChange = false;
                                break;
                            default:
                                isIllegalStatusChange = true;
                        }
                        break;
                    }
                    default:
                        // TODO (異常系) そもそもこの箇所にくるということは、現在のudb_validityがRECOVER_RECOVERなどの異常な状態になっているため、動作を停止するべきかもしれない？
                        isIllegalStatusChange = true;
                }
                
                // udbValidityの遷移に異常がなければ、新しいudbValidityの値を使用する。
                // 遷移に異常があった場合、元の値を使用した上で、その旨ロギングする。
                final List<MngInfo.ServerInfo> newServerInfoList;
                if (isIllegalStatusChange == false) {
                    newServerInfoList = tmpMngInfo.getServerInfoList();
                } else {
                    newServerInfoList = currentMngInfo.getServerInfoList();
                    logging(Level.WARNING, LogStr.WARN_MNGINFO_ILLEGAL_STATE_CHANGE.toString(currentMngInfo.getEnumValidity(), tmpMngInfo.getEnumValidity()));
                }
                
                // コンフィグレーション値等は、新規にDBから読み込んだものを無条件に使用する。
                // udbValidityに関しては上で新しいものを使用するか否か判断しているため、
                // その結果と新しいコンフィグレーション値をここで合成して、最終的な次の管理情報値とする。
                final MngInfo newMngInfo = new MngInfo(newServerInfoList, tmpMngInfo.getGlobalConfig(), tmpMngInfo.getLocalConfigMap());
                
                // 現在のMngInfoと新規に作ったMngInfoが論理的に異なる場合のみ変数を書き換える
                if (!currentMngInfo.equals(newMngInfo)) {
                    if (mngInfo.compareAndSet(currentMngInfo, newMngInfo) == false) {
                        // AtomicReferenceのcompareAndSetがfalseで戻っている場合、
                        // この関数の頭でcurrentMngInfoを読み出してからここまでの間に
                        // 他者にmngInfoの参照を付け替えられていることを意味する。
                        // そのため、再度この関数を実行する必要があるので、
                        // 以下の待ちを行わず、即関数を終える（次のスケジュール実行を行う）
                        logging(Level.INFO, LogStr.INFO_MNGINFO_REFRESH_CONFLICT.toString());
                        return;
                    }
                    // ステータスが変化しているときはステータスが変化したログを出力する
                    if (currentMngInfo.getEnumValidity() != newMngInfo.getEnumValidity()) {
                        logging(Level.SEVERE,
                                LogStr.SEVERE_MNGINFO_STATUS_CHANGED.toString(currentMngInfo.getEnumValidity(),newMngInfo.getEnumValidity()));
                        // 特にVALID（正常）からINVALID（縮退）へと変化があった場合、
                        // 縮退した側のデーターベースからの応答を待っているユーザスレッドに
                        // 即時に応答を返すために、キャンセル用リソースを解放する
                        for (int i = 0; i < 2; i++) {
                            if (currentMngInfo.getValidityList().get(i) == UdbValidity.VALID &&
                                    newMngInfo.getValidityList().get(i) == UdbValidity.INVALID) {
                                cancelUdbExecution(i);
                            }
                        }
                    }
                    
                    // MngInfoの切り替えに成功し、なおかつLogConfigが以前のものと論理的に
                    // 異なっている場合には、ロガーを新しい設定で起動しなおす
                    if (!currentMngInfo.getGlobalConfig().getLogConfig().equals(
                            newMngInfo.getGlobalConfig().getLogConfig())) {
                        initLogger();
                    }
                }
                
                // ステータスが両系正常だが、GlobalConfigで規定された回数以上に連続で
                // 管理情報読み込みに失敗している場合、縮退処理をする
                do {
                    final MngInfo targetMngInfo = getMngInfo();
                    final int permitBurstErrorCount = targetMngInfo.getGlobalConfig().getMngdbMaxBurstErrorCount();
                    // permitBurstErrorCount == -1 の場合は無限に異常を許す。つまり縮退のルートは通らない
                    if (permitBurstErrorCount < 0) {
                        break;
                    }
                    if (targetMngInfo.getEnumValidity() == EnumValidity.VALID_VALID) {
                        for (int i = 0; i < 2; i++) {
                            if (burstRefreshError.get(i) > permitBurstErrorCount &&
                                    burstRefreshError.get(1 - i) <= permitBurstErrorCount) {
                                // 縮退して管理情報を書き換える処理は失敗する可能性があるため、
                                // 失敗した場合には再度ループを実行する
                                if (setUdbInvalid(targetMngInfo, i, exceptionList.get(i), MngInfoRefreshTask.class.getName(), "") == false) {
                                    continue;
                                }
                            }
                        }
                    }
                } while(false);
                
                logging(Level.INFO, LogStr.INFO_MNGINFO_REFRESH_TASK_END.toString(newMngInfo.getEnumValidity()));
                logging(Level.CONFIG, getMngInfo().toString());
                try {
                    final int duration = newMngInfo.getGlobalConfig().getMngdbReadDuration();
                    // 現在のリフレッシュ間隔に従い、タスクを一時停止する
                    Thread.sleep(duration * 1000);
                } catch (InterruptedException e) {
                    // 一時停止中に割り込まれた場合、interruptをセットした上で
                    // すみやかに終了する
                    Thread.currentThread().interrupt();
                }
            } catch (OutOfMemoryError e) {
                // 管理情報の更新処理内でOutOfMemoryが起きた場合には、
                // 管理情報更新スレッドが消えないようにOutOfMemoryを握りつぶし、
                // 10秒後にメモリが解放されていることを信じてsleepする
                try {
                    // OutOfMemoryが定常的に起きる状況だと無限ループでCPUを食って
                    // しまうため、10秒間のsleepを入れる
                    Thread.sleep(10000);
                } catch (OutOfMemoryError ignore) {
                } catch (InterruptedException einterrupt) {
                    // MngInfo読み込み中に割り込まれた場合、割りこみを呼び出し元にそのまま広める
                    // （終了処理に応答するため）
                    Thread.currentThread().interrupt();
                    return;
                }
                try {
                    // OutOfMemoryが起きたことをログ出力する
                    // ここでさらにOutOfMemoryが起きたら何もしない
                    logging(Level.WARNING, LogStr.INFO_MNGINFO_REFRESH_OUTOFMEMORY.toString());
                } catch (OutOfMemoryError ignore) {
                }
            } catch (Throwable e) {
                // 想定外の例外・エラーが起きたため、管理情報定期更新スレッドを停止する
                logging(Level.SEVERE, LogStr.SEVERE_MNGINFO_REFRESH_THREAD_UBNORMAL_STOP.toString(), Level.SEVERE, e);
                if (e instanceof Error) {
                    throw (Error) e;
                } else if (e instanceof RuntimeException) {
                    throw (RuntimeException) e;
                } else {
                    // Error でも RuntimeException でもない場合、Runnable.run関数の仕様上
                    // その例外を投げられないので、RuntimeExceptionにくるんで無理やり投げる
                    throw new RuntimeException(e);
                }
            }
        }
    }
    
    /**
     * 与えられたPgUrlを使って、対応する管理データベースへの
     * コネクションを作成するためのタスククラス<br>
     * コンストラクタで与えられたPgUrlがnullの場合は何も行わない
     */
    @Immutable private static final class CreateConnectionTask implements Callable<Connection> {
        private final PgUrl targetUrl;
        private final Properties prop;
        public CreateConnectionTask(final PgUrl targetUrl, final String user, final String pass) {
            this.targetUrl = targetUrl;
            this.prop = new Properties();
            if (user != null && pass != null) {
                prop.setProperty("user", user);
                prop.setProperty("password", pass);
            }
        }
        public Connection call() throws Exception {
            if (targetUrl == null) {
                return null;
            }
            // DriverManager.getConnectionを使わない理由は、
            // 1. 作るインスタンスが分かっているにも関わらずDriverManagerを
            //    経由するのはオーバーヘッドが大きい
            // 2. Java5のDriverManagerがロックを握る実装になっており、
            //    executorでDriverManager.getConnectionを実行すると、
            //    確実にデッドロックが発生してしまう
            // の2点
            final java.sql.Driver driver = new org.postgresql.Driver();
            
            return driver.connect(targetUrl.getUrl(), prop);
        }
    }
    
    /**
     * 与えられたコネクションを使用して、管理情報DBから情報を読み込んで
     * MngInfoを作成するためのタスククラス<br>
     * このクラスのcall()は、コネクションの解放を行わない<br>
     * また、コンストラクタで与えられたコネクションがnullの場合は何も行わずにnullを返す
     */
    @Immutable private static final class ReadMngInfoTask implements Callable<MngInfo> {
        private final Connection con;
        public ReadMngInfoTask(final Connection con) {
            this.con = con;
        }
        public MngInfo call() throws Exception {
            if (con == null) {
                return null;
            }
            final Statement stmt = con.createStatement();
            try {
                // ServerInfoの生成
                // サーバIDは0か1なので、サイズが2のnullを格納した配列を最初に作り、
                // そこにそれぞれDBから読んだ値を格納する。
                final List<MngInfo.ServerInfo> serverInfoList = new ArrayList<MngInfo.ServerInfo>();
                serverInfoList.add(null);
                serverInfoList.add(null);
                final ResultSet svinfoRes = stmt.executeQuery("SELECT * FROM " + ConstStr.MNGDB_TBL_SERVERINFO.toString());
                // 各行をDBから読んで、ServerInfoのリストを構築する
                while (svinfoRes.next()) {
                    final int serverid = svinfoRes.getInt(ConstStr.MNGDB_COL_SERVERINFO_ID.toString());
                    if (serverid != 0 && serverid != 1) {
                        throw new ForestInvalidMngInfoException(ErrorStr.MNGDB_SERVERID_INVALID.toString());
                    }
                    final String udburl = svinfoRes.getString(ConstStr.MNGDB_COL_SERVERINFO_UDBURL.toString());
                    final UdbValidity udbValidity = 
                        UdbValidity.getEnum(svinfoRes.getInt(ConstStr.MNGDB_COL_SERVERINFO_VALID.toString()));
                    if (udbValidity == null) {
                        throw new ForestInvalidMngInfoException(ErrorStr.MNGDB_UDBVALIDITY_INVALID.toString());
                    }
                    serverInfoList.set(serverid, new MngInfo.ServerInfo(udburl, udbValidity));
                }
                // ServerInfoのリストに、インデックス0・1共に値がセットされているか確認する
                if (serverInfoList.get(0) == null || serverInfoList.get(1) == null) {
                    throw new ForestInvalidMngInfoException(ErrorStr.MNGDB_SERVERID_INVALID.toString());
                }
                
                // GlobalConfig（LogConfig含む）の作成
                final HashMap<String, String> gconfMap = new HashMap<String, String>();
                final ResultSet gconfRes = stmt.executeQuery("SELECT * FROM " + ConstStr.MNGDB_TBL_GLOBALCONFIG.toString());
                while (gconfRes.next()) {
                    final String key = gconfRes.getString(ConstStr.MNGDB_COL_GCONF_KEY.toString());
                    final String value = gconfRes.getString(ConstStr.MNGDB_COL_GCONF_VALUE.toString());
                    gconfMap.put(key, value); 
                }
                final MngInfo.GlobalConfig globalConfig = new MngInfo.GlobalConfig(gconfMap);
                
                // LocalConfigの作成
                final HashMap<String, HashMap<String, String>> lconfMap = new HashMap<String, HashMap<String, String>>();
                // DB上に存在しない場合でも、DEFAULTのコンフィグ値を用意する
                lconfMap.put(ConstStr.FOREST_URL_DEFAULT_CONFIGID.toString(), new HashMap<String, String>());
                final ResultSet lconfRes = stmt.executeQuery("SELECT * FROM " + ConstStr.MNGDB_TBL_LOCALCONFIG.toString());
                while (lconfRes.next()) {
                    final String configid = lconfRes.getString(ConstStr.MNGDB_COL_LCONF_CONFIGID.toString());
                    final String key = lconfRes.getString(ConstStr.MNGDB_COL_LCONF_KEY.toString());
                    final String value = lconfRes.getString(ConstStr.MNGDB_COL_LCONF_VALUE.toString());
                    if (lconfMap.containsKey(configid) == false) {
                        lconfMap.put(configid, new HashMap<String, String>());
                    }
                    final HashMap<String, String> lconfEntry = lconfMap.get(configid);
                    lconfEntry.put(key, value);
                }
                final Map<String, MngInfo.LocalConfig> localConfigMap = new HashMap<String, MngInfo.LocalConfig>();
                for(final Map.Entry<String, HashMap<String, String>> lconfEntry : lconfMap.entrySet()) {
                    final String configid = lconfEntry.getKey();
                    localConfigMap.put(configid, new MngInfo.LocalConfig(lconfEntry.getValue()));
                }
                
                return new MngInfo(serverInfoList, globalConfig, localConfigMap);
                
            } finally {
                stmt.close();
            }
        }
    }
    
    
    // 縮退操作
    
    /**
     * 指定されたidのユーザデータベースの整合性情報をINVALIDとして、
     * 新規MngInfoのスナップショットを作成・公開する。<br>
     * 縮退と判断した時点の管理情報を引数として取るが、この関数が管理情報を
     * 書き換えようとしている最中に他スレッドによって管理情報を変更されている
     * 場合、この関数はfalseを返し、何も実施しない。
     * @param oldMngInfo 縮退と判断した時点でのMngInfoのスナップショット
     * @param serverId 縮退状態へステータスを変更する対象のサーバID（0 or 1）
     * @param exception 縮退と判断した例外
     * @param taskClassName 縮退が発生した時に実行していたタスクのクラス名
     * @param query 縮退が発生した時に実行していたSQL
     * @return 置き換えが成功した場合はtrue、置き換えまでに他のスレッドにより置き換えが発生していた場合はfalse
     */
    public boolean setUdbInvalid(final MngInfo oldMngInfo, final int serverId, final Exception exception, final String taskClassName, final String query) {
        // 現在のMngInfoから取得したServerInfoリストを基に、縮退させた後のServerInfoを作成する
        final List<MngInfo.ServerInfo> oldServerInfoList = oldMngInfo.getServerInfoList();
        final MngInfo.ServerInfo targetServerInfo = oldServerInfoList.get(serverId);
        final MngInfo.ServerInfo newServerInfo = 
            targetServerInfo.getNewValidityServerInfo(UdbValidity.INVALID);
        
        // 縮退させたServerInfoを含む、新しいServerInfoのリストを作成する
        // （oldServerInfoListは変更不可のため、新規にリストを生成する）
        final List<MngInfo.ServerInfo> newServerInfoList = new ArrayList<MngInfo.ServerInfo>();
        newServerInfoList.addAll(oldServerInfoList);
        newServerInfoList.set(serverId, newServerInfo);
        
        // 新規にMngInfoを作って登録する。仮にMngInfoを作ってから登録するまでの間に
        // 別のものに置き換えられていたとしたら、置き換えは失敗したことを呼び出し側に通知する
        final MngInfo newMngInfo = new MngInfo(newServerInfoList, oldMngInfo.getGlobalConfig(), oldMngInfo.getLocalConfigMap());
        if (mngInfo.compareAndSet(oldMngInfo, newMngInfo) == false) {
            return false;
        }
        // ここまで到達すると、オンメモリの管理情報中では縮退が反映されている。
        // 続いて管理DB上に縮退の情報を書き込む（厳密には上のCompareAndSet操作を
        // する段階から、DBへの書き込みが完了するまでの間を、他の処理が同時に行わない
        // ようにロックをとる必要があるが、仮にロックを握っても別アプリ（JVM）から
        // アクセスがあればここでの排他制御の意味はなくなってしまうことと、DBへの
        // アクセス中にロックを握りっぱなしになり危険なことと、そもそも管理DBへの
        // 処理は非同期に行いたいため、ロックを握れない。よって排他制御は行わない）
        final String appThreadName = Thread.currentThread().getName();
        final List<StackTraceElement> appThreadStack =
            Arrays.<StackTraceElement>asList(Thread.currentThread().getStackTrace());
        mngdbApiExecutor.execute(new WriteUdbInvalidTask(targetUrl.getMngDbUrls().get(0), serverId, exception, appThreadName, appThreadStack, taskClassName, query, mngdbApiExecutor, mngdbUsername, mngdbPassword));
        mngdbApiExecutor.execute(new WriteUdbInvalidTask(targetUrl.getMngDbUrls().get(1), serverId, exception, appThreadName, appThreadStack, taskClassName, query, mngdbApiExecutor, mngdbUsername, mngdbPassword));
        // 縮退した側のデーターベースからの応答を待っているユーザスレッドに
        // 即時に応答を返すために、キャンセル用リソースを解放する
        cancelUdbExecution(serverId);
        // 縮退を自発的に行ったことをロガーに通知
        logging(Level.SEVERE, LogStr.SEVERE_BROKEN_SERVER.toString(serverId), Level.SEVERE, exception);
        return true;
    }

    /**
     * 即時停止可能状態で実行されたJDBC-APIを即時停止する。
     * この関数は自発的縮退発見・受動的縮退通知時に呼び出す。
     * @param serverId 縮退となったサーバのID（0 or 1）
     */
    private void cancelUdbExecution(final int serverId) {
        final ExecutorService releaseThreadPool = udbApiProxyExecutor.getAndSet(serverId, null);
        if (releaseThreadPool != null) {
            try {
                releaseThreadPool.shutdownNow();
            } catch (SecurityException ignore) {
            }
        }
    }
    
    /** 縮退情報を管理DBに書き込むためのタスク */
    @Immutable private static final class WriteUdbInvalidTask implements Runnable {
        private final PgUrl pgUrl;
        private final int serverId;
        private final ExecutorService executor;
        private final Exception exception;
        private final String appThreadName;
        private final List<StackTraceElement> appThreadStack;
        private final String taskClassName;
        private final String query;
        private final String username;
        private final String password;
        public WriteUdbInvalidTask(
                final PgUrl pgUrl,
                final int serverId,
                final Exception exception,
                final String appThreadName,
                final List<StackTraceElement> appThreadStack,
                final String taskClassName,
                final String query,
                final ExecutorService executor,
                final String user,
                final String pass) {
            this.pgUrl = pgUrl;
            this.serverId = serverId;
            this.executor = executor;
            this.exception = exception;
            this.appThreadName = appThreadName;
            this.taskClassName = taskClassName;
            this.query = query;
            this.appThreadStack = appThreadStack;
            this.username = user;
            this.password = pass;
        }
        public void run() {
            // 書き込むためのコネクションを作成
            final Callable<Connection> createConTask = new CreateConnectionTask(pgUrl, username, password);
            final Future<Connection> createConFuture = executor.submit(createConTask);
            final Connection con;
            try {
                con = createConFuture.get(10, TimeUnit.SECONDS);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return;
            } catch (ExecutionException e) {
                return;
            } catch (TimeoutException e) {
                return;
            }
            
            // 実際にDBに縮退情報を書き込む
            Statement stmt = null;
            PreparedStatement pstmt = null;
            try {
                
                // serverinfoの更新
                stmt = con.createStatement();
                final StringBuilder updateBuffer = new StringBuilder();
                updateBuffer.append("UPDATE ");                                           // UPDATE
                updateBuffer.append(ConstStr.MNGDB_TBL_SERVERINFO.toString());            // server_info
                updateBuffer.append(" SET ");                                             // SET
                updateBuffer.append(ConstStr.MNGDB_COL_SERVERINFO_VALID.toString());      // udb_validity
                updateBuffer.append(" = ");                                               // =
                updateBuffer.append(UdbValidity.INVALID.getInt());                        // -1
                updateBuffer.append(" WHERE ");                                           // WHERE
                updateBuffer.append(ConstStr.MNGDB_COL_SERVERINFO_ID.toString());         // serverid
                updateBuffer.append(" = ");                                               // =
                updateBuffer.append(serverId);                                            // 引数のID
                stmt.executeUpdate(updateBuffer.toString());
                
                // brokenlogの挿入
                final StringBuilder insertBuffer = new StringBuilder();
                insertBuffer.append("INSERT INTO ");
                insertBuffer.append(ConstStr.MNGDB_TBL_BROKENLOG.toString());
                insertBuffer.append(" (");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_SERVERID.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_TIME.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_CLIENT.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_APITASK.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRMSG.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRTYPE.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRSTATE.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRQUERY.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_ERRSTACK.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_APPTHREAD.toString());
                insertBuffer.append(",");
                insertBuffer.append(ConstStr.MNGDB_COL_BROKENLOG_APPSTACK.toString());
                insertBuffer.append(") VALUES (?, now(), ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                pstmt = con.prepareStatement(insertBuffer.toString());
                
                // serverid
                pstmt.setInt(1, serverId);
                // client
                final List<String> ipList = new ArrayList<String>();
                try {
                    for (final NetworkInterface nic : Collections.list(NetworkInterface.getNetworkInterfaces())) {
                        for (final InetAddress addr : Collections.list(nic.getInetAddresses())) {
                            ipList.add(addr.getHostAddress());
                        }
                    }
                } catch (Exception e) {
                    ipList.clear();
                    ipList.add("-");
                }
                Collections.sort(ipList);
                pstmt.setString(2, ipList.toString());
                // api_task
                pstmt.setString(3, taskClassName);
                // err_msg
                pstmt.setString(4, exception.toString());
                // err_type
                pstmt.setString(5, exception.getClass().getName());
                // err_state
                String sqlState;
                if (exception instanceof SQLException) {
                    sqlState = ((SQLException) exception).getSQLState();
                } else {
                    sqlState = "-";
                }
                pstmt.setString(6, (sqlState != null) ? sqlState : "-");
                // err_query
                pstmt.setString(7, query);
                // err_stack
                final StringBuilder errStackBuffer = new StringBuilder();
                for (final StackTraceElement elem : exception.getStackTrace()) {
                    errStackBuffer.append("\n\t");
                    errStackBuffer.append(elem.toString());
                }
                pstmt.setString(8, errStackBuffer.toString());
                // app_thread
                pstmt.setString(9, appThreadName);
                // app_stack
                final StringBuilder appStackBuffer = new StringBuilder();
                for (final StackTraceElement elem : appThreadStack) {
                    appStackBuffer.append("\n\t");
                    appStackBuffer.append(elem.toString());
                }
                pstmt.setString(10, appStackBuffer.toString());
                
                pstmt.executeUpdate();
                
            } catch (SQLException ignore) {
            } finally {
                if (stmt != null) {
                    try { stmt.close(); } catch (SQLException ignore) { }
                }
                if (pstmt != null) {
                    try { pstmt.close(); } catch (SQLException ignore) { }
                }
                try { con.close(); } catch (SQLException ignore) { }
            }
        }
    }
    
    
    // リカバリ処理
    
    /**
     * リカバリ時のAPI実行抑制のためのラッチ。このAtomicReferenceにラッチをセットした場合、
     * セットしたラッチは必ずいつかのタイミングでCountDownされなくてはならない。
     */
    private final AtomicReference<CountDownLatch> waitRecoveryLatch = new AtomicReference<CountDownLatch>(null);
    
    /**
     * リカバリのために更新抑制フェーズであるならば、リカバリが完了
     * （成功・失敗問わず）するまでこの関数は待ち状態となり、抑制解除に
     * なった段階で戻ってくる。抑制がかかっていない状態であれば何もせず
     * 即座に返る。
     * @throws InterruptedException ラッチを待つ間にinterruptされた場合
     */
    public void waitRecovery() throws InterruptedException {
        final CountDownLatch latch = waitRecoveryLatch.get();
        if (latch == null) {
            return;
        } else {
            latch.await();
        }
    }
    
    /** recoveryListenerListを操作する場合に取得するロック。このロック取得無しに変更してはならない */
    private final Object lockListenerList = new Object();
    
    /**
     * リカバリ通知対象となるオブジェクトを保持するリスト。
     * MngInfoManagerからコネクションに関係するオブジェクトを握るのはまずい
     * （強参照による循環参照となる）ため、弱参照で保持することとする。
     * 弱参照なので、リストの要素を使用する際に必ずnullチェックが必要。
     */
    @GuardedBy("lockListenerList")
    private final List<WeakReference<RecoveryCompletedListener>> recoveryListenerList =
        new LinkedList<WeakReference<RecoveryCompletedListener>>();
    
    /**
     * リカバリが正常完了した際に、RecoveryCompletedListenerで定義された
     * コールバック関数を呼ぶように登録するための関数。
     * @param listener 登録したいコールバック関数を実装したオブジェクト
     */
    @GuardedBy("lockListenerList")
    public void setRecoveryCompletedListener(final RecoveryCompletedListener newListener) {
        synchronized (lockListenerList) {
            final WeakReference<RecoveryCompletedListener> newWeakListener =
                new WeakReference<RecoveryCompletedListener>(newListener);
            
            recoveryListenerList.add(newWeakListener);
            
            // 肥大化（リーク）を防ぐために、10の倍数の数に達した段階でリストを走査して、
            // 弱参照が消えているものをチェックしてそのエントリを削除する
            if (recoveryListenerList.size() % 10 == 0) {
                final ListIterator<WeakReference<RecoveryCompletedListener>> listIter =
                    recoveryListenerList.listIterator();
                
                while (listIter.hasNext()) {
                    final RecoveryCompletedListener listener = listIter.next().get();
                    if (listener == null) {
                        listIter.remove();
                    }
                }
            }
        }
    }
    
    /**
     * setRecoveryCompletedListenerで登録されたリスナーに対して、
     * setRecoveryCompleted関数を呼び出してリカバリ完了を通知する。
     * @param serverId リカバリ処理が完了したサーバID
     */
    @GuardedBy("lockListenerList")
    public void notifyRecoveryCompleted(final int serverId) {
        synchronized (lockListenerList) {
            final ListIterator<WeakReference<RecoveryCompletedListener>> listIter =
                recoveryListenerList.listIterator();
            
            while (listIter.hasNext()) {
                final RecoveryCompletedListener listener = listIter.next().get();
                if (listener != null) {
                    listener.setRecoveryCompleted(serverId);
                } else {
                    listIter.remove();
                }
            }
        }
    }
}
