/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.mng;

import org.postgresforest.constant.ConstStr;
import org.postgresforest.constant.ConstInt;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.constant.UdbValidity;
import org.postgresforest.exception.*;
import org.postgresforest.util.*;

import java.util.*;
import java.util.concurrent.*;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.jcip.annotations.*;

/**
 * 管理DBから取得した管理情報を格納するためのクラス
 */
@Immutable public final class MngInfo {
    
    /** serveridをインデックスとしてserverinfoテーブルの内容を格納したList */
    private final List<MngInfo.ServerInfo> serverInfoList = new CopyOnWriteArrayList<MngInfo.ServerInfo>();
    /**
     * serverinfoテーブルの内容を格納したリストを返却する<br>
     * serveridをリストのインデックスとして、接続先url、接続UDBの有効性、が格納されている。
     * 本関数で取得したListは並行アクセス可能だが、更新はできない。
     * @return
     */
    public List<MngInfo.ServerInfo> getServerInfoList() {
        return Collections.unmodifiableList(serverInfoList);
    }
    
    /** global_configテーブルの内容を格納したオブジェクト */
    private final MngInfo.GlobalConfig globalConfig;
    /**
     * global_configを返却する
     * @return このMngInfoが保持するglobal_configテーブルの内容
     */
    public MngInfo.GlobalConfig getGlobalConfig() {
        return globalConfig;
    }
    
    /** configidをキーとしてlocal_configテーブルの内容を格納したMap */
    private final Map<String, LocalConfig> localConfigMap = new ConcurrentHashMap<String, MngInfo.LocalConfig>();
    
    public Map<String, LocalConfig> getLocalConfigMap() {
        return Collections.unmodifiableMap(localConfigMap);
    }
    
    /**
     * MngInfoのコンストラクタ<br>
     * 次の3つの要素を引数として与える必要がある。<br>
     * - ServerInfoテーブルのserverid=0/1のレコード2件分のリスト<br>
     * - GlobalConfigテーブルのレコード1件分<br>
     * - LocalConfigテーブルのレコードN件分のマップ<br>
     * 
     * @param serverinfos ServerInfoテーブルの各レコードを、ServerIdをインデックスとしてリスト化したもの
     * @param globalconfig GlobalConfigテーブルの1件
     * @param localConfigs LocalConfigテーブルの各レコードを、configidをキーとしてマップ化したもの
     */
    public MngInfo(final List<ServerInfo> serverinfos, final GlobalConfig globalConfig, final Map<String, LocalConfig> localConfigs) {
        this.serverInfoList.addAll(serverinfos);
        this.globalConfig = globalConfig;
        this.localConfigMap.putAll(localConfigs);
        
        // 整合性情報をおさめたリストを作成
        final List<UdbValidity> tmpValidityList = new ArrayList<UdbValidity>(2);
        for (final ServerInfo svinfo : serverInfoList) {
            tmpValidityList.add(svinfo.udbValidity);
        }
        validityList = Collections.unmodifiableList(tmpValidityList);
        
        // 両系の整合性情報を並べた列挙型を作成
        final UdbValidity id0_status = validityList.get(0);
        final UdbValidity id1_status = validityList.get(1);
        if (id0_status == UdbValidity.VALID) {
            if (id1_status == UdbValidity.VALID) {
                enumValidity = EnumValidity.VALID_VALID;
            } else if (id1_status == UdbValidity.RECOVERY) {
                enumValidity =  EnumValidity.VALID_RECOVER;
            } else {
                enumValidity =  EnumValidity.VALID_INVALID;
            }
        } else if (id0_status == UdbValidity.RECOVERY) {
            if (id1_status == UdbValidity.VALID) {
                enumValidity =  EnumValidity.RECOVER_VALID;
            } else if (id1_status == UdbValidity.RECOVERY) {
                enumValidity =  EnumValidity.RECOVER_RECOVER;
            } else {
                enumValidity =  EnumValidity.RECOVER_INVALID;
            }
        } else {
            if (id1_status == UdbValidity.VALID) {
                enumValidity =  EnumValidity.INVALID_VALID;
            } else if (id1_status == UdbValidity.RECOVERY) {
                enumValidity =  EnumValidity.INVALID_RECOVER;
            } else {
                enumValidity =  EnumValidity.INVALID_INVALID;
            }
        }
        
    }
    
    public enum EnumValidity {
        VALID_VALID     ("id0=valid, id1=valid"),
        VALID_RECOVER   ("id0=valid, id1=recovery"),
        VALID_INVALID   ("id0=valid, id1=invalid"),
        RECOVER_VALID   ("id0=recovery, id1=valid"),
        RECOVER_RECOVER ("id0=recovery, id1=recovery"),
        RECOVER_INVALID ("id0=recovery, id1=invalid"),
        INVALID_VALID   ("id0=invalid, id1=valid"),
        INVALID_RECOVER ("id0=invalid, id1=recovery"),
        INVALID_INVALID ("id0=invalid, id1=invalid");
        private final String toStr;
        EnumValidity(String toStr) {
            this.toStr = toStr;
        }
        @Override public String toString() {
            return toStr;
        }
    }
    private final EnumValidity enumValidity;
    /**
     * 両系のUDB整合性情報を列挙型EnumValidityでまとめて返却する関数
     */
    public EnumValidity getEnumValidity() {
        return enumValidity;
    }
    
    private final List<UdbValidity> validityList;
    /**
     * UDB整合性情報をリストの形で返却する関数
     * @return サーバIDをインデックスとして整合性情報を列挙した変更不可能なリスト
     */
    public List<UdbValidity> getValidityList() {
        return validityList;
    }
    
    /**
     * 論理的に同値ならtrueを返す。（Object#equalsのオーバーロード）<br>
     * 保持しているServerInfoリスト、GlobalConfig、LocalConfigマップが全て等しい
     * 場合のみ同値とみなす
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof MngInfo)) {
            return false;
        }
        final MngInfo target = (MngInfo) o;
        if (this.serverInfoList.equals(target.serverInfoList) &&
            this.globalConfig.equals(target.globalConfig) &&
            this.localConfigMap.equals(target.localConfigMap)) {
            return true;
        }
        return false;
    }
    /**
     * hashCodeを返す（Object#hashCodeのオーバーロード）<br>
     * 論理的に同値なら同一の値を返す
     */
    public int hashCode() {
        int result = 17;
        result = 37 * result + this.serverInfoList.hashCode();
        result = 37 * result + this.globalConfig.hashCode();
        result = 37 * result + this.localConfigMap.hashCode();
        return result;
    }
    
    /**
     * このMngInfoの設定値をログ出力可能な形式で返す
     */
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("\nCurrent mnginfo\n");
        sb.append(createConfigLogTitle("server_info"));
        for (int i = 0; i < 2; i++) {
            sb.append(serverInfoList.get(i).toString());
        }
        sb.append(createConfigLogTitle("global_config"));
        sb.append(globalConfig.toString());
        sb.append(createConfigLogTitle("local_config"));
        for (final Map.Entry<String, LocalConfig> entry : localConfigMap.entrySet()) {
            sb.append(createConfigLogEntry(ConstStr.MNGDB_COL_LCONF_CONFIGID.toString(), entry.getKey()));
            sb.append(entry.getValue());
        }
        return sb.toString();
    }
    
    /**
     * MngInfo中のServerInfoテーブルの内容のみを比較する関数
     * @param info 比較対象となるMngInfoオブジェクト
     * @return サーバ情報が等しければtrue、異なっていればfalse
     */
    public boolean equalServerInfo(final MngInfo info) {
        // MngInfoはImmutableオブジェクトのため、参照が等しいなら中身は確実に一緒
        if (this == info) {
            return true;
        }
        if (info == null) {
            return false;
        }
        // ServerInfoのリストの中身が等しければ、同一であることを返す
        return this.serverInfoList.equals(info.serverInfoList);
    }
    
    /**
     * MngInfo中のServerInfoテーブルの内容が、指定したForestUrlの接続文字列
     * と等しいか否かを比較する関数
     * @param targetUrl このMngInfoが保持するServerInfoと比較するForestUrl
     * @return 接続先が等しければtrue、異なっていればfalse
     */
    public boolean equalServerInfo(final ForestUrl targetUrl) {
        
        // まずは新規のmngInfoからIP:PORT文字列を取り出し
        // (2個固定)リストを文字列の自然順にソートする
        final List<String> mngInfoIpPort = new ArrayList<String>(2);
        for (final MngInfo.ServerInfo serverinfo : serverInfoList) {
            mngInfoIpPort.add(serverinfo.getIpPort());
        }
        Collections.sort(mngInfoIpPort);
        
        // 続いてPgUrlからIP:PORT/DBNAME文字列を取り出し、これも文字列の自然順にソートする
        final List<String> pgUrlIpPort = new ArrayList<String>();
        for (final PgUrl pgurl : targetUrl.getMngDbUrls()) {
            pgUrlIpPort.add(pgurl.getIpPort());
        }
        Collections.sort(pgUrlIpPort);
        
        // ソートした二つのリストを比較し、結果を返却する
        return mngInfoIpPort.equals(pgUrlIpPort);
    }
    
    private final List<PgUrl> mngDbs = new CopyOnWriteArrayList<PgUrl>();
    /**
     * 管理情報DBへのPgUrlオブジェクトの、並行アクセス可能・変更不可能なリストを返す。
     * このリストは、ServerIdをインデックスとして並べられたリストとなっている
     */
    public List<PgUrl> getMngDbUrls() {
        return Collections.unmodifiableList(mngDbs);
    }
    
    /**
     * ２つの管理データベースから取得した管理情報を合成し、1つのものとする。
     * この関数は、MngInfo中のServerInfoの情報を比較し、あるUDBのudb_validityが
     * 異なっている場合、以下の規則に従ってudb_validityの新しい値を決定する。<br>
     * <br>
     * - 稼働中 vs 障害中 ⇒ 障害中<br>
     * - 稼働中 vs リカバリ（稼働中だがアクセス抑制）中 ⇒ リカバリ中<br>
     * - 障害中 vs リカバリ（稼働中だがアクセス抑制）中 ⇒ 障害中<br>
     * <br>
     * ２つの管理情報間でIP:PORT/DBNAMEが異なっている場合、合成に失敗する。
     * 
     * @param mngInfo1 取得した管理情報の１つ
     * @param mngInfo2 取得した管理情報の１つ
     * @return 合成した結果の管理情報、但し合成に失敗した場合にはnullが返る
     */
    public static MngInfo synthMngInfoFromDatabase(final MngInfo mngInfo1, final MngInfo mngInfo2) {
        final List<ServerInfo> svinfoList1 = mngInfo1.serverInfoList;
        final List<ServerInfo> svinfoList2 = mngInfo2.serverInfoList;
        final List<ServerInfo> newSvInfoList = new ArrayList<ServerInfo>(2);
        newSvInfoList.add(null);
        newSvInfoList.add(null);
        
        if (svinfoList1.size() != 2 || svinfoList2.size() != 2) {
            // 両系のServerInfo数が2以外の場合は合成不能
            return null;
        }
        
        for (int i = 0; i < 2; i++) {
            final ServerInfo svinfo1 = svinfoList1.get(i);
            final ServerInfo svinfo2 = svinfoList2.get(i);
            if (svinfo1.equals(svinfo2)) {
                // 完全に一致していれば、そのまま片方の値を使う
                newSvInfoList.set(i, svinfo1);
            } else {
                // 完全に一致していない場合、IP:PORT/DBNAMEが等しいか検証
                if (svinfo1.getIpPort().equals(svinfo2.getIpPort()) &&
                        svinfo1.getDbName().equals(svinfo2.getDbName())) {
                    // udb_validityの新しい値を決定する
                    // どちらかに障害中があれば、障害中とし、障害中がなければリカバリ中とする。
                    // これは上記の新しいudb_validityを決定するアルゴリズムに従っている。
                    final UdbValidity newStatus;
                    if (svinfo1.udbValidity == UdbValidity.INVALID || svinfo2.udbValidity == UdbValidity.INVALID) {
                        newStatus = UdbValidity.INVALID;
                    } else {
                        newStatus = UdbValidity.RECOVERY;
                    }
                        
                    // 新しいudb_validityの値を用いて、ServerInfoを作り直す
                    final ServerInfo newSvInfo = svinfo1.getNewValidityServerInfo(newStatus);
                    newSvInfoList.set(i, newSvInfo);
                } else {
                    // IP:PORT/DBNAMEの値が違うので、この場合合成不可能
                    return null;
                    // TODO この関数の仕様として、合成できなくても現在のIP:PORT/DBNAMEと等しい方を返すことが必要
                }
            }
        }
        
        // 生成したnewSvinfoListを使って新しいMngInfoを生成する
        // GlobalConfig/LocalConfigは仕様上どちらも同じでなくてはならないので、片方の値をそのまま使う
        return new MngInfo(newSvInfoList, mngInfo1.globalConfig, mngInfo1.localConfigMap);
    }
    
    /**
     * 設定項目のグループ名をログ出力に適した文字列として返す
     * @param title 設定項目のグループ名
     * @return ログ出力に適した形の文字列
     */
    public static String createConfigLogTitle(final String title) {
        return "  ====================" + title + "====================\n";
    }
    /**
     * 設定項目と値をログ出力に適した文字列として返す
     * @param col 設定項目名
     * @param val 設定値
     * @return ログ出力に適した形の文字列
     */
    public static String createConfigLogEntry(final String col, final String val) {
        return String.format("%30s : %s\n", col, val);
    }
    
    /**
     * 管理情報のserver_infoテーブルのレコードを保持するための内部クラス<br>
     * 但し、server_infoテーブルのserverid列はこのクラス中に含めない。
     * server_infoテーブルの詳細は外部仕様定義書を参照のこと。
     */
    @Immutable public static final class ServerInfo {
        private final String ipport;
        private final String dbname;
        private final UdbValidity udbValidity;
        private final static Pattern pattern = Pattern.compile("(.+:.+)/(.+)");
        /**
         * ServerInfoインスタンスのコンストラクタ<br>
         * 引数が正しい値かどうかを検証し、インスタンスを生成する。
         * @param udburl 各ユーザデータベースへの接続文字列<br>
         * 接続文字列は、「IP:PORT/DB」の形でなくてはならない
         * @param udbValidity 各ユーザデータベースの正当性を表す。-1,0,1の3値
         * @throws ForestInvalidMngInfoException 引数が仕様とマッチしない場合
         */
        public ServerInfo(final String udburl, final UdbValidity udbValidity) throws ForestInvalidMngInfoException {
            final Matcher matcher = pattern.matcher(udburl);
            if (matcher.matches() == false) {
                throw new ForestInvalidMngInfoException(ErrorStr.MNGDB_UDBURL_INVALID.toString());
            }
            this.ipport = matcher.group(1);
            this.dbname = matcher.group(2);
            this.udbValidity = udbValidity;
        }
        /** このServerInfoが保持しているIPとポート番号を連結した文字列を返す */
        public String getIpPort() {
            return ipport;
        }
        /** このServerInfoが保持しているデータベース名を返す */
        public String getDbName() {
            return dbname;
        }
        /** このServerInfoが保持しているユーザデータベースの整合性情報を返す */
        public UdbValidity getUdbValidity() {
            return udbValidity;
        }
        /**
         * 本関数を呼び出されたServerInfoの値を基として、ユーザデータベースの
         * 正当性に関する値のみを変更した新規ServerInfoインスタンスを返却する。
         * @param udbValidity
         * @return 新規にステータス
         */
        public ServerInfo getNewValidityServerInfo(final UdbValidity udbValidity) {
            try {
                return new ServerInfo(ipport + "/" + dbname, udbValidity);
            } catch (ForestInvalidMngInfoException e) {
                // udbValidityの値をプログラム内部でミスっている以外にここに来ることはあり得ない
                throw new IllegalArgumentException(ErrorStr.ILLEGAL_ARGUMENT.toString(), e);
            }
        }
        /**
         * 論理的に同値ならtrueを返す。（Object#equalsのオーバーロード）<br>
         * コンストラクタで与えた接続文字列と
         * 各ユーザデータベースの正当性の情報が全て一致している場合にのみ
         * 論理的に同一であるとみなす。
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof ServerInfo)) {
                return false;
            }
            final ServerInfo target = (ServerInfo) o;
            if (this.ipport.equals(target.ipport) &&
                this.dbname.equals(target.dbname) &&
                this.udbValidity == target.udbValidity) {
                return true;
            }
            return false;
        }
        /**
         * hashCodeを返す（Object#hashCodeのオーバーロード）<br>
         * 論理的に同値なら同一の値を返す
         */
        public int hashCode() {
            int result = 17;
            result = 37 * result + ipport.hashCode();
            result = 37 * result + dbname.hashCode();
            result = 37 * result + udbValidity.hashCode();
            return result;
        }
        
        public String toString() {
            return MngInfo.createConfigLogEntry(ipport + "/" + dbname, udbValidity.toString());
        }
    }
    
    @Immutable public static final class GlobalConfig {
        private final int mngdbReadDuration;
        private final int mngConCreateTimeout;
        private final int mngQueryExecTimeout;
        private final int mngdbMaxBurstErrorCount;
        private final String brokenErrorPrefix;
        private final LogConfig logConfig;
        public GlobalConfig(final HashMap<String, String> globalConfigMap) {
            
            // mngdbReadDuration (0 - inifinity)
            {
                final int mngdbReadDurationDef = ConstInt.MNGDB_DEF_GCONF_READDURATION.getInt();
                int mngdbReadDuration;
                try {
                    mngdbReadDuration = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_MNGREADDURATION.toString()));
                } catch (Exception e) {
                    mngdbReadDuration = mngdbReadDurationDef;
                }
                this.mngdbReadDuration = (mngdbReadDuration < 0) ? mngdbReadDurationDef : mngdbReadDuration;
            }
            
            // mngConCreateTimeout (0 - infinity)
            {
                final int mngConCreateTimeoutDef = ConstInt.MNGDB_DEF_GCONF_MNGCONCREATETO.getInt();
                int mngConCreateTimeout;
                try {
                    mngConCreateTimeout = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_MNGCONCREATETO.toString()));
                } catch (Exception e) {
                    mngConCreateTimeout = mngConCreateTimeoutDef;
                }
                if (mngConCreateTimeout == 0) {
                    // TODO （仕様検討）管理情報出力時に0に見えないがこれでよいのか？
                    this.mngConCreateTimeout = Integer.MAX_VALUE;
                } else {
                    this.mngConCreateTimeout = (mngConCreateTimeout < 0) ? mngConCreateTimeoutDef : mngConCreateTimeout;
                }
                // this.mngConCreateTimeout = (mngConCreateTimeout < 0) ? mngConCreateTimeoutDef : mngConCreateTimeout;
            }
            
            // mngQueryExecTimeout (0 - infinity)
            {
                final int mngQueryExecTimeoutDef = ConstInt.MNGDB_DEF_GCONF_MNGQUERYEXECTO.getInt();
                int mngQueryExecTimeout;
                try {
                    mngQueryExecTimeout = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_MNGQUERYEXECTO.toString()));
                } catch (Exception e) {
                    mngQueryExecTimeout = mngQueryExecTimeoutDef;
                }
                if (mngQueryExecTimeout == 0) {
                    // TODO （仕様検討）管理情報出力時に0に見えないがこれでよいのか？
                    this.mngQueryExecTimeout = Integer.MAX_VALUE;
                } else {
                    this.mngQueryExecTimeout = (mngQueryExecTimeout < 0) ? mngQueryExecTimeoutDef : mngQueryExecTimeout;
                }
                // this.mngQueryExecTimeout = (mngQueryExecTimeout < 0) ? mngQueryExecTimeoutDef : mngQueryExecTimeout;
            }
            
            // mngdbMaxBurstErrorCount (-1 - infinity)
            {
                final int mngdbMaxBurstErrorCountDef = ConstInt.MNGDB_DEF_GCONF_MNGMAXBURSTERROR.getInt();
                int mngdbMaxBurstErrorCount;
                try {
                    mngdbMaxBurstErrorCount = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_MNGMAXBURSTERROR.toString()));
                } catch (Exception e) {
                    mngdbMaxBurstErrorCount = mngdbMaxBurstErrorCountDef;
                }
                this.mngdbMaxBurstErrorCount = (mngdbMaxBurstErrorCount < -1) ? mngdbMaxBurstErrorCountDef : mngdbMaxBurstErrorCount;
            }
            
            // brokenErrorPrefix
            {
                String brokenErrorPrefix = null;
                brokenErrorPrefix = globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_BROKENPREFIX.toString());
                if (brokenErrorPrefix == null) {
                    brokenErrorPrefix = ConstStr.MNGDB_DEF_GCONF_BROKENPREFIX.toString();
                }
                this.brokenErrorPrefix = brokenErrorPrefix;
            }
            
            // logConfig
            this.logConfig = new LogConfig(globalConfigMap);
            
        }
        public int getMngdbReadDuration() {
            return mngdbReadDuration;
        }
        public int getMngConCreateTimeout() {
            return mngConCreateTimeout;
        }
        public int getMngQueryExecTimeout() {
            return mngQueryExecTimeout;
        }
        public String getBrokenErrorPrefix() {
            return brokenErrorPrefix;
        }
        public LogConfig getLogConfig() {
            return logConfig;
        }
        public int getMngdbMaxBurstErrorCount() {
            return mngdbMaxBurstErrorCount;
        }
        /**
         * 論理的に同値ならtrueを返す。（Object#equalsのオーバーロード）<br>
         * 全てのプライベートフィールドが等しい場合に同値とみなす
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof GlobalConfig)) {
                return false;
            }
            final GlobalConfig target = (GlobalConfig) o;
            if (this.brokenErrorPrefix.equals(target.brokenErrorPrefix) &&
                this.mngdbReadDuration == target.mngdbReadDuration &&
                this.mngConCreateTimeout == target.mngConCreateTimeout &&
                this.mngQueryExecTimeout == target.mngQueryExecTimeout &&
                this.mngdbMaxBurstErrorCount == target.mngdbMaxBurstErrorCount &&
                this.logConfig.equals(target.logConfig)) {
                return true;
            }
            return false;
        }
        /**
         * hashCodeを返す（Object#hashCodeのオーバーロード）<br>
         * 論理的に同値なら同一の値を返す
         */
        public int hashCode() {
            int result = 17;
            result = 37 * result + brokenErrorPrefix.hashCode();
            result = 37 * result + mngdbReadDuration;
            result = 37 * result + mngConCreateTimeout;
            result = 37 * result + mngQueryExecTimeout;
            result = 37 * result + mngdbMaxBurstErrorCount;
            result = 37 * result + logConfig.hashCode();
            return result;
        }
        public String toString() {
            final StringBuilder sb = new StringBuilder();
            sb.append(createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_MNGREADDURATION.toString(), String.valueOf(mngdbReadDuration)));
            sb.append(createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_MNGCONCREATETO.toString(), String.valueOf(mngConCreateTimeout)));
            sb.append(createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_MNGQUERYEXECTO.toString(), String.valueOf(mngQueryExecTimeout)));
            sb.append(createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_MNGMAXBURSTERROR.toString(), String.valueOf(mngdbMaxBurstErrorCount)));
            sb.append(createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_BROKENPREFIX.toString(), brokenErrorPrefix));
            sb.append(logConfig.toString());
            return sb.toString();
        }
    }
    
    @Immutable public static final class LocalConfig {
        private final int apiExecTimeout;
        private final boolean isFastApiReturnOnDbFail;
        public LocalConfig(final HashMap<String, String> localConfigMap) {
            
            // apiExecTimeout (0 - infinity, 0の場合MAX_VALUE)
            {
                final int apiExecTimeoutDef = ConstInt.MNGDB_DEF_LCONF_APIEXECTO.getInt();
                int apiExecTimeout;
                try {
                    apiExecTimeout = Integer.valueOf(localConfigMap.get(ConstStr.MNGDB_KEY_LCONF_APIEXECTO.toString()));
                } catch (Exception e) {
                    apiExecTimeout = apiExecTimeoutDef;
                }
                if (apiExecTimeout == 0) {
                    // TODO （仕様検討）管理情報出力時に0に見えないがこれでよいのか？
                    apiExecTimeout = Integer.MAX_VALUE;
                }
                this.apiExecTimeout = (apiExecTimeout < 0) ? apiExecTimeoutDef : apiExecTimeout;
            }
            
            // fast_api_return_on_db_fail (0 or 1)
            {
                final int isFastApiReturnOnDbFail_def = ConstInt.MNGDB_DEF_LCONF_FASTAPIRETURN.getInt();
                int isFastApiReturnOnDbFail;
                try {
                    isFastApiReturnOnDbFail = Integer.valueOf(localConfigMap.get(ConstStr.MNGDB_KEY_LCONF_FASTAPIRETURN.toString()));
                } catch (Exception e) {
                    isFastApiReturnOnDbFail = isFastApiReturnOnDbFail_def;
                }
                if (isFastApiReturnOnDbFail != 0 && isFastApiReturnOnDbFail != 1) {
                    isFastApiReturnOnDbFail = isFastApiReturnOnDbFail_def;
                }
                this.isFastApiReturnOnDbFail = (isFastApiReturnOnDbFail == 1);
            }
        }
        public int apiExecTimeout() {
            return apiExecTimeout;
        }
        
        public boolean getIsFastApiReturnOnDbFail() {
            return isFastApiReturnOnDbFail;
        }
        /**
         * 論理的に同値ならtrueを返す。（Object#equalsのオーバーロード）<br>
         * 全てのプライベートフィールドが等しい場合に同値とみなす
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof LocalConfig)) {
                return false;
            }
            final LocalConfig target = (LocalConfig) o;
            if (this.apiExecTimeout == target.apiExecTimeout &&
                this.isFastApiReturnOnDbFail == target.isFastApiReturnOnDbFail) {
                return true;
            }
            return false;
        }
        /**
         * hashCodeを返す（Object#hashCodeのオーバーロード）<br>
         * 論理的に同値なら同一の値を返す
         */
        public int hashCode() {
            int result = 17;
            result = 37 * result + apiExecTimeout;
            result = 37 * result + (isFastApiReturnOnDbFail ? 1 : 0);
            return result;
        }
        public String toString() {
            final StringBuilder sb = new StringBuilder();
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_LCONF_APIEXECTO.toString(), String.valueOf(apiExecTimeout)));
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_LCONF_FASTAPIRETURN.toString(), String.valueOf(isFastApiReturnOnDbFail)));
            return sb.toString();
        }
    }
    
    @Immutable public static final class LogConfig {
        private final Level level;
        private final Class<?> formatter_class;
        private final boolean console_enable;
        private final boolean file_enable;
        private final String file_name;
        private final int file_size_mb;
        private final int file_rotate_count;
        public LogConfig(final HashMap<String, String> globalConfigMap) {
            
            // level (0 (= OFF) - 5( = ALL))
            {
                final int levelDef = ConstInt.MNGDB_DEF_GCONF_LOGLEVEL.getInt();
                int level;
                try {
                    level = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_LOGLEVEL.toString()));
                } catch (Exception e ) {
                    level = levelDef;
                }
                if (level < 0 || 5 < level) {
                    level = levelDef;
                }
                switch (level) {
                    case 0:
                        this.level = Level.OFF;
                        break;
                    case 1:
                        this.level = Level.SEVERE;
                        break;
                    case 2:
                        this.level = Level.WARNING;
                        break;
                    case 3:
                        this.level = Level.INFO;
                        break;
                    case 4:
                        this.level = Level.CONFIG;
                        break;
                    case 5:
                        this.level = Level.ALL;
                        break;
                    default:
                        throw new IllegalStateException();
                }
            }
            
            // formatter_class ( class extends java.util.logging.Formatter )
            {
                final Class<?> formatter_class_def = org.postgresforest.util.ForestLogFormatter.class;
                Class<?> formatter_class;
                try {
                    formatter_class = Class.forName(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_LOGFORMATTERNAME.toString()));
                    // java.util.logging.Formatterの派生クラスかどうか検証し、
                    if (java.util.logging.Formatter.class.isAssignableFrom(formatter_class) == false) {
                        formatter_class = formatter_class_def;
                    }
                } catch (Exception e) {
                    formatter_class = formatter_class_def;
                }
                this.formatter_class = formatter_class;
            }
            
            // console_enable (0 or 1 -> boolean)
            {
                final int console_enable_def = ConstInt.MNGDB_DEF_GCONF_LOGCONSOLEENABLE.getInt();
                int console_enable;
                try {
                    console_enable = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_LOGCONSOLEENABLE.toString()));
                } catch (Exception e) {
                    console_enable = console_enable_def;
                }
                if (console_enable != 0 && console_enable != 1) {
                    console_enable = console_enable_def;
                }
                this.console_enable = (console_enable == 1);
            }
            
            // file_enable (0 or 1 -> boolean)
            {
                final int file_enable_def = ConstInt.MNGDB_DEF_GCONF_LOGFILEENABLE.getInt();
                int file_enable;
                try {
                    file_enable = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_LOGFILEENABLE.toString()));
                } catch (Exception e) {
                    file_enable = file_enable_def;
                }
                if (file_enable != 0 && file_enable != 1) {
                    file_enable = file_enable_def;
                }
                this.file_enable = (file_enable == 1);
            }
            
            // file_name
            {
                String file_name = null;
                file_name = globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_LOGFILENAME.toString());
                if (file_name == null) {
                    file_name = ConstStr.MNGDB_DEF_GCONF_LOGFILENAME.toString();
                }
                this.file_name = file_name;
            }
            
            // file_size_mb (0 - infinity)
            {
                final int file_size_mb_def = ConstInt.MNGDB_DEF_GCONF_LOGFILESIZE.getInt();
                int file_size_mb;
                try {
                    file_size_mb = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_LOGFILESIZE.toString()));
                } catch (Exception e) {
                    file_size_mb = file_size_mb_def;
                }
                this.file_size_mb = (file_size_mb < 0) ? file_size_mb_def : file_size_mb;
            }
            
            // file_rotate_count (1 - infinity)
            {
                int file_rotate_count_def = ConstInt.MNGDB_DEF_GCONF_LOGFILEROTATE.getInt();
                int file_rotate_count;
                try {
                    file_rotate_count = Integer.valueOf(globalConfigMap.get(ConstStr.MNGDB_KEY_GCONF_LOGFILEROTATE.toString()));
                } catch (Exception e) {
                    file_rotate_count = file_rotate_count_def;
                }
                this.file_rotate_count = (file_rotate_count < 1) ? file_rotate_count_def : file_rotate_count;
            }
            
        }
        @Override public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof LogConfig)) {
                return false;
            }
            final LogConfig target = (LogConfig) o;
            if (this.level.equals(target.level) &&
                this.formatter_class.equals(target.formatter_class) &&
                this.console_enable == target.console_enable &&
                this.file_enable == target.file_enable &&
                this.file_name.equals(target.file_name) &&
                this.file_size_mb == target.file_size_mb &&
                this.file_rotate_count == target.file_rotate_count) {
                return true;
            }
            return false;
        }
        @Override public int hashCode() {
            int result = 17;
            result = 37 * result + level.hashCode();
            result = 37 * result + formatter_class.hashCode();
            result = 37 * result + (console_enable ? 1 : 0);
            result = 37 * result + (file_enable ? 1 : 0);
            result = 37 * result + file_name.hashCode();
            result = 37 * result + file_size_mb;
            result = 37 * result + file_rotate_count;
            return result;
        }
        @Override public String toString() {
            final StringBuilder sb = new StringBuilder();
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_LOGLEVEL.toString(), level.toString()));
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_LOGFORMATTERNAME.toString(), formatter_class.getName()));
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_LOGCONSOLEENABLE.toString(), String.valueOf(console_enable)));
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_LOGFILEENABLE.toString(), String.valueOf(file_enable)));
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_LOGFILENAME.toString(), file_name));
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_LOGFILESIZE.toString(), String.valueOf(file_size_mb)));
            sb.append(MngInfo.createConfigLogEntry(ConstStr.MNGDB_KEY_GCONF_LOGFILEROTATE.toString(), String.valueOf(file_rotate_count)));
            return sb.toString();
        }
        public Level getLevel() { return level; }
        public Class<?> getFormatter() { return formatter_class; }
        public boolean getConsoleEnable() { return console_enable; }
        public boolean getFileEnable() { return file_enable; }
        public String getFileName() { return file_name; }
        public int getFileSizeMb() { return file_size_mb; }
        public int getFileRotateCount() { return file_rotate_count; }
    }
}
