/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.apibase;

import java.sql.*;

import org.postgresforest.exception.ForestTaskNotExecutedException;
import org.postgresforest.util.ForestTask;

/**
 * PostgreSQLのJDBCドライバに対してStatementクラスのAPIを実行するための
 * Callableタスクのインプリを集約したクラス。このクラス自体は、メンバや
 * メソッドを持たない（内部クラスを持つのみ）
 * 
 * 各タスクは、コンストラクタで与えられたJDBCリソースについて、nullか否かを
 * 必ずチェックする必要がある。nullの場合、その系は縮退していることを意味し、
 * call内で即座に「ForestTaskNotExecutedException」をスローする。
 */
public final class StatementTask {
    
    public static final class AddBatch_String extends ForestTask<Void> {
        private final Statement stmt;
        private final String sql;
        public AddBatch_String(final int serverid, final Statement stmt, final String sql) {
            super(serverid);
            this.stmt = stmt;
            this.sql = sql;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.addBatch(sql);
            return null;
        }
    }
    
    public static final class Cancel extends ForestTask<Void> {
        private final Statement stmt;
        public Cancel(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.cancel();
            return null;
        }
    }
    
    public static final class ClearBatch_String extends ForestTask<Void> {
        private final Statement stmt;
        public ClearBatch_String(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.clearBatch();
            return null;
        }
    }
    
    public static final class Execute extends ForestTask<Boolean> {
        private final Statement stmt;
        private final String sql;
        public Execute(final int serverid, final Statement stmt, String sql) {
            super(serverid);
            this.stmt = stmt;
            this.sql = sql;
        }
        public Boolean call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Boolean.valueOf(stmt.execute(sql));
        }
    }
    
    public static final class ExecuteBatch extends ForestTask<int[]> {
        private final Statement stmt;
        public ExecuteBatch(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public int[] call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return stmt.executeBatch();
        }
    }
    
    public static final class ExecuteUpdate extends ForestTask<Integer> {
        private final Statement stmt;
        private final String sql;
        public ExecuteUpdate(final int serverid, final Statement stmt, String sql) {
            super(serverid);
            this.stmt = stmt;
            this.sql = sql;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.executeUpdate(sql));
        }
    }
    
    public static final class ExecuteQuery extends ForestTask<ResultSet> {
        private final Statement stmt;
        private final String sql;
        public ExecuteQuery(final int serverid, final Statement stmt, String sql) {
            super(serverid);
            this.stmt = stmt;
            this.sql = sql;
        }
        public ResultSet call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return stmt.executeQuery(sql);
        }
    }
    
    public static final class ClearWarnings extends ForestTask<Void> {
        private final Statement stmt;
        public ClearWarnings(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.clearWarnings();
            return null;
        }
    }
    
    public static final class SetCursorName extends ForestTask<Void> {
        private final Statement stmt;
        private final String arg0;
        public SetCursorName(final int serverid, final Statement stmt, String arg0) {
            super(serverid);
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setCursorName(arg0);
            return null;
        }
    }
    
    public static final class SetEscapeProcessing extends ForestTask<Void> {
        private final Statement stmt;
        private final boolean arg0;
        public SetEscapeProcessing(final int serverid, final Statement stmt, boolean arg0) {
            super(serverid);
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setEscapeProcessing(arg0);
            return null;
        }
    }
    
    public static final class SetMaxFieldSize extends ForestTask<Void> {
        private final Statement stmt;
        private final int arg0;
        public SetMaxFieldSize(final int serverid, final Statement stmt, int arg0) {
            super(serverid);
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setMaxFieldSize(arg0);
            return null;
        }
    }
    
    public static final class SetMaxRows extends ForestTask<Void> {
        private final Statement stmt;
        private final int arg0;
        public SetMaxRows(final int serverid, final Statement stmt, int arg0) {
            super(serverid);
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setMaxRows(arg0);
            return null;
        }
    }
    
    public static final class SetQueryTimeout extends ForestTask<Void> {
        private final Statement stmt;
        private final int arg0;
        public SetQueryTimeout(final int serverid, final Statement stmt, int arg0) {
            super(serverid);
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setQueryTimeout(arg0);
            return null;
        }
    }
    
    public static final class GetFetchDirection extends ForestTask<Integer> {
        private final Statement stmt;
        public GetFetchDirection(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getFetchDirection());
        }
    }
    
    public static final class GetFetchSize extends ForestTask<Integer> {
        private final Statement stmt;
        public GetFetchSize(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getFetchSize());
        }
    }
    
    public static final class GetResultSetConcurrency extends ForestTask<Integer> {
        private final Statement stmt;
        public GetResultSetConcurrency(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getResultSetConcurrency());
        }
    }
    
    public static final class GetResultSetType extends ForestTask<Integer> {
        private final Statement stmt;
        public GetResultSetType(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getResultSetType());
        }
    }
    
    public static final class SetFetchDirection_Int extends ForestTask<Void> {
        private final Statement stmt;
        private final int direction;
        public SetFetchDirection_Int(final int serverid, final Statement stmt, final int direction) {
            super(serverid);
            this.stmt = stmt;
            this.direction = direction;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setFetchDirection(direction);
            return null;
        }
    }
    
    public static final class SetFetchSize_Int extends ForestTask<Void> {
        private final Statement stmt;
        private final int rows;
        public SetFetchSize_Int(final int serverid, final Statement stmt, final int rows) {
            super(serverid);
            this.stmt = stmt;
            this.rows = rows;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setFetchSize(rows);
            return null;
        }
    }
    
    public static final class GetMaxFieldSize extends ForestTask<Integer> {
        private final Statement stmt;
        public GetMaxFieldSize(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getMaxFieldSize());
        }
    }
    
    public static final class GetMaxRows extends ForestTask<Integer> {
        private final Statement stmt;
        public GetMaxRows(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getMaxRows());
        }
    }
    
    public static final class GetQueryTimeout extends ForestTask<Integer> {
        private final Statement stmt;
        public GetQueryTimeout(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getQueryTimeout());
        }
    }
    
    public static final class GetWarnings extends ForestTask<SQLWarning> {
        private final Statement stmt;
        public GetWarnings(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public SQLWarning call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return stmt.getWarnings();
        }
    }
    
    public static final class GetMoreResults extends ForestTask<Boolean> {
        private final Statement stmt;
        public GetMoreResults(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Boolean call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Boolean.valueOf(stmt.getMoreResults());
        }
    }
    
    public static final class GetResultSet extends ForestTask<ResultSet> {
        private final Statement stmt;
        public GetResultSet(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public ResultSet call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return stmt.getResultSet();
        }
    }
    
    public static final class GetUpdateCount extends ForestTask<Integer> {
        private final Statement stmt;
        public GetUpdateCount(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getUpdateCount());
        }
    }
    
    public static final class Close extends ForestTask<Void> {
        private final Statement stmt;
        public Close(final int serverid, final Statement stmt) {
            super(serverid);
            this.stmt = stmt;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.close();
            return null;
        }
    }
    
}
