/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.apibase;

import java.lang.ref.WeakReference;
import java.sql.*;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.*;

import org.postgresforest.ForestConnection;
import org.postgresforest.constant.ConstStr;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.constant.UdbValidity;
import org.postgresforest.exception.*;
import org.postgresforest.mng.*;
import org.postgresforest.util.*;

import net.jcip.annotations.NotThreadSafe;

/**
 * このクラスのインスタンスは、ユーザデータベースごとに唯一となる
 * ResourceManagerのインスタンスと、ユーザが作るForestConnectionの
 * インスタンスとの間を取り持つ役割を果たす。
 * Forestの各JDBCオブジェクトは、このクラスを介さずして直接的に
 * ResourceManagerやMngInfoManager等にアクセスしてはならない。
 * <br>
 * このクラスは、各ForestConnectionが抱えるJDBCエントリポイントのリソース
 * （ForestStatementやForestResultSet等）に共通のメソッドを集約し、それらが
 * 使用するリソースを保持する。また、ResourceManagerへのアクセスを中継し、
 * ForestConnectionごとにローカルキャッシュとして管理する。<br>
 * <br>
 * 本クラスの大半のフィールド・メソッドはスレッドセーフではない。
 * このクラスは各Forestコネクションごとに1つインスタンスを生成して使用する。
 * 各コネクションを複数のスレッドから同時に使用することは（JDBCの仕様上）
 * 許していないことと、このクラスに対してユーザアプリケーションのスレッド
 * 以外がアクセスすることはないため、スレッドセーフ性を担保する必要がない。
 * （但し、MngInfoManager側からリカバリ実行完了の通知が来ることがあり、
 * 　その内容に対してアクセスする箇所に関しては同期化処理を行っている）
 */
@NotThreadSafe public final class EntrypointCommonResource implements RecoveryCompletedListener {
    
    // XXX （リファクタ）ForestConnectionとEntrypointCommonResourceは分けないほうが自然（EntryPointのほうを基底クラスにするなどのほうが設計上適切）
    
    /** ユーザデータベースに接続する際に使用するURL */
    private final ForestUrl forestUrl;
    /** ユーザデータベースに接続する際に使用するユーザ名 */
    private final String udbUser;
    /** ユーザデータベースに接続する際に使用するパスワード */
    private final String udbPass;
    
    private final ResourceManager resourceManager;
    
    /** このEntryPointで実行中のSQL（障害時出力用） */
    private String executeQuery;
    private String getExecuteQuery() {
        return (executeQuery == null) ? "-" : executeQuery;
    }
    public void setExecuteQuery(final String query) {
        executeQuery = query;
    }
    
    /** トランザクションの境界にいるか否か */
    private boolean isTxBorder = true;
    private boolean getIsTxBorder() {
        return isTxBorder;
    }
    /**
     * トランザクションの境界を実行する際にtrueをセットする。
     * trueをセットした場合、直後に実行されるAPIはロールバックが
     * 不可能なものとしてアクセスコントロールが実施される。
     */
    public void setIsTxBorder(boolean isTxBorder) {
        this.isTxBorder = isTxBorder;
    }
    
    /**
     * このコネクションが既に閉じられているか（＝リソース解放処理を呼んだか否か）を表す。
     * この変数への値の代入は、
     * - コンストラクタ中
     * - releaseForestEntrypoint()
     * の2か所になる。しかしながら、releaseForestEntrypoint()は、ForestConnectionの
     * ファイナライザによって呼ばれる可能性もあり、ファイナライザはアプリケーション
     * スレッドとは別であることから、volatile変数として宣言することでメモリの
     * 可視性を担保することとする
     */
    private volatile boolean isClosed = false;
    
    /**
     * このコネクションが既に閉じられているかどうか（＝リソース解放処理が呼ばれたか否か）
     * を表している。この関数を呼び出すべきは、APIのエントリポイントから直接呼ばれる箇所、
     * 具体的には「executeAllApi」「executeAnyApi」「executeOneApi」の先頭で呼び出し、
     * trueになっている場合にはユーザに例外を返す。
     * @return 既にこのリソースが解放されているならtrue、そうでないならfalse
     */
    public boolean getIsClosed() {
        return isClosed;
    }
    
    /** 現在オートコミットモードか否か */
    private boolean isAutocommit = true;
    private boolean getIsAutocommit() {
        return isAutocommit;
    }
    public void setIsAutocommit(boolean autocommit) {
        this.isAutocommit = autocommit;
    }
    
    /** 片系実行時の優先サーバID */
    private final int preferentialExecServerId;
    public int getPreferentialExecServerId() {
        return preferentialExecServerId;
    }
    
    /** このEntryPointに紐づくコネクションが使用するローカルコンフィグのID名 */
    private final String localConfigId;
    
    /**
     * ForestConnectionとリソースマネージャの間を取り持つコネクションローカル
     * リソースクラスのコンストラクタ。<br>
     * <br>
     * コンストラクションの中で、ResourceManager#getResourceManager()を呼び出し、
     * リソースマネージャインスタンスを取り出す。リソースマネージャの取得に
     * 失敗した場合にはコンストラクションに失敗する。<br>
     * <br>
     * <b>このコンストラクションが成功した場合、必ずEntryPointCommonResource#
     * releaseForestEntrypointを呼ばなくてはならない</b>
     * @param forestUrl このForestConnection生成時に指定された接続文字列オブジェクト
     * @param prop コネクション作成の際に渡された全てのオプション
     * （接続文字列中のものと、APIの引数として与えたプロパティを両方合わせたもの）
     * @throws ForestException リソースマネージャの取得に失敗した場合
     */
    public EntrypointCommonResource(final ForestUrl forestUrl, final Properties prop)
    throws ForestException {
        try {
            // TODO （仕様・実装方法検討）管理情報を取得する際のユーザ名・パスワードは初期アクセス時の値で動作しているが、システム全体として一意に決めるべきか
            final String mngdbUser = prop.getProperty("user", "");
            final String mngdbPass = prop.getProperty("password", "");
            this.resourceManager = ResourceManager.getResourceManager(forestUrl, mngdbUser, mngdbPass);
        } catch (ForestInitFailedException e) {
            isClosed = true;
            throw e;
        }
        // プロパティはここで解析して必要な情報を取り出しておく
        this.forestUrl = forestUrl;
        this.udbUser = prop.getProperty("user", "");
        this.udbPass = prop.getProperty("password", "");
        // 接続文字列中（オプション領域）で指定されたconfigidを読み出す（ない場合デフォルト名を使う）
        this.localConfigId = prop.getProperty(
                ConstStr.FOREST_URL_CONFIGID.toString(),
                ConstStr.FOREST_URL_DEFAULT_CONFIGID.toString());
        // 優先実行サーバIDは生成されたコネクションごとに0と1が交互に払いだされる
        this.preferentialExecServerId = resourceManager.getNextPreferentialServerId();
    }
    
    public void init() throws ForestException {
        // MngInfoManagerにリカバリ完了通知
        resourceManager.setRecoveryCompletedListener(this);
        
        // MngInfoをローカルにキャッシュする
        refreshMngInfoCache();
    }
    
    /** 縮退時に解放する対象となるForestConnection */
    private WeakReference<ForestConnection> forestConnection = new WeakReference<ForestConnection>(null);
    /**
     * このEntryPointCommonResourceが管理する対象のForestConnectionを登録する<br>
     * 縮退が必要になった時や、リカバリが必要になった時に、ここで登録したコネクションの
     * 参照を辿ってコネクションの操作を行う。与えられた引数のコネクションは
     * 弱参照として保持するため、解除の処理は必要ない。
     * @param con ForestConnection
     */
    public void setTargetConnection(final ForestConnection con) {
        this.forestConnection = new WeakReference<ForestConnection>(con);
    }
    
    private int apiExecTimeout = 300;
    private boolean isFastApiReturnOnDbFail = false;
    
    /**
     * このコネクション上でのAPI呼び出しをした際のMngInfoのスナップショット<br>
     * （このフィールドの初期化・読み込み・書き込みは、常に同一のクライアント
     * スレッドなので、volatile宣言やsynchronizedによるガードは必要ない）
     */
    private MngInfo currentMngInfo = null;
    /**
     * コネクションローカルに保持している管理情報をリソースマネージャが持つ
     * 最新のものに置き換え、その過程で縮退を発見すれば適切に縮退処理をし、
     * コンフィグレーションなどの更新をローカルに保持する。<br>
     * 基本的にForestJDBCの殆どのAPIでは、各API固有の処理を行う前と行った後に
     * 本関数を呼び出し、管理情報をチェックし、管理情報に合わせてリソースを
     * 解放する。
     * @throws ForestException 管理情報中に指定したLocalConfigが存在しなかった場合
     */
    private void refreshMngInfoCache() throws ForestException {
        final MngInfo newMngInfo = resourceManager.getMngInfo();
        
        // リソースマネージャが持つ新規MngInfoが、キャッシュしたMngInfoと（論理的に）
        // 異なる場合、内容を精査して縮退やコンフィグレーション読み込みの処理を行う
        if (newMngInfo.equals(currentMngInfo) == false) {
            
            // ServerInfoが論理的に変わっているかどうかをチェックする
            // 変わっているなら、縮退があるか否かを判断し、縮退の系については
            // 以降その系のリソースを使われないようにリソース解放処理を行う
            if (newMngInfo.equalServerInfo(currentMngInfo) == false) {
                final List<UdbValidity> validityList = newMngInfo.getValidityList();
                for (int i = 0; i < validityList.size(); i++) {
                    if (validityList.get(i) == UdbValidity.INVALID) {
                        closeOneSide(i);
                    }
                }
            }
            
            // LocalConfigを読み出す。指定されたconfigidのLocalConfigがない場合は例外を投げる
            if (newMngInfo.getLocalConfigMap().containsKey(localConfigId) == false) {
                throw new ForestException(
                        ErrorStr.MNGDB_LOCALCONFIG_NOTFOUND.toString() +
                        " (configid : " + localConfigId + ")" );
            }
            final MngInfo.LocalConfig localConfig = newMngInfo.getLocalConfigMap().get(localConfigId);
            apiExecTimeout = localConfig.apiExecTimeout();
            isFastApiReturnOnDbFail = localConfig.getIsFastApiReturnOnDbFail();
        }
        
        // キャッシュの参照先を更新する
        currentMngInfo = newMngInfo;
    }
    
    /**
     * リカバリが実行された時に通知するための変数。この変数は、スレッド間での
     * メモリ可視性の担保だけのためにAtomicIntegerArrayを使用している。
     */
    private final AtomicIntegerArray recoveryCompleted = new AtomicIntegerArray(new int[] {0, 0});
    
    /**
     * 本関数は、トランザクションの境界時（つまりCOMMITやROLLBACKが呼ばれた後、
     * ないしはAutocommitの場合は常時）に呼び出すことで、以下の処理を行う。<br>
     * 1. リカバリ状態のデータベースがあればその完了を待つ<br>
     * 2. リカバリが実施された系があれば、その系のコネクションを回復する<br>
     * 3. 現存するのStatement/Prepared/Callable等を閉塞する（以降、既存のStatement類は使用できない）<br>
     * 4. リカバリの実施フラグをクリアする<br>
     * (5.) 但し何らかのエラーが発生した場合にはForestConnectionを閉塞してしまう
     * <br>
     * この関数は<br>
     * 1. Statement系のexecute等のAPIの開始直前<br>
     * 2. Statement等をConnectionから生成する直前<br>
     * で呼び出すべき。また、この関数が正常に働くようにするために、トランザクションを
     * 終了状態とした後（つまりCOMMITなどを行った後）には、setIsTxBorderでtrueをセット
     * する必要がある。
     */
    public void checkRecoveryAndRecoverObjects() {
        // トランザクションの切れ目でないならば特に何もする必要はない
        if (getIsTxBorder() == false && getIsAutocommit() == false) {
            return;
        }
        
        // 抑制状態なら抑制解除を待つ
        try {
            resourceManager.waitRecovery();
        } catch (InterruptedException e) {
            // インタラプションがあったなら、呼び出し元にインタラプションを広げ、
            // 関数自体は速やかに終了する（終了処理への応答）
            Thread.currentThread().interrupt();
            return;
        }
        
        // リカバリが完了したか否かのフラグを見て、完了していればその系の
        // リカバリ完了フラグのクリア、コネクションの復活、閉塞するべき
        // リソースの閉塞、を行う
        for (int i = 0; i < 2; i++) {
            // リカバリ完了フラグがたっていれば、その系のコネクションを復活させる
            if (recoveryCompleted.get(i) != 0) {
                // ローカルに持っているリカバリ完了フラグをクリア
                recoveryCompleted.set(i, 0);
                // 現在存在している全てのStatement類を使用不可能にする
                closeAllStmts();
                // コネクション復活処理
                final ForestConnection targetForestCon = forestConnection.get();
                if (targetForestCon != null) {
                    // コネクション作成のための無名タスク（別スレッド実行するためCallableタスクとする）
                    final PgUrl targetUrl = getUdbPgUrlList().get(i);
                    final Callable<Connection> conCreateTask = new Callable<Connection>() {
                        public Connection call() throws Exception {
                            java.sql.Driver driver = new org.postgresql.Driver();
                            final Properties prop = new Properties();
                            prop.setProperty("user", udbUser);
                            prop.setProperty("password", udbPass);
                            return driver.connect(targetUrl.getUrl(), prop);
                        }
                    };
                    final List<Callable<Connection>> taskList = Collections.singletonList(conCreateTask);
                    // コネクション作成・取得を実際に行う部分
                    try {
                        // 何らかの原因で正常に終了しなかった場合には、このコネクションのリカバリは
                        // 失敗したものとしてコネクションを閉塞し、すみやかに戻る
                        // XXX （情報出力）例外を情報出力するべき
                        final List<Future<Connection>> futureList =
                            resourceManager.execJdbcApiTask(taskList, apiExecTimeout);
                        final Connection newConnection = futureList.get(0).get();
                        if (targetForestCon.recovery(i, newConnection) == false) {
                            closeAll();
                            return;
                        }
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        closeAll();
                        return;
                    } catch (ExecutionException e) {
                        closeAll();
                        return;
                    } catch (CancellationException e) {
                        closeAll();
                        return;
                    } catch (SQLException e) {
                        closeAll();
                        return;
                    }
                }
            }
        }
        // トランザクションの切れ目フラグをクリア
        setIsTxBorder(false);
    }
    
    /**
     * コネクションが無効になった場合に呼ぶ関数<br>
     * 具体的には、ForestConnection.closeから本関数は呼ばれる。
     * <b>注：本関数をEntrypointCommonResource内から呼んではならない。
     * 無限ループとなり、StackOverFlowが起きるはず。
     * EntrypointCommonResource内でコネクションを無効にしたい場合、
     * closeAll()を呼ぶこと。</b>
     */
    public void releaseEntryPointCommonResource() {
        if (getIsClosed() == false) {
            resourceManager.releaseResourceManager();
        }
        isClosed = true;
    }
    
    /**
     * ForestConnectionを無効にしたい場合に呼ぶ関数
     */
    private void closeAll() {
        final ForestConnection con = forestConnection.get();
        if (con != null) {
            try { con.close(); } catch (SQLException ignore) { }
        }
    }
    
    private void closeOneSide(int serverId) {
        final ForestConnection con = forestConnection.get();
        if (con != null) {
            con.closeOneSide(serverId);
        }
    }
    
    private void closeAllStmts() {
        final ForestConnection con = forestConnection.get();
        if (con != null) {
            con.closeActiveStatements();
        }
    }
    
    /**
     * 
     * @param <T>
     * @param task
     * @return
     * @throws ForestException
     * @throws SQLException
     */
    public <T> T executeOneApi(final ForestTask<T> task) 
    throws ForestException, SQLException {
        
        final int execServerId = task.getServerId();
        final Future<T> future;
        try {
            if (isFastApiReturnOnDbFail) {
                future = resourceManager.cancellableExecJdbcApiTask(Collections.<ForestTask<T>>singletonList(task), apiExecTimeout).get(0);
            } else {
                future = resourceManager.execJdbcApiTask(Collections.<Callable<T>>singletonList(task), apiExecTimeout).get(0);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ForestException(ErrorStr.THREAD_INTERRUPTED.toString());
        }
        
        // 結果を格納するリスト。あらかじめ要素数を2個にしてnull初期化しておく
        final List<ResultType> resultTypeList = new ArrayList<ResultType>(2);
        final List<T> resultList = new ArrayList<T>(2);
        final List<Exception> exceptionList = new ArrayList<Exception>(2);
        for (int i = 0; i < 2; i++) {
            resultTypeList.add(null);
            resultList.add(null);
            exceptionList.add(null);
        }
        
        // 実行結果の分類と、リストへの格納
        final Pair3<ResultType, T, Exception> result = getResultAndClassify(future);
        final T returnValue = result.getSecond();
        resultTypeList.set(execServerId, result.getFirst());
        resultList.set(execServerId, returnValue);
        exceptionList.set(execServerId, result.getThird());
        
        // 実行すべき系の結果が「正常終了」ならば、アクセスコントロールをせずに結果を返す。
        // この目的は
        // 1. 例外が起きていない場合の高速化（業務アプリの通常状態の高速化）
        // 2. 仮に異常が発生しても、（支障がない限り）継続することでアプリへの影響を最大限抑える
        // を満たすため。
        if (resultTypeList.get(execServerId) == ResultType.SUCCESS) {
            return returnValue;
        }
        
        // 実行すべき系の結果がエラーだったなら、アクセスコントロールのチェックを行う
        final String taskClassName = task.getClass().getName();
        while (true) {
            // 実行途中に別の系で縮退が発見されていないかチェックし、
            refreshMngInfoCache();
            // 縮退となっている系の結果はnullとする
            final List<UdbValidity> validityList = currentMngInfo.getValidityList();
            for(int svId = 0; svId < 2; svId++) {
                if (validityList.get(svId) == UdbValidity.INVALID) {
                    resultTypeList.set(svId, null);
                    resultList.set(svId, null);
                    exceptionList.set(svId, null);
                }
            }
            if (resultTypeList.get(0) != null || resultTypeList.get(1) != null) {
                // 有効な実行結果があるため、アクセスコントロールを行う
                if (accessCtrl_ExecOne(resultTypeList, exceptionList, taskClassName) == true) {
                    return returnValue;
                } else {
                    continue;
                }
            } else {
                // 有効な実行結果がない場合
                closeAll();
                throw new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
            }
        }
    }
    
    /**
     * executeAnyApiと基本的に同じで、内部的にはexecuteAnyApiを呼び出しているが、
     * API実行前に管理情報を参照して縮退のチェックを行っている部分が異なる。<br>
     * 現在の縮退状態を確認した後に実行するべきタイプのAPIを実行する関数であり、
     * 具体的には、コネクション作成時、Statement類の作成時、Statement等の
     * executeに類する関数等が本関数を呼び出すこととなる。<br>
     * 本関数の第3引数は、何も処理を行わずに、タスクが処理対象でないことを示す
     * ForestTaskNotExecutedException例外をスローするタスクとする必要がある。
     * （つまり、タスクのcall関数内で実質的に何も行わないインスタンス）
     * @param <T>
     * @param task0 ServerId=0側のタスク
     * @param task1 ServerId=1側のタスク
     * @param dummy ダミータスク。call()内部で実質的に何も処理しないタスク
     * @return
     * @throws ForestException
     * @throws SQLException
     */
    public <T> List<T> executeAnyApiWithPreCheck(final ForestTask<T> task0, final ForestTask<T> task1, final ForestTask<T> dummy0, final ForestTask<T> dummy1)
    throws ForestException, SQLException {
        refreshMngInfoCache();
        final List<UdbValidity> validityList = currentMngInfo.getValidityList();
        final ForestTask<T> execTask0 = (validityList.get(0) == UdbValidity.INVALID) ? dummy0 : task0;
        final ForestTask<T> execTask1 = (validityList.get(1) == UdbValidity.INVALID) ? dummy1 : task1;
        return executeAnyApi(execTask0, execTask1);
    }
    
    /**
     * 
     * @param <T>
     * @param task0
     * @param task1
     * @return
     * @throws ForestException
     * @throws SQLException
     */
    public <T> List<T> executeAnyApi(final ForestTask<T> task0, final ForestTask<T> task1)
    throws ForestException, SQLException {
        
        if (getIsClosed()) {
            throw new ForestResourceDisposedException(ErrorStr.CONNECTION_CLOSED.toString());
        }
        if (task0 == null && task1 == null) {
            throw new IllegalArgumentException(ErrorStr.ILLEGAL_ARGUMENT.toString());
        }
        
        // タスクのサーバIDとタスクを列挙できるようにリストにする
        final List<ForestTask<T>> taskList = new ArrayList<ForestTask<T>>(2);
        taskList.add(task0);
        taskList.add(task1);
        
        // このコネクションがServerId=1の系を優先実行する場合、リストの順序を逆にする
        if (getPreferentialExecServerId() == 1) {
            Collections.reverse(taskList);
        }
        
        // 結果を格納するリスト。あらかじめ要素数を2個にしてnull初期化しておく
        final List<ResultType> resultTypeList = new ArrayList<ResultType>(2);
        final List<T> resultList = new ArrayList<T>(2);
        final List<Exception> exceptionList = new ArrayList<Exception>(2);
        for (int i = 0; i < 2; i++) {
            resultTypeList.add(null);
            resultList.add(null);
            exceptionList.add(null);
        }
        
        // 優先実行するべきタスクから順にタスクを実行する。
        // 実行結果は上で用意したリストに収める
        // 実行結果が「成功」または「論理エラー」となった段階で、次のタスクの実行は行わない
        for (final ForestTask<T> task : taskList) {
            if (task == null) {
                continue;
            }
            final int currentServerId = task.getServerId();
            final Future<T> future;
            try {
                if (isFastApiReturnOnDbFail) {
                    future = resourceManager.cancellableExecJdbcApiTask(Collections.<ForestTask<T>>singletonList(task), apiExecTimeout).get(0);
                } else {
                    future = resourceManager.execJdbcApiTask(Collections.<Callable<T>>singletonList(task), apiExecTimeout).get(0);
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new ForestException(ErrorStr.THREAD_INTERRUPTED.toString());
            }
            final Pair3<ResultType, T, Exception> result = getResultAndClassify(future);
            final ResultType resType = result.getFirst();
            resultTypeList.set(currentServerId, resType);
            resultList.set(currentServerId, result.getSecond());
            exceptionList.set(currentServerId, result.getThird());
            if (resType == ResultType.SUCCESS || resType == ResultType.NONCRITICAL_ERROR) {
                break;
            }
        }
        
        // 実行結果が、「正常終了」＋「無実行」ならば、アクセスコントロールをせずに結果を返す。
        // この目的は
        // 1. 例外が起きていない場合の高速化（業務アプリの通常状態の高速化）
        // 2. 仮に異常が発生しても、（支障がない限り）継続することでアプリへの影響を最大限抑える
        // を満たすため。
        if ( (resultTypeList.get(0) == ResultType.SUCCESS && resultTypeList.get(1) == null) ||
                resultTypeList.get(1) == ResultType.SUCCESS && resultTypeList.get(0) == null) {
            return resultList;
        }
        
        // 何らかのエラーがあった場合はアクセスコントロールを実施する
        final String taskClassName = task0.getClass().getName();
        while (true) {
            // 実行途中に別の系で縮退が発見されていないかチェックし、
            refreshMngInfoCache();
            // 縮退となっている系の結果はnullとする
            final List<UdbValidity> validityList = currentMngInfo.getValidityList();
            for(int svId = 0; svId < 2; svId++) {
                if (validityList.get(svId) == UdbValidity.INVALID) {
                    resultTypeList.set(svId, null);
                    resultList.set(svId, null);
                    exceptionList.set(svId, null);
                }
            }
            
            // アクセスコントロールを実行する。問題なくアクセスコントロールができたなら結果を返す
            // アクセスコントロールに失敗した場合は再チェック後に再度アクセスコントロールを実行する。
            // アクセスコントロール中に例外が発生した場合、そのまま上に投げつける（つまり何もしない）
            if (resultTypeList.get(0) != null && resultTypeList.get(1) != null) {
                // 有効な実行結果が両系から出ている場合
                if (accessCtrl_ExecAll_Rollbackable(resultTypeList, exceptionList, taskClassName)) {
                    return resultList;
                } else {
                    continue;
                }
            } else if (resultTypeList.get(0) == null && resultTypeList.get(1) == null) {
                // 有効な実行結果がない場合
                closeAll();
                throw new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
            } else {
                // 有効な実行結果が片系のみの場合
                accessCtrl_ExecOne(resultTypeList, exceptionList, taskClassName);
                return resultList;
            }
        }
    }
    
    /**
     * 全系実行で、投げっぱなし（アクセスコントロールを何もしない）のAPI実行用関数。
     * close/cancel系の処理を行う場合のみこの関数を呼び出すべきで、他のAPI実行処理では
     * アクセスコントロールを実施する関数を呼び出すべきである。<br>
     * 1. close処理で本関数を呼び出す理由は、縮退の過程でcloseなどの関数が呼ばれる
     * ことがあり、そのclose処理の中で再度アクセスコントロールが発生すると
     * 状況によっては無限ループに陥ってしまう可能性がある。そのため、close等の処理に
     * 限っては本関数を呼び出すことで、アクセスコントロールを実施せず、例外等を
     * 無視して強制的に実行する。<br>
     * 2. cancel処理は、実行中のStatementをキャンセルするための処理であり、一般的には
     * Statementを実行しているスレッドとは別のスレッドから呼ばれる。
     * Statementを実行中のスレッドは応答を待っており、cancelによって応答が返った場合、
     * その結果を基にして正常・縮退などのアクセスコントロールを行う。また、cancelに
     * よって応答が返らなかった場合には、ForestのAPI実行タイムアウトにひっかかり、
     * やはりアクセスコントロールされることになる。つまり、cancelのAPI自体の結果を
     * アクセスコントロールに使用するのではなく、cancelを行ったことにより間接的に
     * StatementのAPIからどのような結果が返ってくるかによってアクセスコントロールされる
     * ことになるため、例外等を無視してよいことになる。
     * @param <T>
     * @param task0
     * @param task1
     */
    public <T> void executeAllApiWithoutBroken(final ForestTask<T> task0, final ForestTask<T> task1) {
        if (getIsClosed()) {
            return;
        }
        
        // 実行時のエラーなどに対して何も制御を行わない。
        // タスクは投げっぱなしで一切ハンドリングはしない。
        try {
            if (isFastApiReturnOnDbFail) {
                final List<ForestTask<T>> taskList = new ArrayList<ForestTask<T>>(2);
                taskList.add(task0);
                taskList.add(task1);
                resourceManager.cancellableExecJdbcApiTask(taskList, apiExecTimeout);
            } else {
                final List<Callable<T>> taskList = new ArrayList<Callable<T>>(2);
                taskList.add(task0);
                taskList.add(task1);
                resourceManager.execJdbcApiTask(taskList, apiExecTimeout);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }
    
    /**
     * executeAllApiと基本的に同じで、内部的にはexecuteAllApiを呼び出しているが、
     * API実行前に管理情報を参照して縮退のチェックを行っている部分が異なる。<br>
     * 現在の縮退状態を確認した後に実行するべきタイプのAPIを実行する関数であり、
     * 具体的には、コネクション作成時、Statement類の作成時、Statement等の
     * executeに類する関数等が本関数を呼び出すこととなる。<br>
     * 本関数の第3引数は、何も処理を行わずに、タスクが処理対象でないことを示す
     * ForestTaskNotExecutedException例外をスローするタスクとする必要がある。
     * （つまり、タスクのcall関数内で実質的に何も行わないインスタンス）
     * @param <T>
     * @param task0 ServerId=0側のタスク
     * @param task1 ServerId=1側のタスク
     * @param dummy ダミータスク。call()内部で実質的に何も処理しないタスク
     * @return
     * @throws ForestException
     * @throws SQLException
     */
    public <T> List<T> executeAllApiWithPreCheck(final ForestTask<T> task0, final ForestTask<T> task1, final ForestTask<T> dummy0, final ForestTask<T> dummy1)
    throws ForestException, SQLException {
        refreshMngInfoCache();
        final List<UdbValidity> validityList = currentMngInfo.getValidityList();
        final ForestTask<T> execTask0 = (validityList.get(0) == UdbValidity.INVALID) ? dummy0 : task0;
        final ForestTask<T> execTask1 = (validityList.get(1) == UdbValidity.INVALID) ? dummy1 : task1;
        // TODO （コンフィグ）コネクション生成時のタイムアウト設定
        return executeAllApi(execTask0, execTask1);
    }
    
    /**
     * 
     * @param <T> 
     * @param task0 
     * @param task1 
     * @return 両系タスクの戻り値のリスト。実行していない系や縮退となった系の要素はnullとなる。InterruptedExceptionが発生した場合にはnullが返却される
     * @throws ForestResourceDisposedException JDBCエントリポイントが既に閉塞されている場合
     * @throws SQLException 
     * @throws IllegalArgumentException タスクが共にnullの場合
     */
    public <T> List<T> executeAllApi(final ForestTask<T> task0, final ForestTask<T> task1)
    throws ForestException, SQLException {
        
        if (getIsClosed()) {
            throw new ForestResourceDisposedException(ErrorStr.CONNECTION_CLOSED.toString());
        }
        
        // ExecutorService.invokeAllでタスク実行する。
        final List<Future<T>> futureList;
        try {
            if (isFastApiReturnOnDbFail) {
                final List<ForestTask<T>> tasks = new ArrayList<ForestTask<T>>(2);
                tasks.add(task0);
                tasks.add(task1);
                futureList = resourceManager.cancellableExecJdbcApiTask(tasks, apiExecTimeout);
            } else {
                final List<Callable<T>> tasks = new ArrayList<Callable<T>>(2);
                tasks.add(task0);
                tasks.add(task1);
                futureList = resourceManager.execJdbcApiTask(tasks, apiExecTimeout);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ForestException(ErrorStr.THREAD_INTERRUPTED.toString());
        }
        
        // 実行結果を取得して分類する
        final List<T> returnList = new ArrayList<T>(2);
        final List<Exception> exceptionList = new ArrayList<Exception>(2);
        final List<ResultType> resTypeList = new ArrayList<ResultType>(2);
        for (final Future<T> future : futureList) {
            final Pair3<ResultType, T, Exception> res = getResultAndClassify(future);
            resTypeList.add(res.getFirst());
            returnList.add(res.getSecond());
            exceptionList.add(res.getThird());
        }
        
        // 実行結果が、両系共「正常終了」ならば、アクセスコントロールをせずに結果を返す。
        // この目的は
        // 1. 例外が起きていない場合の高速化（業務アプリの通常状態の高速化）
        // 2. 仮に異常が発生しても、（支障がない限り）継続することでアプリへの影響を最大限抑える
        // を満たすため。
        if (resTypeList.get(0) == ResultType.SUCCESS && resTypeList.get(1) == ResultType.SUCCESS) {
            return returnList;
        }
        
        // 何らかのエラーがあった場合はアクセスコントロールを実施する。
        // 縮退処理は場合によって失敗する。これはロックを取らずにアトミックな
        // CompareAndSetを行っているためであり、他のコネクションや外部要因により
        // 先に縮退処理が行われていることを意味している。
        // この場合には再度checkUdbValidityAndReleaseResourceを呼び、このクラスが
        // キャッシュとして保持している管理情報を最新のものにし、その管理情報を基に
        // 再度縮退の処理などを試みる。よって、以下の処理は、成功するまでループとしている。
        final String taskClassName = task0.getClass().getName();
        while (true) {
            // 外部（このコネクション以外）から縮退が起きていないかをチェックし、
            // 縮退が起きていれば、縮退した系のリソースを解放する
            refreshMngInfoCache();
            // 現時点で縮退とわかっている系のResultTypeについてはnullとする。
            // また、returnListやexceptionListについても同様にnullを代入する。
            for(int i = 0; i < currentMngInfo.getValidityList().size(); i++) {
                if (currentMngInfo.getValidityList().get(i) == UdbValidity.INVALID) {
                    //futures.get(i).cancel(true);
                    //futures.set(i, null);
                    resTypeList.set(i, null);
                    returnList.set(i, null);
                    exceptionList.set(i, null);
                }
            }
            
            // resTypeListを基に、結果集約・縮退処理等を実施する。
            // メソッドの成否・エラーの種類に応じて、
            // - Forestコネクションの閉塞の実施可否
            // - 各PostgreSQLリソースの閉塞の実施可否
            // - 各系の縮退の実施可否
            // を決定する
            if (resTypeList.get(0) != null && resTypeList.get(1) != null) {
                // 有効な実行結果が両系で得られた場合
                if (getIsAutocommit() == true || getIsTxBorder() == true) {
                    // オートコミット時か、トランザクションの終了（つまりCOMMIT/ROLLBACK等）の場合は
                    // 復旧不能なアクセスコントロールを実施する
                    if (accessCtrl_ExecAll_UnRollbackable(resTypeList, exceptionList, taskClassName)) {
                        return returnList;
                    }
                } else {
                    // 有効な実行結果が両系から出ている場合
                    if (accessCtrl_ExecAll_Rollbackable(resTypeList, exceptionList, taskClassName)) {
                        return returnList;
                    }
                }
                // アクセスコントロールでtrue以外が返った場合は、縮退に失敗したため
                // 再度管理情報のチェックを行ってアクセスコントロールを試みる
                continue;
            } else if (resTypeList.get(0) == null && resTypeList.get(1) == null) {
                // 有効な実行結果がない場合
                closeAll();
                throw new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
            } else {
                // 有効な実行結果が片系のみの場合
                accessCtrl_ExecOne(resTypeList, exceptionList, taskClassName);
                return returnList;
            }
        }
    }
    
    /**
     * 両系生存時に、ロールバック可能なタイプの両系実行APIを実行した際の
     * 結果を基に、系の縮退やリソースの解放など、アクセスコントロールを行う。<br>
     * <br>
     * 結果リストの内容によってどのようなアクセスコントロールを行うかは、仕様書を参照<br>
     * <br>
     * この関数がtrueで返った場合、ユーザには結果をそのまま返却してよい。
     * この関数がfalseで返った場合、縮退の際に他のスレッドに縮退操作を割り込まれている可能性があるため、
     * 再度checkUdbValidityAndReleaseResourceを呼び出した上でリトライを行う必要がある。
     * この関数が例外をスローした場合、その例外をそのままアプリケーションに返してよい。
     * @param <T>
     * @param resTypeList
     * @param exceptionList
     * @param taskClassName
     * @return 縮退が発生しなかったか縮退が正常に完了した場合true、縮退が他に割り込まれた場合false（呼び出し側で要リトライ）
     * @throws SQLException 片系で論理エラーが発生した場合（このSQLExceptionをユーザに返す）
     * @throws ForestException 両系での例外が発生した場合（このForestExceptionをユーザに返す）
     */
    private <T> boolean accessCtrl_ExecAll_Rollbackable(final List<ResultType> resTypeList, final List<Exception> exceptionList, final String taskClassName)
    throws SQLException, ForestException {
        switch(resTypeList.get(0)) {
            case SUCCESS: {
                switch (resTypeList.get(1)) {
                    case SUCCESS: {
                        return true;
                    }
                    case NONCRITICAL_ERROR: {
                        // 1系正常・2系論理エラー -> 2系論理エラーでロールバックを促す
                        throw (SQLException) exceptionList.get(1);
                    }
                    case CRITICAL_ERROR: {
                        // 1系正常・2系物理エラー -> 2系を縮退させる
                        if (resourceManager.setUdbInvalid(currentMngInfo, 1, exceptionList.get(1), taskClassName, getExecuteQuery())) {
                            closeOneSide(1);
                            return true;
                        }
                    }
                    break;
                }
            }
            break;
            
            case NONCRITICAL_ERROR: {
                switch (resTypeList.get(1)) {
                    case SUCCESS: {
                        // 1系論理エラー・2系成功 -> 1系論理エラーでロールバックを促す
                        throw (SQLException) exceptionList.get(0);
                    }
                    case NONCRITICAL_ERROR: {
                        // 1系論理エラー・2系論理エラー -> 両系のエラーを集約して返却
                        final ForestException exception = new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
                        exception.setMultipleCause(exceptionList);
                        throw exception;
                    }
                    case CRITICAL_ERROR: {
                        // 1系論理エラー・2系物理エラー -> 2系を縮退した上で1系のエラー返却
                        if (resourceManager.setUdbInvalid(currentMngInfo, 1, exceptionList.get(1), taskClassName, getExecuteQuery())) {
                            closeOneSide(1);
                            final ForestException exception = new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
                            exception.setMultipleCause(exceptionList);
                            throw exception;
                        }
                        break;
                    }
                }
            }
            break;
            
            case CRITICAL_ERROR: {
                switch (resTypeList.get(1)) {
                    case SUCCESS: {
                        // 1系物理エラー・2系成功 -> 1系を縮退させる
                        if (resourceManager.setUdbInvalid(currentMngInfo, 0, exceptionList.get(0), taskClassName, getExecuteQuery())) {
                            closeOneSide(0);
                            return true;
                        }
                        break;
                    }
                    case NONCRITICAL_ERROR: {
                        // 1系物理エラー・2系論理エラー -> 1系を縮退した上で2系のエラーを返却
                        if (resourceManager.setUdbInvalid(currentMngInfo, 0, exceptionList.get(0), taskClassName, getExecuteQuery())) {
                            closeOneSide(0);
                            final ForestException exception = new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
                            exception.setMultipleCause(exceptionList);
                            throw exception;
                        }
                        break;
                    }
                    case CRITICAL_ERROR: {
                        // 1系物理エラー・2系物理エラー -> このForestコネクションを閉塞し、エラーを返却
                        closeAll();
                        final ForestException exception = new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
                        exception.setMultipleCause(exceptionList);
                        throw exception;
                    }
                }
            }
            break;
            
        }
        return false;
    }
    
    /**
     * 両系生存時に、ロールバック不可能なタイプの両系実行APIを実行した際の
     * 結果を基に、系の縮退やリソースの解放など、アクセスコントロールを行う。<br>
     * <br>
     * 結果リストの内容によってどのようなアクセスコントロールを行うかは、仕様書を参照<br>
     * <br>
     * この関数がtrueで返った場合、ユーザには結果をそのまま返却してよい。
     * この関数がfalseで返った場合、縮退の際に他のスレッドに縮退操作を割り込まれている可能性があるため、
     * 再度checkUdbValidityAndReleaseResourceを呼び出した上でリトライを行う必要がある。
     * この関数が例外をスローした場合、その例外をそのままアプリケーションに返してよい。
     * @param <T>
     * @param resTypeList
     * @param exceptionList
     * @param taskClassName
     * @return 縮退が発生しなかったか縮退が正常に完了した場合true、縮退が他に割り込まれた場合false（呼び出し側で要リトライ）
     * @throws SQLException 片系で論理エラーが発生した場合（このSQLExceptionをユーザに返す）
     * @throws ForestException 両系での例外が発生した場合（このForestExceptionをユーザに返す）
     */
    private <T> boolean accessCtrl_ExecAll_UnRollbackable(final List<ResultType> resTypeList, final List<Exception> exceptionList, final String taskClassName)
    throws SQLException, ForestException {
        switch(resTypeList.get(0)) {
            case SUCCESS: {
                switch (resTypeList.get(1)) {
                    case SUCCESS: {
                        return true;
                    }
                    case NONCRITICAL_ERROR:
                    case CRITICAL_ERROR: {
                        // 1系正常・2系論理エラー or 2系縮退対象 -> 1系正常を返し、2系は縮退
                        if (resourceManager.setUdbInvalid(currentMngInfo, 1, exceptionList.get(1), taskClassName, getExecuteQuery())) {
                            closeOneSide(1);
                            return true;
                        }
                        break;
                    }
                }
            }
            break;
            
            case NONCRITICAL_ERROR: {
                switch (resTypeList.get(1)) {
                    case SUCCESS: {
                        // 1系論理エラー・2系成功 -> 2系の正常終了を返し、1系は縮退とする
                        if (resourceManager.setUdbInvalid(currentMngInfo, 0, exceptionList.get(0), taskClassName, getExecuteQuery())) {
                            closeOneSide(0);
                            return true;
                        }
                        break;
                    }
                    case NONCRITICAL_ERROR: {
                        // 1系論理エラー・2系論理エラー -> 両系のエラーを集約して返却
                        final ForestException exception = new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
                        exception.setMultipleCause(exceptionList);
                        throw exception;
                    }
                    case CRITICAL_ERROR: {
                        // 1系論理エラー・2系物理エラー -> 2系を縮退した上で1系のエラー返却
                        if (resourceManager.setUdbInvalid(currentMngInfo, 1, exceptionList.get(1), taskClassName, getExecuteQuery())) {
                            closeOneSide(1);
                            if (exceptionList.get(0) instanceof SQLException) {
                                throw (SQLException) exceptionList.get(0);
                            } else {
                                final ForestException e = new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
                                e.setMultipleCause(exceptionList);
                                throw e;
                            }
                        }
                        break;
                    }
                }
            }
            break;
            
            case CRITICAL_ERROR: {
                switch (resTypeList.get(1)) {
                    case SUCCESS: {
                        // 1系物理エラー・2系成功 -> 1系を縮退させ、2系の結果を返す
                        if (resourceManager.setUdbInvalid(currentMngInfo, 0, exceptionList.get(0), taskClassName, getExecuteQuery())) {
                            closeOneSide(0);
                            return true;
                        }
                        break; 
                    }
                    case NONCRITICAL_ERROR: {
                        // 1系物理エラー・2系論理エラー -> 1系を縮退した上で2系のエラーを返却
                        if (resourceManager.setUdbInvalid(currentMngInfo, 0, exceptionList.get(0), taskClassName, getExecuteQuery())) {
                            closeOneSide(0);
                            throw (SQLException) exceptionList.get(1);
                        }
                        break;
                    }
                    case CRITICAL_ERROR: {
                        // 1系物理エラー・2系物理エラー -> このForestコネクションを閉塞し、エラーを返却
                        closeAll();
                        final ForestException exception = new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
                        exception.setMultipleCause(exceptionList);
                        throw exception;
                    }
                }
            }
            break;
            
        }
        return false;
    }
    
    /**
     * 
     * @param <T>
     * @param resTypeList
     * @param exceptionList
     * @param taskClassName
     * @throws SQLException
     * @throws ForestException
     */
    private <T> boolean accessCtrl_ExecOne(final List<ResultType> resTypeList, final List<Exception> exceptionList, final String taskClassName)
    throws SQLException, ForestException {
        final int execServerId = (resTypeList.get(0) != null) ? 0 : 1;
        switch (resTypeList.get(execServerId)) {
            case SUCCESS: {
                // 実行結果をそのまま返却
                return true;
            }
            
            case NONCRITICAL_ERROR: {
                // 実行した系で起きたSQLExceptionを返却
                throw (SQLException) exceptionList.get(execServerId);
            }
            
            case CRITICAL_ERROR: {
                // 現在の管理情報中に縮退があるかどうか調べる。これは、既に縮退がある状況での
                // 片系実行なのか、縮退がない状態での片系実行なのかによって、その後縮退させるか
                // 否かが異なるため。両方正常状態で縮退対象エラーが出たなら縮退をさせる。
                // 片系縮退状態で縮退対象エラーが出たなら、縮退をさせない
                int faultCount = 0;
                for (final UdbValidity validity : currentMngInfo.getValidityList()) {
                    if (validity == UdbValidity.INVALID) {
                        faultCount++;
                    }
                }
                switch (faultCount) {
                    case 0:
                        // 両系で動作している場合、その系を縮退させる
                        return resourceManager.setUdbInvalid(currentMngInfo, execServerId, exceptionList.get(execServerId), taskClassName, getExecuteQuery());
                        
                    case 1:
                        // 片系のみの動作している場合は、このForestコネクションを閉塞、
                        // リソースは解放した上で、エラーを返し、縮退しない。
                        closeAll();
                        final Exception targetException = exceptionList.get(execServerId);
                        if (targetException instanceof SQLException) {
                            throw (SQLException) targetException;
                        } else {
                            final ForestException exception = new ForestException(ErrorStr.NOEFFECTIVE_RESULT.toString());
                            exception.setMultipleCause(exceptionList);
                            throw exception;
                        }
                    default:
                        throw new IllegalStateException(ErrorStr.ILLEGAL_STATEMENT.toString());
                }
            }
            
            default:
                throw new IllegalArgumentException(ErrorStr.ILLEGAL_ARGUMENT.toString()); 
        }
    }
    
    /**
     * 与えられたFutureを基に、結果セット（結果や例外）と結果の分類を作成する。
     * <b>引数として与えるFutureは、ExecutorService.invokeAllにより、実行時間を指定
     * したものでなくてはならない。</b>
     * （これは本関数内でFuture.getを呼ぶ際、タイムアウト時間は指定していないため）
     * また、与えられたfutureがnullの場合や、実行したタスクがダミータスクの場合、
     * すべての値をnullとしてPair3オブジェクトを構築して返却する
     * @param <T>
     * @param future ExecutorService.invokeAllで時間制限を設けて実行した結果のFuture。nullを許す
     * @return 引数のFutureに応じた実行結果。引数がnullだった場合、全ての要素がnull
     * @throws ForestException
     */
    private <T> Pair3<ResultType, T, Exception> getResultAndClassify(Future<T> future) 
    throws ForestException {
        
        if (future == null) {
            // futureそのものがnullの場合、そもそもこのfutureはダミーであり、
            // その系で何も実行されていないことを意味している
            return new Pair3<ResultType, T, Exception>(null, null, null);
        }
        try {
            return new Pair3<ResultType, T, Exception>(ResultType.SUCCESS, future.get(), null);
        } catch (CancellationException e) {
            // invokeAllで実行時間が長くタイムアウトした場合、縮退対象エラーとして処理を行う
            return new Pair3<ResultType, T, Exception>(ResultType.CRITICAL_ERROR, null, e);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ForestException(ErrorStr.THREAD_INTERRUPTED.toString());
        } catch (ExecutionException e) {
            // invokeAllの実行中に例外が出た場合
            if (e.getCause() instanceof Error) {
                throw (Error) e.getCause();
            }
            if (e.getCause() instanceof ForestTaskNotExecutedException){
                // Taskが実行不可だったことを示す例外の場合、これはそもそもそのタスクの系が
                // 縮退状態である場合なので、全ての値をnullで返す
                return new Pair3<ResultType, T, Exception>(null, null, null);
            } else if (e.getCause() instanceof SQLException) {
                // SqlStateの文字列がGlobalConfig中のbroken_error_prefixで始まっているなら縮退対象
                final String sqlState = ((SQLException) e.getCause()).getSQLState();
                final String brokenPrefix = currentMngInfo.getGlobalConfig().getBrokenErrorPrefix();
                for (final String prefix : brokenPrefix.split(",")) {
                    if (sqlState != null && sqlState.startsWith(prefix)) {
                        return new Pair3<ResultType, T, Exception>(ResultType.CRITICAL_ERROR, null, (Exception) e.getCause());
                    }
                }
                return new Pair3<ResultType, T, Exception>(ResultType.NONCRITICAL_ERROR, null, (Exception) e.getCause());
            } else {
                // その他の例外の場合、プログラム的なエラーなどの可能性
                return new Pair3<ResultType, T, Exception>(ResultType.CRITICAL_ERROR, null, (Exception) e.getCause());
            }
        }
    }
    
    /**
     * 実行結果の分類
     */
    public static enum ResultType {
        SUCCESS,
        NONCRITICAL_ERROR,
        CRITICAL_ERROR;
    }
    
    /**
     * 両ユーザデータベースへの接続先を表すPgUrlを返却する。
     * このPgUrlのリストは、管理情報中のServerIdをインデックスとして
     * 返却されることが保障されている。
     */
    public List<PgUrl> getUdbPgUrlList() {
        final List<PgUrl> returnList = new ArrayList<PgUrl>(2);
        for (final MngInfo.ServerInfo svinfo : currentMngInfo.getServerInfoList()) {
            returnList.add(new PgUrl(svinfo.getIpPort(), svinfo.getDbName()));
        }
        return returnList;
    }
    
    /**
     * ユーザデータベースに接続する際に使用されたURLを返却する。
     * @return
     */
    public String getForestUrl() {
        return forestUrl.getUrl();
    }
    
    /**
     * リカバリが完了した際に呼び出される関数<br>
     * 本関数はRecoveryCompletedListenerインターフェースの実装であり、リカバリが
     * 完了した際にMngInfoManager側から呼び出されるいわばコールバック関数である。
     */
    public void setRecoveryCompleted(int serverId) {
        recoveryCompleted.set(serverId, 1);
    }
    
    /**
     * 現在のMngInfoを外部に公開するための関数<br>
     * 具体的には、ForestJdbcInfoインターフェースの実装のForestDatabaseMetaDataの
     * 中で呼びだす目的で作成したものであり、これ以外の用途で使用するのは望ましくない。
     * @return 呼び出し時点でのオンメモリの管理情報
     * @throws ForestResourceDisposedException 既にこのコネクションが閉じられている場合
     */
    public MngInfo getCurrentMngInfo() throws ForestResourceDisposedException {
        if (getIsClosed() == true) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        return resourceManager.getMngInfo();
    }
}
