/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.apibase;

import java.sql.*;
import java.util.Map;

import org.postgresforest.exception.ForestTaskNotExecutedException;
import org.postgresforest.util.ForestTask;

/**
 * PostgreSQLのJDBCドライバに対してConnectionクラスのAPIを実行するための
 * Callableタスクのインプリを集約したクラス。このクラス自体は、メンバや
 * メソッドを持たない（内部クラスを持つのみ）
 * 
 * 各タスクは、コンストラクタで与えられたJDBCリソースについて、nullか否かを
 * 必ずチェックする必要がある。nullの場合、その系は縮退していることを意味し、
 * call内で即座に「ForestTaskNotExecutedException」をスローする。
 */
public final class ConnectionTask {
    
    public static final class Close extends ForestTask<Void> {
        private final Connection con;
        public Close(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.close();
            return null;
        }
        
    }
    
    public static final class Commit extends ForestTask<Void> {
        private final Connection con;
        public Commit(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.commit();
            return null;
        }
        
    }
    
    public static final class Rollback extends ForestTask<Void> {
        private final Connection con;
        public Rollback(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.rollback();
            return null;
        }
        
    }
    
    public static final class SetAutoCommit extends ForestTask<Void> {
        private final Connection con;
        private final boolean autocommit;
        public SetAutoCommit(final int serverid, final Connection con, final boolean autocommit) {
            super(serverid);
            this.con = con;
            this.autocommit = autocommit;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setAutoCommit(autocommit);
            return null;
        }
        
    }
    
    public static final class ClearWarnings extends ForestTask<Void> {
        private final Connection con;
        public ClearWarnings(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.clearWarnings();
            return null;
        }
        
    }
    
    public static final class SetCatalog extends ForestTask<Void> {
        private final Connection con;
        private final String catalog;
        public SetCatalog(final int serverid, final Connection con, final String catalog) {
            super(serverid);
            this.con = con;
            this.catalog = catalog;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setCatalog(catalog);
            return null;
        }
        
    }
    
    public static final class SetReadOnly extends ForestTask<Void> {
        private final Connection con;
        private final boolean readOnly;
        public SetReadOnly(final int serverid, final Connection con, final boolean readOnly) {
            super(serverid);
            this.con = con;
            this.readOnly = readOnly;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setReadOnly(readOnly);
            return null;
        }
        
    }
    
    public static final class SetTransactionIsolation extends ForestTask<Void> {
        private final Connection con;
        private final int level;
        public SetTransactionIsolation(final int serverid, final Connection con, final int level) {
            super(serverid);
            this.con = con;
            this.level = level;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setTransactionIsolation(level);
            return null;
        }
    }
    
    public static final class SetTypeMap_Map extends ForestTask<Void> {
        private final Connection con;
        private final Map<String, Class<?>> map;
        public SetTypeMap_Map(final int serverid, final Connection con, final Map<String, Class<?>> map) {
            super(serverid);
            this.con = con;
            this.map = map;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setTypeMap(map);
            return null;
        }
    }
    
    public static final class GetAutoCommit extends ForestTask<Boolean> {
        private final Connection con;
        public GetAutoCommit(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Boolean call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Boolean.valueOf(con.getAutoCommit());
        }
    }
    
    public static final class GetCatalog extends ForestTask<String> {
        private final Connection con;
        public GetCatalog(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public String call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.getCatalog();
        }
    }
    
    public static final class GetTransactionIsolation extends ForestTask<Integer> {
        private final Connection con;
        public GetTransactionIsolation(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Integer call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(con.getTransactionIsolation());
        }
    }
    
    public static final class GetTypeMap extends ForestTask<Map<String,Class<?>>> {
        private final Connection con;
        public GetTypeMap(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Map<String,Class<?>> call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.getTypeMap();
        }
    }
    
    public static final class PrepareCall_StringIntInt extends ForestTask<CallableStatement> {
        private final Connection con;
        private final String sql;
        private final int resultSetType;
        private final int resultSetConcurrency;
        public PrepareCall_StringIntInt(final int serverid, final Connection con, final String sql, final int resultSetType, final int resultSetConcurrency) {
            super(serverid);
            this.con = con;
            this.sql = sql;
            this.resultSetType = resultSetType;
            this.resultSetConcurrency = resultSetConcurrency;
        }
        public CallableStatement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.prepareCall(sql, resultSetType, resultSetConcurrency);
        }
    }
    
    public static final class GetWarnings extends ForestTask<SQLWarning> {
        private final Connection con;
        public GetWarnings(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public SQLWarning call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.getWarnings();
        }
    }
    
    public static final class IsReadOnly extends ForestTask<Boolean> {
        private final Connection con;
        public IsReadOnly(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Boolean call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.isReadOnly();
        }
    }
    
    public static final class CreateStatement extends ForestTask<Statement> {
        private final Connection con;
        public CreateStatement(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public Statement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.createStatement();
        }
    }
    
    public static final class CreateStatement_IntInt extends ForestTask<Statement> {
        private final Connection con;
        private final int resultSetType;
        private final int resultSetConcurrency;
        public CreateStatement_IntInt(final int serverid, final Connection con, final int resultSetType, final int resultSetConcurrency) {
            super(serverid);
            this.con = con;
            this.resultSetType = resultSetType;
            this.resultSetConcurrency = resultSetConcurrency;
        }
        public Statement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.createStatement(resultSetType, resultSetConcurrency);
        }
    }
    
    public static final class NativeSQL_String extends ForestTask<String> {
        private final Connection con;
        private final String sql;
        public NativeSQL_String(final int serverid, final Connection con, final String sql) {
            super(serverid);
            this.con = con;
            this.sql = sql;
        }
        public String call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.nativeSQL(sql);
        }
    }
    
    public static final class PrepareCall extends ForestTask<CallableStatement> {
        private final Connection con;
        private final String sql;
        public PrepareCall(final int serverid, final Connection con, final String sql) {
            super(serverid);
            this.con = con;
            this.sql = sql;
        }
        public CallableStatement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.prepareCall(sql);
        }
    }
    
    public static final class PrepareStatement extends ForestTask<PreparedStatement> {
        private final Connection con;
        private final String sql;
        public PrepareStatement(final int serverid, final Connection con, final String sql) {
            super(serverid);
            this.con = con;
            this.sql = sql;
        }
        public PreparedStatement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.prepareStatement(sql);
        }
    }
    
    public static final class PrepareStatement_IntInt extends ForestTask<PreparedStatement> {
        private final Connection con;
        private final String sql;
        private final int resultSetType;
        private final int resultSetConcurrency;
        public PrepareStatement_IntInt(final int serverid, final Connection con, final String sql, final int resultSetType, final int resultSetConcurrency) {
            super(serverid);
            this.con = con;
            this.sql = sql;
            this.resultSetType = resultSetType;
            this.resultSetConcurrency = resultSetConcurrency;
        }
        public PreparedStatement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.prepareStatement(sql, resultSetType, resultSetConcurrency);
        }
    }
    
    public static final class GetMetaData extends ForestTask<DatabaseMetaData> {
        private final Connection con;
        public GetMetaData(final int serverid, final Connection con) {
            super(serverid);
            this.con = con;
        }
        public DatabaseMetaData call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.getMetaData();
        }
    }
    
}
