/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */

package org.postgresforest.apibase;

import java.io.InputStream;
import java.io.Reader;
import java.sql.Clob;

import org.postgresforest.exception.ForestTaskNotExecutedException;
import org.postgresforest.util.ForestTask;

/**
 * PostgreSQLのJDBCドライバに対してClobクラスのAPIを実行するための
 * Callableタスクのインプリを集約したクラス。このクラス自体は、メンバや
 * メソッドを持たない（内部クラスを持つのみ）
 * 
 * 各タスクは、コンストラクタで与えられたJDBCリソースについて、nullか否かを
 * 必ずチェックする必要がある。nullの場合、その系は縮退していることを意味し、
 * call内で即座に「ForestTaskNotExecutedException」をスローする。
 */
public class ClobTask {
    
    public static final class GetAsciiStream extends ForestTask<InputStream> {
        private final Clob clob;
        public GetAsciiStream(int serverid, Clob clob) {
            super(serverid);
            this.clob = clob;
        }
        public InputStream call() throws Exception {
            if (clob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return clob.getAsciiStream();
        }
    }
    
    public static final class GetCharacterStream extends ForestTask<Reader> {
        private final Clob clob;
        public GetCharacterStream(int serverid, Clob clob) {
            super(serverid);
            this.clob = clob;
        }
        public Reader call() throws Exception {
            if (clob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return clob.getCharacterStream();
        }
    }
    
    public static final class GetSubString_LongInt extends ForestTask<String> {
        private final Clob clob;
        private final long pos;
        private final int length;
        public GetSubString_LongInt(int serverid, Clob clob, long pos, int length) {
            super(serverid);
            this.clob = clob;
            this.pos = pos;
            this.length = length;
        }
        public String call() throws Exception {
            if (clob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return clob.getSubString(pos, length);
        }
    }
    
    public static final class Length extends ForestTask<Long> {
        private final Clob clob;
        public Length(int serverid, Clob clob) {
            super(serverid);
            this.clob = clob;
        }
        public Long call() throws Exception {
            if (clob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Long.valueOf(clob.length());
        }
    }
    
    public static final class Position_StringLong extends ForestTask<Long> {
        private final Clob clob;
        private final String searchstr;
        private final long start;
        public Position_StringLong(int serverid, Clob clob, String searchstr, long start) {
            super(serverid);
            this.clob = clob;
            this.searchstr = searchstr;
            this.start = start;
        }
        public Long call() throws Exception {
            if (clob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Long.valueOf(clob.position(searchstr, start));
        }
    }
    
    public static final class Position_ClobLong extends ForestTask<Long> {
        private final Clob clob;
        private final Clob searchstr;
        private final long start;
        public Position_ClobLong(int serverid, Clob clob, Clob searchstr, long start) {
            super(serverid);
            this.clob = clob;
            this.searchstr = searchstr;
            this.start = start;
        }
        public Long call() throws Exception {
            if (clob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Long.valueOf(clob.position(searchstr, start));
        }
    }
    
}
