/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */

package org.postgresforest.apibase;

import java.io.InputStream;
import java.sql.Blob;

import org.postgresforest.exception.ForestTaskNotExecutedException;
import org.postgresforest.util.ForestTask;

/**
 * PostgreSQLのJDBCドライバに対してBlobクラスのAPIを実行するための
 * Callableタスクのインプリを集約したクラス。このクラス自体は、メンバや
 * メソッドを持たない（内部クラスを持つのみ）
 * 
 * 各タスクは、コンストラクタで与えられたJDBCリソースについて、nullか否かを
 * 必ずチェックする必要がある。nullの場合、その系は縮退していることを意味し、
 * call内で即座に「ForestTaskNotExecutedException」をスローする。
 */
public class BlobTask {
    
    public static final class GetBinaryStream extends ForestTask<InputStream> {
        private final Blob blob;
        public GetBinaryStream(int serverid, Blob blob) {
            super(serverid);
            this.blob = blob;
        }
        public InputStream call() throws Exception {
            if (blob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return blob.getBinaryStream();
        }
    }
    
    public static final class GetBytes extends ForestTask<byte[]> {
        private final Blob blob;
        private final long pos;
        private final int length;
        public GetBytes(int serverid, Blob blob, long pos, int length) {
            super(serverid);
            this.blob = blob;
            this.pos = pos;
            this.length = length;
        }
        public byte[] call() throws Exception {
            if (blob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return blob.getBytes(pos, length);
        }
    }
    
    public static final class Length extends ForestTask<Long> {
        private final Blob blob;
        public Length(int serverid, Blob blob) {
            super(serverid);
            this.blob = blob;
        }
        public Long call() throws Exception {
            if (blob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Long.valueOf(blob.length());
        }
    }
    
    public static final class Position_BlobLong extends ForestTask<Long> {
        private final Blob blob;
        private final Blob pattern;
        private final long start;
        public Position_BlobLong(int serverid, Blob blob, Blob pattern, long start) {
            super(serverid);
            this.blob = blob;
            this.pattern = pattern;
            this.start = start;
        }
        public Long call() throws Exception {
            if (blob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Long.valueOf(blob.position(pattern, start));
        }
    }
    
    public static final class Position_BytesLong extends ForestTask<Long> {
        private final Blob blob;
        private final byte[] pattern;
        private final long start;
        public Position_BytesLong(int serverid, Blob blob, byte[] pattern, long start) {
            super(serverid);
            this.blob = blob;
            this.pattern = pattern;
            this.start = start;
        }
        public Long call() throws Exception {
            if (blob == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Long.valueOf(blob.position(pattern, start));
        }
    }
    
}
