/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.apibase;

import java.sql.Array;
import java.sql.ResultSet;
import java.util.Map;

import org.postgresforest.exception.ForestTaskNotExecutedException;
import org.postgresforest.util.ForestTask;

/**
 * PostgreSQLのJDBCドライバに対してArrayクラスのAPIを実行するための
 * Callableタスクのインプリを集約したクラス。このクラス自体は、メンバや
 * メソッドを持たない（内部クラスを持つのみ）
 * 
 * 各タスクは、コンストラクタで与えられたJDBCリソースについて、nullか否かを
 * 必ずチェックする必要がある。nullの場合、その系は縮退していることを意味し、
 * call内で即座に「ForestTaskNotExecutedException」をスローする。
 */
public final class ArrayTask {
    
    public static final class GetArray extends ForestTask<Object> {
        private final Array array;
        public GetArray(int serverid, Array array) {
            super(serverid);
            this.array = array;
        }
        @SuppressWarnings("deprecation")
        public Object call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getArray();
        }
    }
    
    public static final class GetArray_Map extends ForestTask<Object> {
        private final Array array;
        private final Map<String, Class<?>> arg0;
        public GetArray_Map(int serverid, Array array, Map<String, Class<?>> arg0) {
            super(serverid);
            this.array = array;
            this.arg0 = arg0;
        }
        public Object call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getArray(arg0);
        }
    }
    
    public static final class GetArray_LongInt extends ForestTask<Object> {
        private final Array array;
        private final long index;
        private final int count;
        public GetArray_LongInt(int serverid, Array array, long index, int count) {
            super(serverid);
            this.array = array;
            this.index = index;
            this.count = count;
        }
        public Object call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getArray(index, count);
        }
    }
    
    public static final class GetArray_LongIntMap extends ForestTask<Object> {
        private final Array array;
        private final long index;
        private final int count;
        private final Map<String, Class<?>> map;
        public GetArray_LongIntMap(int serverid, Array array, long index, int count, Map<String, Class<?>> map) {
            super(serverid);
            this.array = array;
            this.index = index;
            this.count = count;
            this.map = map;
        }
        public Object call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getArray(index, count, map);
        }
    }
    
    public static final class GetBaseType extends ForestTask<Integer> {
        private final Array array;
        public GetBaseType(int serverid, Array array) {
            super(serverid);
            this.array = array;
        }
        public Integer call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(array.getBaseType());
        }
    }
    
    public static final class GetBaseTypeName extends ForestTask<String> {
        private final Array array;
        public GetBaseTypeName(int serverid, Array array) {
            super(serverid);
            this.array = array;
        }
        public String call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getBaseTypeName();
        }
    }
    
    public static final class GetResultSet extends ForestTask<ResultSet> {
        private final Array array;
        public GetResultSet(int serverid, Array array) {
            super(serverid);
            this.array = array;
        }
        public ResultSet call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getResultSet();
        }
    }
    
    public static final class GetResultSet_Map extends ForestTask<ResultSet> {
        private final Array array;
        @SuppressWarnings("unchecked")
        private final Map map;
        public GetResultSet_Map(int serverid, Array array, Map<String, Class<?>> map) {
            super(serverid);
            this.array = array;
            this.map = map;
        }
        @SuppressWarnings("unchecked")
        public ResultSet call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getResultSet(map);
        }
    }
    
    public static final class GetResultSet_LongInt extends ForestTask<ResultSet> {
        private final Array array;
        private final long index;
        private final int count;
        public GetResultSet_LongInt(int serverid, Array array, long index, int count) {
            super(serverid);
            this.array = array;
            this.index = index;
            this.count = count;
        }
        public ResultSet call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getResultSet(index, count);
        }
    }
    
    public static final class GetResultSet_LongIntMap extends ForestTask<ResultSet> {
        private final Array array;
        private final long index;
        private final int count;
        private final Map<String, Class<?>> map;
        public GetResultSet_LongIntMap(int serverid, Array array, long index, int count, Map<String, Class<?>> map) {
            super(serverid);
            this.array = array;
            this.index = index;
            this.count = count;
            this.map = map;
        }
        public ResultSet call() throws Exception {
            if (array == null) {
                throw new ForestTaskNotExecutedException();
            }
            return array.getResultSet(index, count, map);
        }
    }
    
}
