/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest;

import java.lang.ref.WeakReference;
import java.sql.*;
import java.util.*;
import java.util.concurrent.*;
import java.util.regex.Pattern;

import org.postgresforest.apibase.*;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.ForestResourceDisposedException;
import org.postgresforest.util.*;

import net.jcip.annotations.*;

@NotThreadSafe public class ForestStatement implements Statement, ForestCloseable {
    
    protected final EntrypointCommonResource epCommonResource;
    
    /**
     * このForestStatementに紐づく各系へのStatement。
     * Statement#cancel()関数は単一スレッド以外からアクセスされる可能性が高いため
     * この変数はスレッドセーフ（特にスレッド間のメモリ可視性）を担保するために
     * CopyOnWriteArrayListを使用する
     */
    protected final CopyOnWriteArrayList<Statement> stmts;
    private ForestResultSet currentForestResultSet = null;
    protected void setForestResultSet(ForestResultSet newForestResultSet) {
        if (currentForestResultSet != null) {
            try { currentForestResultSet.close(); } catch (SQLException e) { }
        }
        currentForestResultSet = newForestResultSet;
    }
    private boolean isClosed = false;
    /** 
     * このオブジェクトが閉じられていれば例外を投げる 
     * @throws ForestResourceDisposedException このForestオブジェクトが閉じられている場合
     */
    protected void checkObjectClosed() throws ForestResourceDisposedException {
        if (isClosed == true) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
    }
    
    /**
     * このオブジェクトが実行されていなければ例外を投げる
     * @throws SQLException このオブジェクトが実行されていない場合
     */
    protected void checkObjectExecuted() throws SQLException {
        if (getFixExecuteId() == -1) {
            throw new SQLException(org.postgresforest.constant.ErrorStr.CALLED_BEFORE_FETCH.toString());
        }
    }
    
    private final WeakReference<ForestConnection> con;
    
    /**
     * 系固定で実行するタイプのAPIを実行するサーバID。0/1を取る。
     * （-1の場合は結果取得の準備が整っていない、つまり、execute系を実行されていない）
     */
    private int fixExecuteId = -1;
    protected int getFixExecuteId() {
        return fixExecuteId;
    }
    protected void setFixExecuteId(int fixExecuteId) {
        this.fixExecuteId = fixExecuteId;
    }
    
    public ForestStatement(final EntrypointCommonResource epCommonResource, final List<? extends Statement> stmts, final WeakReference<ForestConnection> con) {
        this.epCommonResource = epCommonResource;
        // Statement（あるいはその派生クラス）は、cancel関数により並行アクセスされる
        // 可能性を含んでいる。そのため、CopyOnWriteArrayListに格納することで、
        // スレッド間のメモリ可視性を担保する
        this.stmts = new CopyOnWriteArrayList<Statement>(stmts);
        this.con = con;
    }
    
    public void addBatch(String sql) throws SQLException {
        checkObjectClosed();
        // 全系実行
        final ForestTask<Void> task0 = new StatementTask.AddBatch_String(0, stmts.get(0), sql);
        final ForestTask<Void> task1 = new StatementTask.AddBatch_String(1, stmts.get(1), sql);
        epCommonResource.executeAllApi(task0, task1);
    }
    
    public void cancel() throws SQLException {
        // リソース全体がまだあるか否かのチェックは行う。これは、実行中のSQLが
        // あるにも関わらずStatementのキャンセルが呼ばれずにcloseされた場合など
        // closeが最後まで行ききれない場合などに備えるため。
        // closeが最後まで行ききれていない場合は、epCommonResource中のexecuteは
        // まだ実行可能なので、epCommonResourceに実行するか否かの判断を任せる
        final ForestTask<Void> task0 = new StatementTask.Cancel(0, stmts.get(0));
        final ForestTask<Void> task1 = new StatementTask.Cancel(1, stmts.get(1));
        epCommonResource.executeAllApiWithoutBroken(task0, task1);
    }
    
    public void clearBatch() throws SQLException {
        checkObjectClosed();
        // 全系実行
        final ForestTask<Void> task0 = new StatementTask.ClearBatch_String(0, stmts.get(0));
        final ForestTask<Void> task1 = new StatementTask.ClearBatch_String(1, stmts.get(1));
        epCommonResource.executeAllApi(task0, task1);
    }
    
    public void clearWarnings() throws SQLException {
        checkObjectClosed();
        // 全系実行
        final ForestTask<Void> task0 = new StatementTask.ClearWarnings(0, stmts.get(0));
        final ForestTask<Void> task1 = new StatementTask.ClearWarnings(1, stmts.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public boolean execute(String sql) throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<Boolean> task0 = new StatementTask.Execute(0, stmts.get(0), sql);
        final ForestTask<Boolean> task1 = new StatementTask.Execute(1, stmts.get(1), sql);
        final ForestTask<Boolean> dummy0 = new StatementTask.Execute(0, null, null);
        final ForestTask<Boolean> dummy1 = new StatementTask.Execute(1, null, null);
        final List<Boolean> result;
        try {
            epCommonResource.setExecuteQuery(sql);
            result = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        } finally {
            epCommonResource.setExecuteQuery(null);
        }
        // 次のgetUpdateCountなどの固定系実行のAPIのために実行成功した系をセットする
        // 優先実行系で結果が取得できていればその系を次の実行系IDとし、そうでなければ
        // 逆のサーバID（0/1の2値なので1の補数を取ればよい）次の実行系とする
        final int prefServerId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (result.get(prefServerId) != null) ? prefServerId : (1 - prefServerId) );
        return result.get(getFixExecuteId()).booleanValue();
    }
    
    public boolean execute(String sql, int autoGeneratedKeys)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean execute(String sql, String[] columnNames)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int[] executeBatch() throws SQLException {
        checkObjectClosed();
        
     // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<int[]> task0 = new StatementTask.ExecuteBatch(0, stmts.get(0));
        final ForestTask<int[]> task1 = new StatementTask.ExecuteBatch(1, stmts.get(1));
        final ForestTask<int[]> dummy0 = new StatementTask.ExecuteBatch(0, null);
        final ForestTask<int[]> dummy1 = new StatementTask.ExecuteBatch(1, null);
        
        final List<int[]> results;
        results = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        
        return results.get( (results.get(0) != null) ? 0 : 1);
    }
    
    public ResultSet executeQuery(String sql) throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // 全系実行か任意一系実行かの振り分けを行う（パース）
        final boolean isAllExec;
        switch (parseQuery(sql)) {
            case SELECT: {
                isAllExec = false;
                break;
            }
            case COMMIT: {
                epCommonResource.setIsTxBorder(true);
                isAllExec = true;
                break;
            }
            default: {
                isAllExec = true;
                break;
            }
        }
        
        final ForestTask<ResultSet> task0 = new StatementTask.ExecuteQuery(0, stmts.get(0), sql);
        final ForestTask<ResultSet> task1 = new StatementTask.ExecuteQuery(1, stmts.get(1), sql);
        final ForestTask<ResultSet> dummy0 = new StatementTask.ExecuteQuery(0, null, null);
        final ForestTask<ResultSet> dummy1 = new StatementTask.ExecuteQuery(1, null, null);
        final List<ResultSet> results;
        try {
            epCommonResource.setExecuteQuery(sql);
            if (isAllExec) {
                results = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
            } else {
                results = epCommonResource.executeAnyApiWithPreCheck(task0, task1, dummy0, dummy1);
            }
        } finally {
            epCommonResource.setExecuteQuery(null);
        }
        
        // 優先実行系の結果が存在していればそちらを使い、結果がない場合は逆を使う。
        // ここで取得したResultSetを基にForestResultSetを作成し、固定実行系を定める。
        final int prefExecId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (results.get(prefExecId) != null) ? prefExecId : 1 - prefExecId );
        final ForestResultSet newResult = new ForestResultSet(epCommonResource, results, getFixExecuteId(), new WeakReference<ForestStatement>(this));
        setForestResultSet(newResult);
        return newResult;
    }
    
    public int executeUpdate(String sql) throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // SQLがCOMMITなら、Txの境目であることを宣言する
        if (parseQuery(sql) == SQLType.COMMIT) {
            epCommonResource.setIsTxBorder(true);
        }
        
        final ForestTask<Integer> task0 = new StatementTask.ExecuteUpdate(0, stmts.get(0), sql);
        final ForestTask<Integer> task1 = new StatementTask.ExecuteUpdate(1, stmts.get(1), sql);
        final ForestTask<Integer> dummy0 = new StatementTask.ExecuteUpdate(0, null, null);
        final ForestTask<Integer> dummy1 = new StatementTask.ExecuteUpdate(1, null, null);
        final List<Integer> result;
        try {
            epCommonResource.setExecuteQuery(sql);
            result = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        } finally {
            epCommonResource.setExecuteQuery(null);
        }
        
        // 次のgetUpdateCountなどの固定系実行のAPIのために実行系をセットする
        // 優先実行系で結果が取得できていればその系を次の実行系とし、そうでなければ
        // 逆のサーバID（0/1の2値なので1の補数を取ればよい）次の実行系とする
        final int prefServerId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (result.get(prefServerId) != null) ? prefServerId : (1 - prefServerId) );
        return result.get(getFixExecuteId());
    }
    
    public int executeUpdate(String sql, int autoGeneratedKeys)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int executeUpdate(String sql, int[] columnIndexes)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int executeUpdate(String sql, String[] columnNames)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public ForestConnection getConnection() throws SQLException {
        checkObjectClosed();
        return con.get();
    }
    
    public int getFetchDirection() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final ForestTask<Integer> task0 = new StatementTask.GetFetchDirection(0, stmts.get(0));
        final ForestTask<Integer> task1 = new StatementTask.GetFetchDirection(1, stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public int getFetchSize() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final ForestTask<Integer> task0 = new StatementTask.GetFetchSize(0, stmts.get(0));
        final ForestTask<Integer> task1 = new StatementTask.GetFetchSize(1, stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public ResultSet getGeneratedKeys() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getMaxFieldSize() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final ForestTask<Integer> task0 = new StatementTask.GetMaxFieldSize(0, stmts.get(0));
        final ForestTask<Integer> task1 = new StatementTask.GetMaxFieldSize(1, stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public int getMaxRows() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final ForestTask<Integer> task0 = new StatementTask.GetMaxRows(0, stmts.get(0));
        final ForestTask<Integer> task1 = new StatementTask.GetMaxRows(1, stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public boolean getMoreResults() throws SQLException {
        checkObjectClosed();
        // 固定系実行
        // executeを１回も実行されていないならfalseを返す
        if (getFixExecuteId() == -1) {
            return false;
        }
        final ForestTask<Boolean> task = new StatementTask.GetMoreResults(getFixExecuteId(), stmts.get(getFixExecuteId()));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public boolean getMoreResults(int current) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getQueryTimeout() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final ForestTask<Integer> task0 = new StatementTask.GetQueryTimeout(0, stmts.get(0));
        final ForestTask<Integer> task1 = new StatementTask.GetQueryTimeout(1, stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public ResultSet getResultSet() throws SQLException {
        checkObjectClosed();
        // 固定系実行
        // executeを１回も実行されていないならnullを返す
        if (getFixExecuteId() == -1) {
            return null;
        }
        final ForestTask<ResultSet> task = new StatementTask.GetResultSet(getFixExecuteId(), stmts.get(getFixExecuteId()));
        final ResultSet newResultSet = epCommonResource.executeOneApi(task);
        
        // 新しく取得したResultSetから、ForestResultSetオブジェクトを作り出し、
        // ForestStatementへのResultSetの登録を行う
        final List<ResultSet> resultSetList = new ArrayList<ResultSet>(2);
        resultSetList.add(null);
        resultSetList.add(null);
        resultSetList.set(getFixExecuteId(), newResultSet);
        final ForestResultSet newForestResultSet = new ForestResultSet(epCommonResource, resultSetList, getFixExecuteId(), new WeakReference<ForestStatement>(this));
        this.setForestResultSet(newForestResultSet);
        return newResultSet == null ? null : newForestResultSet;
    }
    
    public int getResultSetConcurrency() throws SQLException {
        checkObjectClosed();
        
        // 読み取り専用のみが利用可能
        return ResultSet.CONCUR_READ_ONLY;
    }
    
    public int getResultSetHoldability() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getResultSetType() throws SQLException {
        checkObjectClosed();
        
        // 任意系実行（その後の実行系に影響無し）
        final ForestTask<Integer> task0 = new StatementTask.GetResultSetType(0, stmts.get(0));
        final ForestTask<Integer> task1 = new StatementTask.GetResultSetType(1, stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public int getUpdateCount() throws SQLException {
        checkObjectClosed();
        // 固定系実行
        // executeを１回も実行されていないなら-1を返す
        if (getFixExecuteId() == -1) {
            return -1;
        }
        final ForestTask<Integer> task = new StatementTask.GetUpdateCount(getFixExecuteId(), stmts.get(getFixExecuteId()));
        return epCommonResource.executeOneApi(task);
    }
    
    public SQLWarning getWarnings() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final ForestTask<SQLWarning> task0 = new StatementTask.GetWarnings(0, stmts.get(0));
        final ForestTask<SQLWarning> task1 = new StatementTask.GetWarnings(1, stmts.get(1));
        final List<SQLWarning> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 );
    }
    
    public void setCursorName(String name) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final ForestTask<Void> task0 = new StatementTask.SetCursorName(0, stmts.get(0), name);
        final ForestTask<Void> task1 = new StatementTask.SetCursorName(1, stmts.get(1), name);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setEscapeProcessing(boolean enable) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final ForestTask<Void> task0 = new StatementTask.SetEscapeProcessing(0, stmts.get(0), enable);
        final ForestTask<Void> task1 = new StatementTask.SetEscapeProcessing(1, stmts.get(1), enable);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setFetchDirection(int direction) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final ForestTask<Void> task0 = new StatementTask.SetFetchDirection_Int(0, stmts.get(0), direction);
        final ForestTask<Void> task1 = new StatementTask.SetFetchDirection_Int(1, stmts.get(1), direction);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setFetchSize(int rows) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final ForestTask<Void> task0 = new StatementTask.SetFetchSize_Int(0, stmts.get(0), rows);
        final ForestTask<Void> task1 = new StatementTask.SetFetchSize_Int(1, stmts.get(1), rows);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setMaxFieldSize(int max) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final ForestTask<Void> task0 = new StatementTask.SetMaxFieldSize(0, stmts.get(0), max);
        final ForestTask<Void> task1 = new StatementTask.SetMaxFieldSize(1, stmts.get(1), max);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setMaxRows(int max) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final ForestTask<Void> task0 = new StatementTask.SetMaxRows(0, stmts.get(0), max);
        final ForestTask<Void> task1 = new StatementTask.SetMaxRows(1, stmts.get(1), max);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setQueryTimeout(int seconds) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final ForestTask<Void> task0 = new StatementTask.SetQueryTimeout(0, stmts.get(0), seconds);
        final ForestTask<Void> task1 = new StatementTask.SetQueryTimeout(1, stmts.get(1), seconds);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void close() throws SQLException {
        if (isClosed == false) {
            if (currentForestResultSet != null) {
                ((ForestCloseable) currentForestResultSet).close();
            }
            final ForestTask<Void> task0 = new StatementTask.Close(0, stmts.get(0));
            final ForestTask<Void> task1 = new StatementTask.Close(1, stmts.get(1));
            // closeは縮退対象としない。これは、close関数の中で仮に縮退などが発生すると
            // 内部から再度closeを呼ばれることになり、無限ループが発生するため
            epCommonResource.executeAllApiWithoutBroken(task0, task1);
        }
        isClosed = true;
    }
    
    public void closeOneSide(int serverId) {
        if (isClosed == false && currentForestResultSet != null) {
            ((ForestCloseable) currentForestResultSet).closeOneSide(serverId);
            // PostgreSQLのstatementは、nullをセットするものの、既存のものを閉塞はしない
            // これは既存のStatementのResultSetが、既に答えをDBから取り出し終わっている場合に
            // そのままデータを取り出し続けることができるようにするため
            stmts.set(serverId, null);
        }
    }
    
    /**
     * parseQueryにて出力されるSQLの種類
     */
    protected enum SQLType {
        /** FOR UPDATE/INSERTの無いSELECT */
        SELECT,
        /** SELECT FOR UPDATE や、 SELECT INTO */
        SELECT_FORUP,
        /** COMMIT */
        COMMIT,
        /** SELECT/SELECT FOR UPDATE/COMMIT 以外のSQL */
        OTHER;
    }
    
    protected final static Pattern selectSqlRegex =
        Pattern.compile("\\A\\s*[Ss][Ee][Ll][Ee][Cc][Tt]\\s");
    
    protected final static Pattern forUpdateSqlRegex =
        Pattern.compile("[Ff][Oo][Rr]\\s+[Uu][Pp][Dd][Aa][Tt][Ee]");
    
    protected final static Pattern commitSqlRegex =
        Pattern.compile("\\A\\s*[Cc][Oo][Mm][Mm][Ii][Tt]");
    
    protected final static Pattern selectIntoSqlRegex =
        Pattern.compile("[Ii][Nn][Tt][Oo]");
    
    /**
     * SQLを解析して、SQLTypeにマッピングする
     * @param sql 解析対象のSQL
     * @return SQLに対応したSQLTypeの値
     */
    protected SQLType parseQuery(final String sql) {
        if (selectSqlRegex.matcher(sql).find()) {
            if (forUpdateSqlRegex.matcher(sql).find() || selectIntoSqlRegex.matcher(sql).find()) {
                return SQLType.SELECT_FORUP;
            } else {
                return SQLType.SELECT;
            }
        } else {
            if (commitSqlRegex.matcher(sql).find()) {
                return SQLType.COMMIT;
            } else {
                return SQLType.OTHER;
            }
        }
    }
}
