/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest;

import java.io.InputStream;
import java.io.Reader;
import java.lang.ref.WeakReference;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;

import org.postgresforest.apibase.*;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.*;
import org.postgresforest.util.*;

import net.jcip.annotations.*;

@NotThreadSafe public class ForestResultSet implements ResultSet, ForestCloseable {
    
    private final EntrypointCommonResource epCommonResource;
    private final List<ResultSet> resList;
    private final int execServerId;
    private final WeakReference<ForestStatement> stmt;
    
    public ForestResultSet(final EntrypointCommonResource epCommonResource, final List<ResultSet> resList, final int execServerId, final WeakReference<ForestStatement> stmt) {
        this.epCommonResource = epCommonResource;
        this.resList = resList;
        this.execServerId = execServerId;
        this.stmt = stmt;
    }
    
    private boolean isClosed = false;
    
    public boolean absolute(int row) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.Absolute_Int(execServerId, resList.get(execServerId), row);
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public void afterLast() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task = new ResultSetTask.AfterLast(execServerId, resList.get(execServerId));
        epCommonResource.executeOneApi(task);
        return;
    }
    
    public void beforeFirst() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task = new ResultSetTask.BeforeFirst(execServerId, resList.get(execServerId));
        epCommonResource.executeOneApi(task);
        return;
    }
    
    public void cancelRowUpdates() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void clearWarnings() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task = new ResultSetTask.ClearWarning(execServerId, resList.get(execServerId));
        epCommonResource.executeOneApi(task);
        return;
    }
    
    public void deleteRow() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int findColumn(String arg0) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Integer> task = new ResultSetTask.FindColumn(execServerId, resList.get(execServerId), arg0);
        return epCommonResource.executeOneApi(task).intValue();
    }
    
    public boolean first() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.First(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public Array getArray(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Array> task = new ResultSetTask.GetArray_Int(execServerId, resList.get(execServerId), columnIndex);
        Array result = epCommonResource.executeOneApi(task);
        final List<Array> results = new ArrayList<Array>(2);
        results.add(null);
        results.add(null);
        results.set(execServerId, result);
        final ForestArray newArray = new ForestArray(epCommonResource, results, execServerId);
        return newArray;
    }
    
    public Array getArray(String columnLabel) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Array> task = new ResultSetTask.GetArray_Str(execServerId, resList.get(execServerId), columnLabel);
        Array result = epCommonResource.executeOneApi(task);
        final List<Array> results = new ArrayList<Array>(2);
        results.add(null);
        results.add(null);
        results.set(execServerId, result);
        final ForestArray newArray = new ForestArray(epCommonResource, results, execServerId);
        return newArray;
    }
    
    public InputStream getAsciiStream(int arg0) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<InputStream> task = new ResultSetTask.GetAsciiStream_Int(execServerId, resList.get(execServerId), arg0);
        return epCommonResource.executeOneApi(task);
    }
    
    public InputStream getAsciiStream(String arg0) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<InputStream> task = new ResultSetTask.GetAsciiStream_Str(execServerId, resList.get(execServerId), arg0);
        return epCommonResource.executeOneApi(task);
    }
    
    public BigDecimal getBigDecimal(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<BigDecimal> task = new ResultSetTask.GetBigDecimal_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public BigDecimal getBigDecimal(String columnLabel) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<BigDecimal> task = new ResultSetTask.GetBigDecimal_Str(execServerId, resList.get(execServerId), columnLabel);
        return epCommonResource.executeOneApi(task);
    }
    
    public BigDecimal getBigDecimal(int arg0, int arg1) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<BigDecimal> task = new ResultSetTask.GetBigDecimal_IntInt(execServerId, resList.get(execServerId), arg0, arg1);
        return epCommonResource.executeOneApi(task);
    }
    
    public BigDecimal getBigDecimal(String arg0, int arg1) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<BigDecimal> task = new ResultSetTask.GetBigDecimal_StrInt(execServerId, resList.get(execServerId), arg0, arg1);
        return epCommonResource.executeOneApi(task);
    }
    
    public InputStream getBinaryStream(int arg0) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<InputStream> task = new ResultSetTask.GetBinaryStream_Int(execServerId, resList.get(execServerId), arg0);
        return epCommonResource.executeOneApi(task);
    }
    
    public InputStream getBinaryStream(String arg0) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<InputStream> task = new ResultSetTask.GetBinaryStream_Str(execServerId, resList.get(execServerId), arg0);
        return epCommonResource.executeOneApi(task);
    }
    
    public Blob getBlob(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Blob> task = new ResultSetTask.GetBlob_Int(execServerId, resList.get(execServerId), columnIndex);
        final Blob result = epCommonResource.executeOneApi(task);
        final List<Blob> results = new ArrayList<Blob>(2);
        results.add(null);
        results.add(null);
        results.set(execServerId, result);
        final ForestBlob newBlob = new ForestBlob(epCommonResource, results, execServerId);
        return newBlob;
    }
    
    public Blob getBlob(String columnLabel) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Blob> task = new ResultSetTask.GetBlob_Str(execServerId, resList.get(execServerId), columnLabel);
        final Blob result = epCommonResource.executeOneApi(task);
        final List<Blob> results = new ArrayList<Blob>(2);
        results.add(null);
        results.add(null);
        results.set(execServerId, result);
        final ForestBlob newBlob = new ForestBlob(epCommonResource, results, execServerId);
        return newBlob;
    }
    
    public boolean getBoolean(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.GetBoolean_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public boolean getBoolean(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.GetBoolean_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public byte getByte(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Byte> task = new ResultSetTask.GetByte_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task).byteValue();
    }
    
    public byte getByte(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Byte> task = new ResultSetTask.GetByte_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task).byteValue();
    }
    
    public byte[] getBytes(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<byte[]> task = new ResultSetTask.GetBytes_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public byte[] getBytes(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<byte[]> task = new ResultSetTask.GetBytes_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task);
    }
    
    public Reader getCharacterStream(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Reader> task = new ResultSetTask.GetCharacterStream_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public Reader getCharacterStream(String columnLabel) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Reader> task = new ResultSetTask.GetCharacterStream_Str(execServerId, resList.get(execServerId), columnLabel);
        return epCommonResource.executeOneApi(task);
    }
    
    public Clob getClob(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Clob> task = new ResultSetTask.GetClob_Int(execServerId, resList.get(execServerId), columnIndex);
        final Clob result = epCommonResource.executeOneApi(task);
        final List<Clob> results = new ArrayList<Clob>(2);
        results.add(null);
        results.add(null);
        results.set(execServerId, result);
        final ForestClob newClob = new ForestClob(epCommonResource, results, execServerId);
        return newClob;
    }
    
    public Clob getClob(String columnLabel) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Clob> task = new ResultSetTask.GetClob_Str(execServerId, resList.get(execServerId), columnLabel);
        final Clob result = epCommonResource.executeOneApi(task);
        final List<Clob> results = new ArrayList<Clob>(2);
        results.add(null);
        results.add(null);
        results.set(execServerId, result);
        final ForestClob newClob = new ForestClob(epCommonResource, results, execServerId);
        return newClob;
    }
    
    public int getConcurrency() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        return ResultSet.CONCUR_READ_ONLY;
    }
    
    public String getCursorName() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<String> task = new ResultSetTask.GetCursorName(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task);
    }
    
    public Date getDate(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Date> task = new ResultSetTask.GetDate_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public Date getDate(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Date> task = new ResultSetTask.GetDate_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task);
    }
    
    public Date getDate(int columnIndex, Calendar cal) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Date> task = new ResultSetTask.GetDate_IntCal(execServerId, resList.get(execServerId), columnIndex, cal);
        return epCommonResource.executeOneApi(task);
    }
    
    public Date getDate(String columnLabel, Calendar cal) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Date> task = new ResultSetTask.GetDate_StrCal(execServerId, resList.get(execServerId), columnLabel, cal);
        return epCommonResource.executeOneApi(task);
    }
    
    public double getDouble(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Double> task = new ResultSetTask.GetDouble_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task).doubleValue();
    }
    
    public double getDouble(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Double> task = new ResultSetTask.GetDouble_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task).doubleValue();
    }
    
    public int getFetchDirection() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Integer> task = new ResultSetTask.GetFetchDirection(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).intValue();
    }
    
    public int getFetchSize() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Integer> task = new ResultSetTask.GetFetchSize(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).intValue();
    }
    
    public float getFloat(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Float> task = new ResultSetTask.GetFloat_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task).floatValue();
    }
    
    public float getFloat(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Float> task = new ResultSetTask.GetFloat_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task).floatValue();
    }
    
    public int getInt(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Integer> task = new ResultSetTask.GetInt_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task).intValue();
    }
    
    public int getInt(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Integer> task = new ResultSetTask.GetInt_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task).intValue();
    }
    
    public long getLong(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Long> task = new ResultSetTask.GetLong_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task).longValue();
    }
    
    public long getLong(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Long> task = new ResultSetTask.GetLong_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task).longValue();
    }
    
    public ResultSetMetaData getMetaData() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<ResultSetMetaData> task = new ResultSetTask.GetMetaData(execServerId, resList.get(execServerId));
        final ResultSetMetaData rsmd = epCommonResource.executeOneApi(task);
        final List<ResultSetMetaData> rsmdList = new ArrayList<ResultSetMetaData>(2);
        rsmdList.add(null);
        rsmdList.add(null);
        rsmdList.set(execServerId, rsmd);
        // XXX （縮退・閉塞）本来Statement系と同様にForestCloseableを実装してコントロールできるようにするべき
        return new ForestResultSetMetaData(epCommonResource, rsmdList, execServerId);
    }
    
    public Object getObject(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Object> task = new ResultSetTask.GetObject_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public Object getObject(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Object> task = new ResultSetTask.GetObject_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task);
    }
    
    public Object getObject(int columnIndex, Map<String, Class<?>> map)
            throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Object> task = new ResultSetTask.GetObject_IntMap(execServerId, resList.get(execServerId), columnIndex, map);
        return epCommonResource.executeOneApi(task);
    }
    
    public Object getObject(String columnLabel, Map<String, Class<?>> map)
            throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Object> task = new ResultSetTask.GetObject_StrMap(execServerId, resList.get(execServerId), columnLabel, map);
        return epCommonResource.executeOneApi(task);
    }
    
    public Ref getRef(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Ref> task = new ResultSetTask.GetRef_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public Ref getRef(String columnLabel) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Ref> task = new ResultSetTask.GetRef_Str(execServerId, resList.get(execServerId), columnLabel);
        return epCommonResource.executeOneApi(task);
    }
    
    public int getRow() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Integer> task = new ResultSetTask.GetRow(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).intValue();
    }
    
    public short getShort(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Short> task = new ResultSetTask.GetShort_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task).shortValue();
    }
    
    public short getShort(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Short> task = new ResultSetTask.GetShort_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task).shortValue();
    }
    
    public Statement getStatement() throws SQLException {
        return stmt.get();
    }
    
    public String getString(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<String> task = new ResultSetTask.GetString_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public String getString(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<String> task = new ResultSetTask.GetString_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task);
    }
    
    public Time getTime(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Time> task = new ResultSetTask.GetTime_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public Time getTime(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Time> task = new ResultSetTask.GetTime_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task);
    }
    
    public Time getTime(int columnIndex, Calendar cal) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Time> task = new ResultSetTask.GetTime_IntCal(execServerId, resList.get(execServerId), columnIndex, cal);
        return epCommonResource.executeOneApi(task);
    }
    
    public Time getTime(String columnLabel, Calendar cal) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Time> task = new ResultSetTask.GetTime_StrCal(execServerId, resList.get(execServerId), columnLabel, cal);
        return epCommonResource.executeOneApi(task);
    }
    
    public Timestamp getTimestamp(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Timestamp> task = new ResultSetTask.GetTimestamp_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public Timestamp getTimestamp(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Timestamp> task = new ResultSetTask.GetTimestamp_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task);
    }
    
    public Timestamp getTimestamp(int columnIndex, Calendar cal)
            throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Timestamp> task = new ResultSetTask.GetTimestamp_IntCal(execServerId, resList.get(execServerId), columnIndex, cal);
        return epCommonResource.executeOneApi(task);
    }
    
    public Timestamp getTimestamp(String columnLabel, Calendar cal)
            throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Timestamp> task = new ResultSetTask.GetTimestamp_StrCal(execServerId, resList.get(execServerId), columnLabel, cal);
        return epCommonResource.executeOneApi(task);
    }
    
    public int getType() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Integer> task = new ResultSetTask.GetType(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).intValue();
    }
    
    public URL getURL(int columnIndex) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public URL getURL(String columnName) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public InputStream getUnicodeStream(int columnIndex) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<InputStream> task = new ResultSetTask.GetUnicodeStream_Int(execServerId, resList.get(execServerId), columnIndex);
        return epCommonResource.executeOneApi(task);
    }
    
    public InputStream getUnicodeStream(String columnName) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<InputStream> task = new ResultSetTask.GetUnicodeStream_Str(execServerId, resList.get(execServerId), columnName);
        return epCommonResource.executeOneApi(task);
    }
    
    public SQLWarning getWarnings() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<SQLWarning> task = new ResultSetTask.GetWarnings(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task);
    }
    
    public void insertRow() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean isAfterLast() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.IsAfterLast(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public boolean isBeforeFirst() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.IsBeforeFirst(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public boolean isFirst() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.IsFirst(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public boolean isLast() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.IsLast(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public boolean last() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.Last(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public void moveToCurrentRow() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void moveToInsertRow() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean next() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.Next(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public boolean previous() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.Previous(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public void refreshRow() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task = new ResultSetTask.RefreshRow(execServerId, resList.get(execServerId));
        epCommonResource.executeOneApi(task);
        return;
    }
    
    public boolean relative(int rows) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.Relative(execServerId, resList.get(execServerId), rows);
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public boolean rowDeleted() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean rowInserted() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean rowUpdated() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setFetchDirection(int direction) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task = new ResultSetTask.SetFetchDirection(execServerId, resList.get(execServerId), direction);
        epCommonResource.executeOneApi(task);
        return;
    }
    
    public void setFetchSize(int rows) throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task = new ResultSetTask.SetFetchSize(execServerId, resList.get(execServerId), rows);
        epCommonResource.executeOneApi(task);
        return;
    }
    
    public void updateArray(int columnIndex, Array x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateArray(String columnName, Array x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateAsciiStream(int columnIndex, InputStream x, int length)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateAsciiStream(String columnName, InputStream x, int length)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBigDecimal(int columnIndex, BigDecimal x)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBigDecimal(String columnName, BigDecimal x)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBinaryStream(int columnIndex, InputStream x, int length)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBinaryStream(String columnName, InputStream x, int length)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBlob(int columnIndex, Blob x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBlob(String columnName, Blob x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBoolean(int columnIndex, boolean x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBoolean(String columnName, boolean x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateByte(int columnIndex, byte x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateByte(String columnName, byte x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBytes(int columnIndex, byte[] x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateBytes(String columnName, byte[] x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateCharacterStream(int columnIndex, Reader x, int length)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateCharacterStream(String columnName, Reader reader,
            int length) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateClob(int columnIndex, Clob x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateClob(String columnName, Clob x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateDate(int columnIndex, Date x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateDate(String columnName, Date x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateDouble(int columnIndex, double x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateDouble(String columnName, double x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateFloat(int columnIndex, float x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateFloat(String columnName, float x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateInt(int columnIndex, int x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateInt(String columnName, int x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateLong(int columnIndex, long x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateLong(String columnName, long x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateNull(int columnIndex) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateNull(String columnName) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateObject(int columnIndex, Object x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateObject(String columnName, Object x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateObject(int columnIndex, Object x, int scale)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateObject(String columnName, Object x, int scale)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateRef(int columnIndex, Ref x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateRef(String columnName, Ref x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateRow() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateShort(int columnIndex, short x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateShort(String columnName, short x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateString(int columnIndex, String x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateString(String columnName, String x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateTime(int columnIndex, Time x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateTime(String columnName, Time x) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateTimestamp(int columnIndex, Timestamp x)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void updateTimestamp(String columnName, Timestamp x)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean wasNull() throws SQLException {
        if (isClosed) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task = new ResultSetTask.WasNull(execServerId, resList.get(execServerId));
        return epCommonResource.executeOneApi(task).booleanValue();
    }
    
    public void close() throws SQLException {
        if (isClosed == false) {
            final ForestTask<Void> task0 = new ResultSetTask.Close(0, resList.get(0));
            final ForestTask<Void> task1 = new ResultSetTask.Close(1, resList.get(1));
            // closeは縮退対象としない。これは、close関数の中で仮に縮退などが発生すると
            // 内部から再度closeを呼ばれることになり、無限ループが発生するため
            epCommonResource.executeAllApiWithoutBroken(task0, task1);
        }
        isClosed = true;
    }
    
    public void closeOneSide(int serverId) {
        // 縮退で閉塞する場合にPostgreSQLのResultSetオブジェクトは
        // 閉塞しない。これは、既に解を取得済みの場合に、リソースを
        // 解放してしまうとその結果が得られないため
    }
    
}
