/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.lang.ref.WeakReference;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;

import net.jcip.annotations.NotThreadSafe;

import org.postgresforest.apibase.*;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.*;
import org.postgresforest.util.*;

@NotThreadSafe public class ForestPreparedStatement extends ForestStatement implements PreparedStatement {
    
    protected final String executeSql;
    
    public ForestPreparedStatement(EntrypointCommonResource epCommonResource, List<? extends PreparedStatement> stmts, String executeSql, final WeakReference<ForestConnection> con) {
        super(epCommonResource, stmts, con);
        this.executeSql = executeSql;
    }
    
    public void addBatch() throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.AddBatch(0, (PreparedStatement) stmts.get(0));
        final ForestTask<Void> task1 = new PreparedStatementTask.AddBatch(1, (PreparedStatement) stmts.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void clearParameters() throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.ClearParameters(0, (PreparedStatement) stmts.get(0));
        final ForestTask<Void> task1 = new PreparedStatementTask.ClearParameters(1, (PreparedStatement) stmts.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public boolean execute() throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // FIXME （仕様調整） Tx関連のDCLはexecute系APIでは許可しないようにするべきか？
        
        final ForestTask<Boolean> task0 = new PreparedStatementTask.Execute(0, (PreparedStatement) stmts.get(0));
        final ForestTask<Boolean> task1 = new PreparedStatementTask.Execute(1, (PreparedStatement) stmts.get(1));
        final ForestTask<Boolean> dummy0 = new PreparedStatementTask.Execute(0, null);
        final ForestTask<Boolean> dummy1 = new PreparedStatementTask.Execute(1, null);
        final List<Boolean> result;
        try {
            epCommonResource.setExecuteQuery(executeSql);
            result = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        } finally {
            epCommonResource.setExecuteQuery(null);
        }
        
        // 次のgetUpdateCountなどの固定系実行のAPIのために実行成功した系をセットする
        // 優先実行系で結果が取得できていればその系を次の実行系IDとし、そうでなければ
        // 逆のサーバID（0/1の2値なので1の補数を取ればよい）次の実行系とする
        final int prefServerId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (result.get(prefServerId) != null) ? prefServerId : (1 - prefServerId) );
        return result.get(getFixExecuteId()).booleanValue();
    }
    
    public ResultSet executeQuery() throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // 全系実行か任意一系実行かの振り分けを行う（パース）
        final boolean isAllExec;
        switch (parseQuery(executeSql)) {
            case SELECT: {
                isAllExec = false;
                break;
            }
            case COMMIT: {
                epCommonResource.setIsTxBorder(true);
                isAllExec = true;
                break;
            }
            default: {
                isAllExec = true;
                break;
            }
        }
        
        final ForestTask<ResultSet> task0 = new PreparedStatementTask.ExecuteQuery(0, (PreparedStatement) stmts.get(0));
        final ForestTask<ResultSet> task1 = new PreparedStatementTask.ExecuteQuery(1, (PreparedStatement) stmts.get(1));
        final ForestTask<ResultSet> dummy0 = new PreparedStatementTask.ExecuteQuery(0, null);
        final ForestTask<ResultSet> dummy1 = new PreparedStatementTask.ExecuteQuery(1, null);
        final List<ResultSet> results;
        
        try {
            epCommonResource.setExecuteQuery(executeSql);
            if (isAllExec) {
                // 全系実行の場合
                results = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
                
            } else {
                // 任意1系実行の場合
                results = epCommonResource.executeAnyApiWithPreCheck(task0, task1, dummy0, dummy1);
            }
        } finally {
            epCommonResource.setExecuteQuery(null);
        }
        
        // 優先実行系の結果が存在していればそちらを使い、結果がない場合は逆を使う。
        // ここで取得したResultSetを基にForestResultSetを作成し、固定実行系を定める。
        final int prefExecId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (results.get(prefExecId) != null) ? prefExecId : 1 - prefExecId );
        final ForestResultSet newResult = new ForestResultSet(epCommonResource, results, getFixExecuteId(), new WeakReference<ForestStatement>(this));
        setForestResultSet(newResult);
        return newResult;
    }
    
    public int executeUpdate() throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // SQLがCOMMITなら、Txの境目であることを宣言する
        if (parseQuery(executeSql) == SQLType.COMMIT) {
            epCommonResource.setIsTxBorder(true);
        }
        
        final ForestTask<Integer> task0 = new PreparedStatementTask.ExecuteUpdate(0, (PreparedStatement) stmts.get(0));
        final ForestTask<Integer> task1 = new PreparedStatementTask.ExecuteUpdate(1, (PreparedStatement) stmts.get(1));
        final ForestTask<Integer> dummy0 = new PreparedStatementTask.ExecuteUpdate(0, null);
        final ForestTask<Integer> dummy1 = new PreparedStatementTask.ExecuteUpdate(1, null);
        final List<Integer> result;
        try {
            epCommonResource.setExecuteQuery(executeSql);
            result = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        } finally {
            epCommonResource.setExecuteQuery(null);
        }
        
        // 次のgetUpdateCountなどの固定系実行のAPIのために実行系をセットする
        // 優先実行系で結果が取得できていればその系を次の実行系とし、そうでなければ
        // 逆のサーバID（0/1の2値なので1の補数を取ればよい）次の実行系とする
        final int prefServerId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (result.get(prefServerId) != null) ? prefServerId : (1 - prefServerId) );
        return result.get(getFixExecuteId());
    }
    
    public ResultSetMetaData getMetaData() throws SQLException {
        checkObjectClosed();
        
        final ForestTask<ResultSetMetaData> task0 = new PreparedStatementTask.GetMetadata(0, (PreparedStatement) stmts.get(0));
        final ForestTask<ResultSetMetaData> task1 = new PreparedStatementTask.GetMetadata(1, (PreparedStatement) stmts.get(1));
        final List<ResultSetMetaData> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1);
        
    }
    
    public ParameterMetaData getParameterMetaData() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setArray(int parameterIndex, Array x) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetArray(0, (PreparedStatement) stmts.get(0), parameterIndex, x);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetArray(1, (PreparedStatement) stmts.get(1), parameterIndex, x);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setAsciiStream(int arg0, InputStream arg1, int arg2) throws SQLException {
        checkObjectClosed();
        
        final List<InputStream> streamList;
        try {
            streamList = SimpleCopyParallelInputStream.createParallelInputStream(arg1, arg2);
        } catch (IOException e) {
            final ForestException exp = new ForestException(ErrorStr.STREAM_DUPLICATE_FAILED.toString());
            exp.setMultipleCause( Arrays.<Exception>asList(e, null) );
            throw exp;
        }
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetAsciiStream(0, (PreparedStatement) stmts.get(0), arg0, streamList.get(0), arg2);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetAsciiStream(1, (PreparedStatement) stmts.get(1), arg0, streamList.get(1), arg2);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBigDecimal(int arg0, BigDecimal arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetBigDecimal(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetBigDecimal(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBinaryStream(int arg0, InputStream arg1, int arg2) throws SQLException {
        checkObjectClosed();
        
        final List<InputStream> streamList;
        try {
            streamList = SimpleCopyParallelInputStream.createParallelInputStream(arg1, arg2);
        } catch (IOException e) {
            final ForestException exp = new ForestException(ErrorStr.STREAM_DUPLICATE_FAILED.toString());
            exp.setMultipleCause( Arrays.<Exception>asList(e, null) );
            throw exp;
        }
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetBinaryStream(0, (PreparedStatement) stmts.get(0), arg0, streamList.get(0), arg2);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetBinaryStream(1, (PreparedStatement) stmts.get(1), arg0, streamList.get(1), arg2);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBlob(int parameterIndex, Blob x) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetBlob(0, (PreparedStatement) stmts.get(0), parameterIndex, x);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetBlob(1, (PreparedStatement) stmts.get(1), parameterIndex, x);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBoolean(int arg0, boolean arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetBoolean(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetBoolean(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setByte(int arg0, byte arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetByte(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetByte(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setBytes(int arg0, byte[] arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetBytes(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetBytes(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setCharacterStream(int parameterIndex, Reader reader, int length)
            throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetCharacterStream(0, (PreparedStatement) stmts.get(0), parameterIndex, reader, length);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetCharacterStream(1, (PreparedStatement) stmts.get(1), parameterIndex, reader, length);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setClob(int parameterIndex, Clob x) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetClob(0, (PreparedStatement) stmts.get(0), parameterIndex, x);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetClob(1, (PreparedStatement) stmts.get(1), parameterIndex, x);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setDate(int arg0, Date arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetDate_IntDate(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetDate_IntDate(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetDate_IntDateCal(0, (PreparedStatement) stmts.get(0), parameterIndex, x, cal);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetDate_IntDateCal(1, (PreparedStatement) stmts.get(1), parameterIndex, x, cal);
        epCommonResource.executeAllApi(task0, task1);
    }
    
    public void setDouble(int arg0, double arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetDouble(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetDouble(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setFloat(int arg0, float arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetFloat(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetFloat(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setInt(int arg0, int arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetInt(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetInt(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setLong(int arg0, long arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetLong(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetLong(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setNull(int arg0, int arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetNull_IntInt(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetNull_IntInt(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setNull(int parameterIndex, int sqlType, String typeName) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetNull_IntIntStr(0, (PreparedStatement) stmts.get(0), parameterIndex, sqlType, typeName);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetNull_IntIntStr(1, (PreparedStatement) stmts.get(1), parameterIndex, sqlType, typeName);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setObject(int arg0, Object arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetObject_IntObj(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetObject_IntObj(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setObject(int arg0, Object arg1, int arg2) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetObject_IntObjInt(0, (PreparedStatement) stmts.get(0), arg0, arg1, arg2);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetObject_IntObjInt(1, (PreparedStatement) stmts.get(1), arg0, arg1, arg2);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setObject(int arg0, Object arg1, int arg2, int arg3)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setRef(int parameterIndex, Ref x) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetRef(0, (PreparedStatement) stmts.get(0), parameterIndex, x);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetRef(1, (PreparedStatement) stmts.get(1), parameterIndex, x);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setShort(int arg0, short arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetShort(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetShort(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setString(int arg0, String arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetString(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetString(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTime(int arg0, Time arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetTime_IntTime(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetTime_IntTime(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetTime_IntTimeCal(0, (PreparedStatement) stmts.get(0), parameterIndex, x, cal);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetTime_IntTimeCal(1, (PreparedStatement) stmts.get(1), parameterIndex, x, cal);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTimestamp(int arg0, Timestamp arg1) throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetTimestamp_IntTime(0, (PreparedStatement) stmts.get(0), arg0, arg1);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetTimestamp_IntTime(1, (PreparedStatement) stmts.get(1), arg0, arg1);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal)
            throws SQLException {
        checkObjectClosed();
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetTimestamp_IntTimeCal(0, (PreparedStatement) stmts.get(0), parameterIndex, x, cal);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetTimestamp_IntTimeCal(1, (PreparedStatement) stmts.get(1), parameterIndex, x, cal);
        epCommonResource.executeAllApi(task0, task1);
        return;
        }
    
    public void setURL(int arg0, URL arg1) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setUnicodeStream(int arg0, InputStream arg1, int arg2)
            throws SQLException {
        checkObjectClosed();
        
        final List<InputStream> streamList;
        try {
            streamList = SimpleCopyParallelInputStream.createParallelInputStream(arg1, arg2);
        } catch (IOException e) {
            final ForestException exp = new ForestException(ErrorStr.STREAM_DUPLICATE_FAILED.toString());
            exp.setMultipleCause( Arrays.<Exception>asList(e, null) );
            throw exp;
        }
        
        final ForestTask<Void> task0 = new PreparedStatementTask.SetUnicodeStream(0, (PreparedStatement) stmts.get(0), arg0, streamList.get(0), arg2);
        final ForestTask<Void> task1 = new PreparedStatementTask.SetUnicodeStream(1, (PreparedStatement) stmts.get(1), arg0, streamList.get(1), arg2);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
}
