/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest;

import java.sql.*;
import java.util.*;
import java.lang.ref.WeakReference;

import org.postgresforest.apibase.*;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.ForestException;
import org.postgresforest.util.*;

import net.jcip.annotations.*;


@NotThreadSafe public final class ForestConnection implements Connection, ForestCloseable {
    
    /** このForestConnectionに紐づくリソースを管理するコントローラ */
    private final EntrypointCommonResource epCommonResource;
    
    private final List<Connection> conList;
    
    /**
     *  このコネクションオブジェクトから生成される子を格納するリンクリスト
     *  （弱参照のリストであり、列挙→get→参照がnullなら要素を消去、という
     *  走査処理を行うためリンクリストの実装を選択している）
     */
    private final List<WeakReference<ForestCloseable>> childList = new LinkedList<WeakReference<ForestCloseable>>();
    /**
     * ForestConnectionの直接の子要素となるForestのJDBCオブジェクトを登録する。
     * この関数で登録したオブジェクトに対しては、特定タイミングで次の操作を行う
     * 1. 縮退が発生した場合に、系の閉塞処理を行う（ｘｘｘｘ関数）
     * 2. リカバリが発生した場合に、オブジェクトそのものの閉塞処理を行う（ｘｘｘｘ関数）
     * 但し、この関数での登録は、WeakReferenceとして保持することになるため、
     * 対応する参照解除の関数は存在しない。
     * （ユーザアプリケーションがStatementなどのオブジェクトを解放する際に
     * Closeなどを呼ぶことを義務付けていないため、登録解除はGCの処理に任せる）
     * @param child
     */
    protected void addChild(final ForestCloseable child) {
        final WeakReference<ForestCloseable> childRef = new WeakReference<ForestCloseable>(child);
        childList.add(childRef);
        
        // リストの無駄な拡大を防ぐため、適当な大きさになった段階でリストの整理を行う
        // 弱参照の参照先がnullになっているエントリを削除する
        // サイズが10の倍数になった時にやっているのは、単に走査作業をサボって高速化している。
        // このエントリが使われる状況は、リカバリ時や縮退時など限られた時であり、
        // 通常のStatement等の作成処理を高速化することを優先するため。
        if (childList.size() % 10 == 0) {
            ListIterator<WeakReference<ForestCloseable>> iter = childList.listIterator();
            while (iter.hasNext()) {
                if (iter.next().get() == null) {
                    iter.remove();
                }
            }
        }
    }
    
    /**
     * ForestConnectionのコンストラクタ<br>
     * このクラスはDriverクラス以外から作成されてはいけないため、protectedとしている
     * @param epCommonResource ForestConStateControllerオブジェクト
     * @param conList 
     */
    protected ForestConnection(final EntrypointCommonResource epCommonResource, final List<Connection> conList) {
        this.epCommonResource = epCommonResource;
        this.conList = conList;
    }
    
    /**
     * ForestConnectionオブジェクトのファイナライザ（オーバーライド）<br>
     * アプリケーションがcloseを閉じ忘れた場合にも、EntrypointCommonResourceの
     * releaseForestEntryPointを実行しなくてはならない（スレッドが永久に終わらない）
     * ため、その処理を実装する。ファイナライザが呼ばれるのはアプリケーション
     * スレッドとは全く別なので、メモリの可視性について細心の注意を払う必要がある
     */
    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }
    
    public void commit() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // トランザクションの境界であることを最初に宣言してAPIを実行する
        epCommonResource.setIsTxBorder(true);
        
        final ForestTask<Void> task0 = new ConnectionTask.Commit(0, conList.get(0));
        final ForestTask<Void> task1 = new ConnectionTask.Commit(1, conList.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void clearWarnings() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task0 = new ConnectionTask.ClearWarnings(0, conList.get(0));
        final ForestTask<Void> task1 = new ConnectionTask.ClearWarnings(1, conList.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public Statement createStatement() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<Statement> task0 = new ConnectionTask.CreateStatement(0, conList.get(0));
        final ForestTask<Statement> task1 = new ConnectionTask.CreateStatement(1, conList.get(1));
        final ForestTask<Statement> dummy0 = new ConnectionTask.CreateStatement(0, null);
        final ForestTask<Statement> dummy1 = new ConnectionTask.CreateStatement(1, null);
        final List<Statement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        final ForestStatement newStatement = new ForestStatement(epCommonResource, stmtList, new WeakReference<ForestConnection>(this));
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        
        // 更新可能なカーソルを取得しようとした場合、サポート対象外の例外をスローする
        if (resultSetConcurrency == ResultSet.CONCUR_UPDATABLE) {
            throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED_UPDATEABLE_CURSOR.toString());
        }
        
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<Statement> task0 = new ConnectionTask.CreateStatement_IntInt(0, conList.get(0), resultSetType, resultSetConcurrency);
        final ForestTask<Statement> task1 = new ConnectionTask.CreateStatement_IntInt(1, conList.get(1), resultSetType, resultSetConcurrency);
        final ForestTask<Statement> dummy0 = new ConnectionTask.CreateStatement_IntInt(0, null, resultSetType, resultSetConcurrency);
        final ForestTask<Statement> dummy1 = new ConnectionTask.CreateStatement_IntInt(1, null, resultSetType, resultSetConcurrency);
        final List<Statement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        final ForestStatement newStatement = new ForestStatement(epCommonResource, stmtList, new WeakReference<ForestConnection>(this));
        // close 縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public Statement createStatement(int arg0, int arg1, int arg2)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean getAutoCommit() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task0 = new ConnectionTask.GetAutoCommit(0, conList.get(0));
        final ForestTask<Boolean> task1 = new ConnectionTask.GetAutoCommit(1, conList.get(1));
        final List<Boolean> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1).booleanValue();
    }
    
    public String getCatalog() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<String> task0 = new ConnectionTask.GetCatalog(0, conList.get(0));
        final ForestTask<String> task1 = new ConnectionTask.GetCatalog(1, conList.get(1));
        final List<String> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1);
    }
    
    public int getHoldability() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public DatabaseMetaData getMetaData() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // DatabaseMetaDataを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<DatabaseMetaData> task0 = new ConnectionTask.GetMetaData(0, conList.get(0));
        final ForestTask<DatabaseMetaData> task1 = new ConnectionTask.GetMetaData(1, conList.get(1));
        final ForestTask<DatabaseMetaData> dummy0 = new ConnectionTask.GetMetaData(0, null);
        final ForestTask<DatabaseMetaData> dummy1 = new ConnectionTask.GetMetaData(1, null);
        final List<DatabaseMetaData> dbmds = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        final ForestDatabaseMetaData newMetaData = new ForestDatabaseMetaData(epCommonResource, dbmds, new WeakReference<ForestConnection>(this));
        // close・縮退の対象として登録する
        addChild(newMetaData);
        return newMetaData;
    }
    
    public int getTransactionIsolation() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Integer> task0 = new ConnectionTask.GetTransactionIsolation(0, conList.get(0));
        final ForestTask<Integer> task1 = new ConnectionTask.GetTransactionIsolation(1, conList.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1).intValue();
    }
    
    public Map<String, Class<?>> getTypeMap() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Map<String, Class<?>>> task0 = new ConnectionTask.GetTypeMap(0, conList.get(0));
        final ForestTask<Map<String, Class<?>>> task1 = new ConnectionTask.GetTypeMap(1, conList.get(1));
        final List<Map<String, Class<?>>> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1);
    }
    
    public SQLWarning getWarnings() throws SQLException {
        final ForestTask<SQLWarning> task0 = new ConnectionTask.GetWarnings(0, conList.get(0));
        final ForestTask<SQLWarning> task1 = new ConnectionTask.GetWarnings(1, conList.get(1));
        final List<SQLWarning> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1);
    }
    
    public boolean isClosed() {
        return epCommonResource.getIsClosed();
    }
    
    public boolean isReadOnly() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Boolean> task0 = new ConnectionTask.IsReadOnly(0, conList.get(0));
        final ForestTask<Boolean> task1 = new ConnectionTask.IsReadOnly(1, conList.get(1));
        final List<Boolean> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1).booleanValue();
    }
    
    public String nativeSQL(String sql) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<String> task0 = new ConnectionTask.NativeSQL_String(0, conList.get(0), sql);
        final ForestTask<String> task1 = new ConnectionTask.NativeSQL_String(1, conList.get(1), sql);
        final List<String> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get((results.get(0) != null) ? 0 : 1);
    }
    
    public CallableStatement prepareCall(String sql) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<CallableStatement> task0 = new ConnectionTask.PrepareCall(0, conList.get(0), sql);
        final ForestTask<CallableStatement> task1 = new ConnectionTask.PrepareCall(1, conList.get(1), sql);
        final ForestTask<CallableStatement> dummy0 = new ConnectionTask.PrepareCall(0, null, null);
        final ForestTask<CallableStatement> dummy1 = new ConnectionTask.PrepareCall(1, null, null);
        final List<CallableStatement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        final ForestCallableStatement newStatement = new ForestCallableStatement(epCommonResource, stmtList, sql, new WeakReference<ForestConnection>(this));
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency)
            throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        
     // 更新可能なカーソルを取得しようとした場合、サポート対象外の例外をスローする
        if (resultSetConcurrency == ResultSet.CONCUR_UPDATABLE) {
            throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED_UPDATEABLE_CURSOR.toString());
        }
        
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<CallableStatement> task0 = new ConnectionTask.PrepareCall_StringIntInt(0, conList.get(0), sql, resultSetType, resultSetConcurrency);
        final ForestTask<CallableStatement> task1 = new ConnectionTask.PrepareCall_StringIntInt(1, conList.get(1), sql, resultSetType, resultSetConcurrency);
        final ForestTask<CallableStatement> dummy0 = new ConnectionTask.PrepareCall_StringIntInt(0, null, null, resultSetType, resultSetConcurrency);
        final ForestTask<CallableStatement> dummy1 = new ConnectionTask.PrepareCall_StringIntInt(1, null, null, resultSetType, resultSetConcurrency);
        final List<CallableStatement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        final ForestCallableStatement newStatement = new ForestCallableStatement(epCommonResource, stmtList, sql, new WeakReference<ForestConnection>(this));
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public CallableStatement prepareCall(String arg0, int arg1, int arg2,
            int arg3) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<PreparedStatement> task0 = new ConnectionTask.PrepareStatement(0, conList.get(0), sql);
        final ForestTask<PreparedStatement> task1 = new ConnectionTask.PrepareStatement(1, conList.get(1), sql);
        final ForestTask<PreparedStatement> dummy0 = new ConnectionTask.PrepareStatement(0, null, null);
        final ForestTask<PreparedStatement> dummy1 = new ConnectionTask.PrepareStatement(1, null, null);
        final List<PreparedStatement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        final ForestPreparedStatement newStatement = new ForestPreparedStatement(epCommonResource, stmtList, sql, new WeakReference<ForestConnection>(this));
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public PreparedStatement prepareStatement(String arg0, int arg1)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public PreparedStatement prepareStatement(String arg0, int[] arg1)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public PreparedStatement prepareStatement(String sql, String[] columnNames)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public PreparedStatement prepareStatement(String sql, int resultSetType,
            int resultSetConcurrency) throws SQLException {
        
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // アップデート可能なResultSetは本JDBCではサポートしない
        if (resultSetConcurrency == ResultSet.CONCUR_UPDATABLE) {
            throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED_UPDATEABLE_CURSOR.toString());
        }
        // 行儀のよいアプリケーションのために、リカバリ最中であるならば
        // Statementを作成する直前でリカバリ完了を待ち、リソースを回復する
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final ForestTask<PreparedStatement> task0 = new ConnectionTask.PrepareStatement_IntInt(0, conList.get(0), sql, resultSetType, resultSetConcurrency);
        final ForestTask<PreparedStatement> task1 = new ConnectionTask.PrepareStatement_IntInt(1, conList.get(1), sql, resultSetType, resultSetConcurrency);
        final ForestTask<PreparedStatement> dummy0 = new ConnectionTask.PrepareStatement_IntInt(0, null, null, resultSetType, resultSetConcurrency);
        final ForestTask<PreparedStatement> dummy1 = new ConnectionTask.PrepareStatement_IntInt(1, null, null, resultSetType, resultSetConcurrency);
        final List<PreparedStatement> stmtList = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy0, dummy1);
        final ForestPreparedStatement newStatement = new ForestPreparedStatement(epCommonResource, stmtList, sql, new WeakReference<ForestConnection>(this));
        // close・縮退の対象として登録する
        addChild(newStatement);
        return newStatement;
    }
    
    public PreparedStatement prepareStatement(String sql, int resultSetType,
            int resultSetConcurrency, int resultSetHoldability)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void rollback() throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // トランザクションの境界であることを最初に宣言してAPIを実行する
        epCommonResource.setIsTxBorder(true);
        
        final ForestTask<Void> task0 = new ConnectionTask.Rollback(0, conList.get(0));
        final ForestTask<Void> task1 = new ConnectionTask.Rollback(1, conList.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void rollback(Savepoint savepoint) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        // setAutoCommit関数は事前のトランザクションをコミットするため、
        // トランザクションの境界であることを最初に宣言してからAPIを実行する
        epCommonResource.setIsTxBorder(true);
        
        final ForestTask<Void> task0 = new ConnectionTask.SetAutoCommit(0, conList.get(0), autoCommit);
        final ForestTask<Void> task1 = new ConnectionTask.SetAutoCommit(1, conList.get(1), autoCommit);
        epCommonResource.executeAllApi(task0, task1);
        epCommonResource.setIsAutocommit(autoCommit);
        return;
    }
    
    public void setCatalog(String catalog) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task0 = new ConnectionTask.SetCatalog(0, conList.get(0), catalog);
        final ForestTask<Void> task1 = new ConnectionTask.SetCatalog(1, conList.get(1), catalog);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setHoldability(int holdability) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setReadOnly(boolean readOnly) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task0 = new ConnectionTask.SetReadOnly(0, conList.get(0), readOnly);
        final ForestTask<Void> task1 = new ConnectionTask.SetReadOnly(1, conList.get(1), readOnly);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public Savepoint setSavepoint() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public Savepoint setSavepoint(String name) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setTransactionIsolation(int level) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task0 = new ConnectionTask.SetTransactionIsolation(0, conList.get(0), level);
        final ForestTask<Void> task1 = new ConnectionTask.SetTransactionIsolation(1, conList.get(1), level);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
        if (isClosed() == true) {
            throw new ForestException(ErrorStr.RESOURCE_CLOSED.toString());
        }
        final ForestTask<Void> task0 = new ConnectionTask.SetTypeMap_Map(0, conList.get(0), map);
        final ForestTask<Void> task1 = new ConnectionTask.SetTypeMap_Map(1, conList.get(1), map);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    /**
     * Forestコネクションから生成した、現在有効なStatement等のオブジェクトを全て
     * 使用出来ないものとする。これはリカバリ時に
     */
    public void closeActiveStatements() {
        if (isClosed() == true) {
            return;
        }
        
        final ListIterator<WeakReference<ForestCloseable>> iter = childList.listIterator();
        while (iter.hasNext()) {
            final ForestCloseable jdbcObject = iter.next().get();
            if (jdbcObject == null) {
                iter.remove();
            } else {
                try { jdbcObject.close(); } catch (SQLException ignore) { }
            }
        }
    }
    
    public void closeOneSide(int serverId) {
        if (isClosed() == true) {
            return;
        }
        
        // まず子のStatement系を閉じる
        final ListIterator<WeakReference<ForestCloseable>> iter = childList.listIterator();
        while (iter.hasNext()) {
            final ForestCloseable jdbcObject = iter.next().get();
            if (jdbcObject == null) {
                iter.remove();
            } else {
                jdbcObject.closeOneSide(serverId);
            }
        }
        
        // 続いて子のコネクションを閉じる
        final Connection targetCon = conList.get(serverId);
        if (targetCon != null) {
            try {
                targetCon.close();
            } catch (SQLException e) { }
        }
        conList.set(serverId, null);
    }
    
    public void close() throws SQLException {
        if (isClosed() == true) {
            return;
        }
        try {
            // まず子のStatement系を閉じる
            closeActiveStatements();
            // 続いて子のConnectionを閉じる
            final ForestTask<Void> task0 = new ConnectionTask.Close(0, conList.get(0));
            final ForestTask<Void> task1 = new ConnectionTask.Close(1, conList.get(1));
            // closeは縮退対象としない。これは、close関数の中で仮に縮退などが発生すると
            // 内部から再度closeを呼ばれることになり、無限ループが発生するため
            epCommonResource.executeAllApiWithoutBroken(task0, task1);
        } finally {
            // 最終的に全てのリソースを閉じる
            epCommonResource.releaseEntryPointCommonResource();
        }
    }
    
    /**
     * リカバリ時にEntrypointCommonResourceから呼び出される関数。以前から存在している
     * コネクションの各パラメータを、新規に与えられたコネクションに適用し、以降両系で
     * 使用できるようにする
     * @param recoverServerId
     * @param newCon
     * @return パラメータの取得・適用に成功したならtrue、失敗したならfalse
     * @throws SQLException パラメータの取得・適用中のDBアクセスで失敗した場合
     */
    public boolean recovery(final int recoverServerId, final Connection newCon) throws SQLException {
        if (isClosed() == true) {
            return false;
        }
        
        final Connection srcCon = conList.get(1 - recoverServerId);
        if (srcCon != null) {
            // XXX （サポートAPI拡張時要チェック）
            // 新規に生成したコネクションの状態を動いている系のコネクションと同じにする
            newCon.setTransactionIsolation(srcCon.getTransactionIsolation());
            newCon.setAutoCommit(srcCon.getAutoCommit());
            // コネクションリストに新規に作ったコネクションを登録する
            conList.set(recoverServerId, newCon);
            return true;
        }
        return false;
    }
}
