/* 
 * Copyright (C) since 2008 NTT DATA Corporation 
 *  
 */ 

package org.postgresforest.apibase;

import java.sql.*;
import java.util.concurrent.*;

import org.postgresforest.exception.ForestTaskNotExecutedException;

/**
 * PostgreSQLのJDBCドライバに対してConnectionクラスのAPIを実行するための
 * Callableタスクのインプリを集約したクラス。このクラス自体は、メンバや
 * メソッドを持たない（内部クラスを持つのみ）
 * 
 * 各タスクは、コンストラクタで与えられたJDBCリソースについて、nullか否かを
 * 必ずチェックする必要がある。nullの場合、その系は縮退していることを意味し、
 * call内で即座に「ForestTaskNotExecutedException」をスローする。
 */
public final class ConnectionTask {
    
    public static final class Close implements Callable<Void> {
        private final Connection con;
        public Close(final Connection con) {
            this.con = con;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.close();
            return null;
        }
        
    }
    
    public static final class Commit implements Callable<Void> {
        private final Connection con;
        public Commit(final Connection con) {
            this.con = con;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.commit();
            return null;
        }
        
    }
    
    public static final class Rollback implements Callable<Void> {
        private final Connection con;
        public Rollback(final Connection con) {
            this.con = con;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.rollback();
            return null;
        }
        
    }
    
    public static final class SetAutoCommit implements Callable<Void> {
        private final Connection con;
        private final boolean autocommit;
        public SetAutoCommit(final Connection con, final boolean autocommit) {
            this.con = con;
            this.autocommit = autocommit;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setAutoCommit(autocommit);
            return null;
        }
        
    }
    
    public static final class ClearWarnings implements Callable<Void> {
        private final Connection con;
        public ClearWarnings(final Connection con) {
            this.con = con;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.clearWarnings();
            return null;
        }
        
    }
    
    public static final class SetCatalog implements Callable<Void> {
        private final Connection con;
        private final String catalog;
        public SetCatalog(final Connection con, final String catalog) {
            this.con = con;
            this.catalog = catalog;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setCatalog(catalog);
            return null;
        }
        
    }
    
    public static final class SetReadOnly implements Callable<Void> {
        private final Connection con;
        private final boolean readOnly;
        public SetReadOnly(final Connection con, final boolean readOnly) {
            this.con = con;
            this.readOnly = readOnly;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setReadOnly(readOnly);
            return null;
        }
        
    }
    
    public static final class SetTransactionIsolation implements Callable<Void> {
        private final Connection con;
        private final int level;
        public SetTransactionIsolation(final Connection con, final int level) {
            this.con = con;
            this.level = level;
        }
        public Void call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            con.setTransactionIsolation(level);
            return null;
        }
        
    }
    
    public static final class GetAutoCommit implements Callable<Boolean> {
        private final Connection con;
        public GetAutoCommit(final Connection con) {
            this.con = con;
        }
        public Boolean call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Boolean.valueOf(con.getAutoCommit());
        }
    }
    
    public static final class GetCatalog implements Callable<String> {
        private final Connection con;
        public GetCatalog(final Connection con) {
            this.con = con;
        }
        public String call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.getCatalog();
        }
    }
    
    public static final class GetTransactionIsolation implements Callable<Integer> {
        private final Connection con;
        public GetTransactionIsolation(final Connection con) {
            this.con = con;
        }
        public Integer call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(con.getTransactionIsolation());
        }
    }
    
    public static final class GetWarnings implements Callable<SQLWarning> {
        private final Connection con;
        public GetWarnings(final Connection con) {
            this.con = con;
        }
        public SQLWarning call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.getWarnings();
        }
    }
    
    public static final class IsReadOnly implements Callable<Boolean> {
        private final Connection con;
        public IsReadOnly(final Connection con) {
            this.con = con;
        }
        public Boolean call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.isReadOnly();
        }
    }
    
    public static final class CreateStatement implements Callable<Statement> {
        private final Connection con;
        public CreateStatement(final Connection con) {
            this.con = con;
        }
        public Statement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.createStatement();
        }
    }
    
    public static final class PrepareCall implements Callable<CallableStatement> {
        private final Connection con;
        private final String sql;
        public PrepareCall(final Connection con, final String sql) {
            this.con = con;
            this.sql = sql;
        }
        public CallableStatement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.prepareCall(sql);
        }
    }
    
    public static final class PrepareStatement implements Callable<PreparedStatement> {
        private final Connection con;
        private final String sql;
        public PrepareStatement(final Connection con, final String sql) {
            this.con = con;
            this.sql = sql;
        }
        public PreparedStatement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.prepareStatement(sql);
        }
    }
    
    public static final class PrepareStatement_IntInt implements Callable<PreparedStatement> {
        private final Connection con;
        private final String sql;
        private final int resultSetType;
        private final int resultSetConcurrency;
        public PrepareStatement_IntInt(final Connection con, final String sql, final int resultSetType, final int resultSetConcurrency) {
            this.con = con;
            this.sql = sql;
            this.resultSetType = resultSetType;
            this.resultSetConcurrency = resultSetConcurrency;
        }
        public PreparedStatement call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.prepareStatement(sql, resultSetType, resultSetConcurrency);
        }
    }
    
    public static final class GetMetaData implements Callable<DatabaseMetaData> {
        private final Connection con;
        public GetMetaData(final Connection con) {
            this.con = con;
        }
        public DatabaseMetaData call() throws Exception {
            if (con == null) {
                throw new ForestTaskNotExecutedException();
            }
            return con.getMetaData();
        }
    }
    
}
