package org.postgresforest.tool.cli;

import org.postgresforest.tool.cli.action.BrokenLogActions;
import org.postgresforest.tool.cli.action.ConfigActions;
import org.postgresforest.tool.cli.action.DbActions;
import org.postgresforest.tool.cli.action.GscActions;
import org.postgresforest.tool.cli.action.HashActions;
import org.postgresforest.tool.cli.action.InstanceActions;
import org.postgresforest.tool.lib.ForestToolException;
import org.postgresforest.tool.lib.GSCdata;
import org.postgresforest.tool.lib.GT;
import org.postgresforest.tool.Logger;
import org.postgresforest.tool.util.CommandParser;
import org.postgresforest.tool.util.MessagesCommandLine;

/**
 * コマンドラインクラス コマンドの受けつけ、割り振りを行う。
 */
public class GscDbCommandProcessor extends AbstractCommandProcessor {

	private String m_dbName;

	public GscDbCommandProcessor(GSCdata gsc) {
		super(gsc);
		m_dbName = gsc.getGscName();
	}


	public void printPrompt() {
		System.out.println();
		System.out.print("[" + m_dbName + "]$ "); //$NON-NLS-1$ //$NON-NLS-2$
		System.out.flush();

	}

	/**
	 * 
	 * コマンドの実行 CommandParserによるパース結果に基づいて、各ロジックに処理を割り振る。
	 * 
	 * @return void
	 * @param command
	 *            コマンド
	 */
	protected AbstractCommandProcessor executeCommand(CommandParser cp, String command) {

		// 環境構築コマンドタイプの取得
		int cpType = cp.getType();

		switch (cpType) {

		case CommandParser.NONE: // 解析不可
			// 未対応のコマンド
			Logger.error( GT.tr("コマンドまたは構文が違います。") ); //$NON-NLS-1$
			break;

		case CommandParser.CREATE_INSTANCE:
			InstanceActions.create(cp, m_gsc);
			break;

		case CommandParser.DROP_INSTANCE:
			InstanceActions.drop(cp, m_gsc);
			break;

		case CommandParser.SHOW_INSTANCE:
			InstanceActions.show(m_gsc);
			break;

		case CommandParser.SET_INSTANCE_UP:
			InstanceActions.set(cp, m_gsc, InstanceActions.SERVER_RUNNING);
			break;

		case CommandParser.SET_INSTANCE_DOWN:
			InstanceActions.set(cp, m_gsc, InstanceActions.SERVER_TROUBLE);
			break;

		case CommandParser.CREATE_GSC:
			GscActions.create(cp, m_gsc);
			break;

		case CommandParser.DROP_GSC:
			GscActions.drop(cp, m_gsc);
			break;

		case CommandParser.SHOW_GSC:
			GscActions.show(m_gsc);
			break;

		// case CommandParser.DUMP_GSC :
		// System.out.println(MessagesCommandLine.getString("cui.message.gsc.dump")
		// ); //$NON-NLS-1$
		// break;
		//	
		// case CommandParser.RESTORE_GSC :
		// System.out.println(MessagesCommandLine.getString("cui.message.gsc.restore"))
		// ; //$NON-NLS-1$
		// break;
		//	
		case CommandParser.CHECK_GSC :
			GscActions.check(m_gsc);
			break;

		case CommandParser.CREATE_DB:
			DbActions.create(cp, m_gsc);
			break;

		case CommandParser.DROP_DB:
			DbActions.drop(cp, m_gsc);
			break;

		case CommandParser.RESTORE_DB:
			DbActions.restore(cp, m_gsc);
			break;

		case CommandParser.SHOW_DB:
			DbActions.show(m_gsc);
			break;

		case CommandParser.CREATE_DBINSTANCE:
			DbActions.createDbInstance(cp, m_gsc);
			break;

		case CommandParser.DROP_DBINSTANCE:
			DbActions.dropDbInstance(cp, m_gsc);
			break;

		case CommandParser.CREATE_CONFIG:
			ConfigActions.create(cp, m_gsc);
			break;

		case CommandParser.DROP_CONFIG:
			ConfigActions.drop(cp, m_gsc);
			break;

		case CommandParser.SHOW_CONFIG:
			ConfigActions.show(m_gsc);
			break;

		case CommandParser.SET_CONFIG:
			ConfigActions.set(cp, m_gsc);
			break;

		case CommandParser.REPLICATE_CONFIG:
			ConfigActions.replicate(cp, m_gsc);
			break;

		case CommandParser.CREATE_HASH:
			HashActions.create(cp, m_gsc);
			break;

		case CommandParser.DROP_HASH:
			HashActions.drop(cp, m_gsc);
			break;

		case CommandParser.SHOW_HASH:
			HashActions.show(m_gsc);
			break;

		case CommandParser.SHOW_BROKENLOG:
			BrokenLogActions.show(m_gsc);
			break;

		case CommandParser.CLEAR_BROKENLOG:
			BrokenLogActions.clear(m_gsc);
			break;

		case CommandParser.OPEN_DB:

			// UserDbコマンド処理クラスを生成して戻り値にセット

			String selectDb = cp.getDB();
			if(selectDb == null || selectDb.equals("")){
				// DB名が指定されていない場合
				Logger.error( GT.tr("データベース名が入力されていません。") );
				break;
			}

			DbCommandProcessor dbCommandProcessor = null;
			try {
				/*
				 * 指定された名前のDBが存在するか確認
				 */
				if ( m_gsc.getDatabase(selectDb)==null )
				{
					Logger.error( GT.tr("データベース \"{0}\" が見つかりません。",
										selectDb) );
					break;
				}

				dbCommandProcessor = new DbCommandProcessor(m_gsc, this, selectDb);
			} catch (ForestToolException e) {
				Logger.error( GT.tr("データベース \"{0}\" をオープンできません。",
									selectDb) );

				Logger.detail(e.getDetail().getMessage());
				Logger.trace(e.getDetail());
				break;
			}

			Logger.notice( GT.tr("データベース \"{0}\" をオープンしました。",
								 selectDb) );

			// ユーザDB処理クラスを戻り地にセット
			return dbCommandProcessor;

		default:
			Logger.error( GT.tr("データベースがオープンされていません。") );
			break;
		}

		// 自インスタンスを戻り地にセット
		return this;
	}

	protected void processSlashCommandTable(String line) {
		Logger.println(MessagesCommandLine.getString("cui.message.db.notopen"));
	}

	protected void processSlashCommandPriority(String line) {
		Logger.println(MessagesCommandLine.getString("cui.message.db.notopen"));
	}

}
