#!/bin/sh -efu
#
# Copyright (C) 2006 Securedog
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
# $Id: pkgdb_fix.sh,v 1.5 2006/10/31 08:52:02 securedog Exp $

usage() {
	echo "usage: ${0##*/} [-habBfFiQrRv] [-s /OLD/NEW/] pkgname[=origin] ..."
	exit ${1:-0}
}

is_yes() {
	case ${1-} in
	[Yy][Ee][Ss]|[Yy]|[Tt][Rr][Uu][Ee])
		return 0 ;;
	*)	return 1 ;;
	esac
}

warn() {
	echo "** $@" >&2
}

prompt_yesno() {
	local prompt default input

	prompt=${1-"OK?"}
	default=${2-"yes"}

	echo -n "${prompt} [${default}] " >&2
	read input

	is_yes ${input:-"${default}"}
}

init_variables() {
	: ${PKG_DBDIR="/var/db/pkg"}
	: ${TMPDIR="/tmp"}
	REGENERATE_REQUIRED_BY_COOKIE=.regenerate.required_by
	export PKG_DBDIR
}

init_options() {
	opt_all=NO
	opt_update_reqby=YES
	opt_force=NO
	opt_fix=NO
	opt_fix_duporigin=NO
	opt_interactive=NO
	opt_quiet=NO
	opt_substitute=
	opt_verbose=NO
	opt_depends=NO
	opt_required_by=NO
	opt_replace=
	opt_origin=
}

parse_options() {
	local OPTS OPT OPTARG OPTIND

	while getopts habBfFhiQrRs:v OPT; do
		case ${OPT} in
		a)	opt_all=YES ;;
		b)	opt_update_reqby=ALWAYS ;;
		B)	opt_update_reqby=NO ;;
		f)	opt_force=YES ;;
		F)	opt_fix=YES ;;
		h)	usage ;;
		i)	opt_interactive=YES ;;
		Q)	opt_quiet=YES ;;
		r)	opt_required_by=YES ;;
		R)	opt_depends=YES ;;
		s)	opt_substitute="${opt_substitute} ${OPTARG}" ;;
		v)	opt_verbose=YES ;;
		\?)	usage 1 ;;
		esac
	done
	OPTC=$((${OPTIND}-1))
}

is_installed() {
#ifdef WITH_PKGSRC
	if pkg_info -qe "$1"; then
#else
	if [ -e "${PKG_DBDIR}/$1/+CONTENTS" ]; then
#endif
		return 0
	else
		return 1
	fi
}

pkg_glob() {
	local - pattern name contents

	set +f
	while [ $# -gt 0 ]; do
		pattern=${1#${PKG_DBDIR}/}

		case ${pattern} in
		''|*[\<\>/]*)	shift; continue ;;
		*\**|*-pl[0-9]*|*-[0-9]*[0-9.][a-z]|*-[0-9]*[0-9]) ;;
		*)		pattern="${pattern}-[0-9]*[0-9a-z]" ;;
		esac

		for contents in ${PKG_DBDIR}/${pattern}/+CONTENTS; do
			if [ -e "${contents}" ]; then
				name=${contents%/+CONTENTS}

				if is_yes ${opt_depends}; then
					pkg_depends "${name##*/}"
				fi

				echo "${name##*/}"

				if is_yes ${opt_required_by}; then
					pkg_required_by "${name##*/}"
				fi
			else
				warn "No such installed package: $1"
			fi
		done
		shift
	done
}

pkg_depends() {
	local dep xvar

	case ${2-} in
	\$*)	xvar=${2#$}; eval ${xvar}= ;;
	*)	xvar= ;;
	esac

#ifdef WITH_OPENBSD
	if [ -r "${PKG_DBDIR}/$1/+REQUIRING" ]; then
		while read dep; do
			if [ -n "${xvar}" ]; then
				eval ${xvar}=\"\$${xvar} ${dep}\"
			else
				echo "${dep}"
			fi
		done < "${PKG_DBDIR}/$1/+REQUIRING"
	fi
#else
	if [ -r "${PKG_DBDIR}/$1/+CONTENTS" ]; then
		while read dep; do
			case ${dep} in
#ifdef WITH_PKGSRC
			@pkgdep\ *)
				if [ -n "${xvar}" ]; then
					eval ${xvar}=\"\$${xvar} $(pkg_info -e "${dep#@pkgdep }")\"
				else
					pkg_info -e "${dep#@pkgdep }"
				fi ;;
#else
			@pkgdep\ *)
				if [ -n "${xvar}" ]; then
					eval ${xvar}=\"\$${xvar} ${dep#@pkgdep }\"
				else
					echo "${dep#@pkgdep }"
				fi ;;
#endif
			[!@]*)	break ;;
			esac
		done < "${PKG_DBDIR}/$1/+CONTENTS"
	fi
#endif
}

pkg_required_by() {
	local req xvar
	
	case ${2-} in
	\$*)	xvar=${2#$}; eval ${xvar}= ;;
	*)	xvar= ;;
	esac

	if [ -r "${PKG_DBDIR}/$1/+REQUIRED_BY" ]; then
		while read req; do
			if [ -n "${xvar}" ]; then
				eval ${xvar}=\"\$${xvar} ${req}\"
			else
				echo "${req}"
			fi
		done < "${PKG_DBDIR}/$1/+REQUIRED_BY"
	fi
}

set_pkg_name() {
	pkg_name=$1
}

set_pkg_pkgdir() {
	pkg_pkgdir=${PKG_DBDIR}/$1
}

set_pkg_origin() {
	local LINE

	pkg_origin=
#ifdef WITH_PKGSRC
	if [ -r "${PKG_DBDIR}/$1/+BUILD_INFO" ]; then
		while read LINE; do
			case ${LINE} in
			PKGPATH=*)
				pkg_origin=${LINE#PKGPATH=}
				break ;;
			esac
		done < "${PKG_DBDIR}/$1/+BUILD_INFO"
	fi
#else
	if [ -r "${PKG_DBDIR}/$1/+CONTENTS" ]; then
		while read LINE; do
			case ${LINE} in
#ifdef WITH_OPENBSD
			@comment*\ subdir=*)
				LINE=${LINE##*subdir=}
				pkg_origin=${LINE%% *}
				break ;;
#else
			@comment\ ORIGIN:*)
				pkg_origin=${LINE#@comment ORIGIN:}
				break ;;
#endif
			[!@]*)	break ;;
			esac
		done < "${PKG_DBDIR}/$1/+CONTENTS"
	fi
#endif
}

create_pkg_copy() {
	local source

	source=${PKG_DBDIR}/$1/$2
	created=${WORKDIR}/$1$2

	if [ -e "${source}" ]; then
		cp "${source}" "${created}" || return 1
	else
		: > "${created}" || return 1
	fi
}

set_pkg_contents() {
	local created

	create_pkg_copy "$1" "+CONTENTS" || return 1
	pkg_contents=${created}
}

#ifdef WITH_PKGSRC
set_pkg_build_info() {
	local created

	create_pkg_copy "$1" "+BUILD_INFO" || return 1
	pkg_build_info=${created}
}
#endif

pkg_preprocess() {
	local i

	set_pkg_name "$1"
	set_pkg_pkgdir "$1"
	set_pkg_origin "$1" || return 1
	set_pkg_contents "$1" || return 1

	if [ -z "${pkg_origin}" ]; then
		warn "'${pkg_name}' has no origin recorded."
		echo "Skipped."
		return 1
	elif [ ! -e "${pkg_pkgdir}/+CONTENTS" ]; then
		warn "'${pkg_name}' is not installed. (broken dependencies?)"
		return 1
	fi

	opt_origin=
	for i in ${opt_replace}; do
		case "|${pkg_name}|${pkg_name%-*}|" in
		*\|${i%=*}\|*)
			opt_origin=${i##*=}
			break ;;
		esac
	done
}


run_pkgdb_fix() {
	local -	pkg_name pkg_pkgdir pkg_origin pkg_contents pkg_build_info \
		_subst _subst_old _subst_new

	pkg_preprocess "$1" || return 0
	is_yes ${opt_verbose} && echo "Checking for ${pkg_name}."

	if [ -z "${opt_origin}${opt_substitute}" ]; then
		pkgdb_fix_dependencies || return 0

		if is_yes ${opt_update_reqby} && \
			[ ! -e "${pkg_pkgdir}/+REQUIRED_BY" ] || \
			[ ${opt_update_reqby} = ALWAYS ]; then
			echo "${pkg_name}" >> \
			"${WORKDIR}/${REGENERATE_REQUIRED_BY_COOKIE}"
		fi
	else
		if [ -n "${opt_origin}" ]; then
			echo "${pkg_name}: ${pkg_origin} -> ${opt_origin}"
			modify_origin "${opt_origin}"
		fi

		for _subst in ${opt_substitute}; do
			_subst=${_subst#/}
			_subst=${_subst%/}
			_subst_old=${_subst%/*}
			_subst_new=${_subst#*/}

			if [ -n "${_subst_new}" ]; then
				modify_dep "${_subst_old}" "${_subst_new}"
			else
				delete_dep "${_subst_old}"
			fi
		done
	fi
}

run_pkgdb_fix_duporigin() {
	local pkg_origin

	is_yes ${opt_verbose} && echo "Checking for origin duplicates."

#ifdef WITH_PKGSRC
	for pkg_origin in $(pkg_info -aQ PKGPATH | sort | uniq -d); do
#else
#ifdef WITH_OPENBSD
	while false; do
#endif
	for pkg_origin in $(pkg_info -aqo | sort | uniq -d); do
#endif
		pkgdb_fix_duporigin "${pkg_origin}"
	done
}

replace() {
	sed "$1" "$2" > "$2.new"
	mv -f "$2.new" "$2"
}

modify_origin() {
        local new_origin

        new_origin=$1

#ifdef WITH_PKGSRC
	set_pkg_build_info "${pkg_name}"

	if grep -q "^PKGPATH=" ${pkg_build_info}; then
		replace "s|^PKGPATH=.*$|PKGPATH=${new_origin}|" "${pkg_build_info}"
	else
		echo "PKGPATH=${new_origin}" >> "${pkg_build_info}"
	fi
#else
#ifdef WITH_OPENBSD
	replace "s| subdir=[^[:space:]]*| subdir=${new_origin}|" "${pkg_contents}"
#else
	if grep -q "^@comment ORIGIN:" ${pkg_contents}; then
		replace "s|^@comment ORIGIN:.*$|@comment ORIGIN:${new_origin}|" "${pkg_contents}"
	else
		replace "/^@name /a\\
@comment ORIGIN:${new_origin}
" "${pkg_contents}"
	fi
#endif
#endif
}

delete_dep() {
	local deppkg

	deppkg=$1
#ifdef WITH_PKGSRC
	replace "/^@blddep /,/^@pkgdep ${deppkg}$/d" "${pkg_contents}"
#else
#ifdef WITH_OPENBSD
	return 1
#else
	replace "/^@pkgdep ${deppkg}$/,/^@comment DEPORIGIN:/d" "${pkg_contents}"
#endif
#endif
}

modify_dep() {
	local oldpkg newpkg neworigin

	oldpkg=$1
	newpkg=$2
	neworigin=${3-}

#ifdef WITH_PKGSRC
	replace "s|^@pkgdep ${oldpkg}$|@pkgdep ${newpkg}|g" "${pkg_contents}"
#else
#ifdef WITH_OPENBSD
	return 1
#else
	if [ -n "${neworigin}" ]; then
		replace "/^@pkgdep ${oldpkg}$/,/^@comment DEPORIGIN:/c\\
@pkgdep ${newpkg}\\
@comment DEPORIGIN:${neworigin}
" "${pkg_contents}"
	else
		replace "s|^@pkgdep ${oldpkg}$|@pkgdep ${newpkg}|g" "${pkg_contents}"
	fi
#endif
#endif
}

fix_dep() {
#ifndef WITH_PKGSRC
	local pkg_origin

	if [ ${3:+fix_origin} ]; then
		set_pkg_origin "$2"
		modify_dep "$1" "$2" "${pkg_origin}"
		echo "Fixed. (-> $2, ${pkg_origin})"
	else
#endif
		modify_dep "$1" "$2"
		echo "Fixed. (-> $2)"
#ifndef WITH_PKGSRC
	fi
#endif
}

fix_multiple_dep() {
	local oldpkg pkg_origin

	oldpkg=$1; shift
	while [ $# -gt 0 ]; do
		set_pkg_origin "$1"
		if prompt_yesno "Use $1 (${pkg_origin})?"; then
			fix_dep "${oldpkg}" "$1" 1
			return 0
		fi
		shift
	done

	return 1
}

try_fix_dep() {
	local - input pkg found

	set +f
	while :; do
		echo -n "New dependency? " >&2
		read input

		case ${input} in
		''|[Nn][Oo]|[Nn])
			if prompt_yesno "Delete this?" "no"; then
				delete_dep "$1"
				echo "Deleted."
				break
			elif prompt_yesno "Skip this?" "yes"; then
				break
			fi ;;
		*)
			if is_installed "${input}"; then
				fix_dep "$1" "${input}" 1
				break
			fi

			found=
			for pkg in ${PKG_DBDIR}/${input%-*}*; do
				case ${pkg} in
				*\**) ;;
				*) found="${found} ${pkg##*/}" ;;
				esac
			done

			if [ -n "${found}" ]; then
				echo "Please choose one of these:" >&2
				echo "${found}" >&2
			else
				echo "'${input}' is not installed." >&2
			fi
		esac
	done
}

pkgdb_fix_duporigin() {
	local - pkg pkg_delete origin

	origin=$1
#ifdef WITH_PKGSRC
	cd "${PKG_DBDIR}" || return 1
	set +f -- ''

	for pkg in $(echo */+BUILD_INFO |
		xargs grep -lx "PKGPATH=${origin}"); do
		set -- ${1+"$@"} ${pkg%/+BUILD_INFO}
	done
#else
	set -- $(pkg_info -qO "${origin}")
#endif
	pkg_delete=

	if [ $# -gt 1 ]; then
		echo "Duplicated origin: ${origin} - $@" >&2
		warn "Please remove unnecessary packages."

		while [ $# -gt 1 ]; do
			if prompt_yesno "Delete $1?" "no"; then
				pkg_delete="${pkg_delete} $1"
			else
				set -- "$@" "$1"
			fi
			shift
		done

		for pkg in ${pkg_delete}; do
			echo "Unregistering ${pkg}." >&2
			rm -rf "${PKG_DBDIR}/${pkg}"
		done
	fi
}

pkgdb_fix_dependencies() {
	local IFS LINE dep_pkg_name dep_pkg_origin pkg_origin

	IFS='
'
#ifdef WITH_PKGSRC
	for LINE in $(grep -e '^@pkgdep' \
		"${PKG_DBDIR}/${pkg_name}/+CONTENTS"); do
		dep_pkg_name=${LINE#@pkgdep }
#else
	for LINE in $(grep -e '^@pkgdep ' -e '^@comment DEPORIGIN:' \
		"${PKG_DBDIR}/${pkg_name}/+CONTENTS"); do
		case ${LINE} in
		@pkgdep\ *)
			dep_pkg_name=${LINE#@pkgdep }
			continue ;;
		@comment\ DEPORIGIN:*)
			dep_pkg_origin=${LINE#@comment DEPORIGIN:} ;;
		esac
#endif

		is_installed "${dep_pkg_name}" && continue

		echo "Stale dependency: ${pkg_name} -> ${dep_pkg_name}:"

#ifdef WITH_PKGSRC
		set -- $(pkg_info -e "${dep_pkg_name%[-<>]*}*")
#else
		set -- $(pkg_info -qO "${dep_pkg_origin}")
#endif

		if [ $# -eq 1 ]; then
			fix_dep "${dep_pkg_name}" "$1"
			continue
#ifndef WITH_PKGSRC
		elif [ $# -gt 1 ]; then
			warn "Detected the origin duplicates," \
			"please run '${0##*/} -aF' to fix."
			return 1
		fi

		set -- $(pkg_info -E "${dep_pkg_name%-*}*" "??-${dep_pkg_name%-*}*")

		if [ $# -eq 1 ]; then
			fix_dep "${dep_pkg_name}" "$1" 1
			continue
#endif
		elif [ $# -gt 1 ] && is_yes ${opt_fix}; then
			echo "Detected the following packages:" >&2
			echo "$@" >&2

			if fix_multiple_dep "${dep_pkg_name}" "$@"; then
				continue
			fi
		fi

		if is_yes ${opt_fix}; then
			try_fix_dep "${dep_pkg_name}"
		else
			echo "Skipped. (specify -F to fix)"
		fi
	done
}

regenerate_required_by() {
	local - pkg_name req

	[ -e "${WORKDIR}/${REGENERATE_REQUIRED_BY_COOKIE}" ] || return 0

	echo "Regenerating +REQUIRED_BY files."
	cd "${PKG_DBDIR}" || return 1

	set +f
	while read pkg_name; do
		echo */+CONTENTS |
#ifdef WITH_PKGSRC
		xargs grep -l "^@pkgdep ${pkg_name%-*}[<>-]" |
#else
		xargs grep -lx "@pkgdep ${pkg_name}" |
#endif
		while read req; do
			echo "${req%/+CONTENTS}"
		done > "${WORKDIR}/${pkg_name}+REQUIRED_BY"
	done < "${WORKDIR}/${REGENERATE_REQUIRED_BY_COOKIE}"

	rm -f "${WORKDIR}/${REGENERATE_REQUIRED_BY_COOKIE}"
}

pkgdb_fix_update() {
	local - source target file

	set +f
	for source in ${WORKDIR}/*+*; do
		case ${source} in
		*\**) break ;;
		esac

		file=${source##*/}
		target="${PKG_DBDIR}/${file%+*}/+${file##*+}"

		if confirm_install "${source}" "${target}"; then
			cp -f "${source}" "${target}" || :
		fi
	done

	rm -f ${WORKDIR}/*+* || :
}

confirm_install() {
	local old_file new_file

	new_file=$1
	old_file=$2

	if [ ! -s "${new_file}" ]; then
		return 1
	elif is_yes ${opt_interactive}; then
		if [ -e "${old_file}" ]; then
			diff -c "${old_file}" "${new_file}" && return 1
		else
			echo "${old_file}:"
			cat "${new_file}"
		fi
		prompt_yesno "Install?" || return 1
	else
		cmp -s "${old_file}" "${new_file}" && return 1
	fi

	return 0
}

pkgdb_fix_clean() {
	rm -rf "${WORKDIR}"
}

pkgdb_fix_done() {
	pkgdb_fix_update
	regenerate_required_by
	pkgdb_fix_update
	pkgdb_fix_clean

	echo "Done."
}

check_permission() {
	if [ ! -w "$1" ]; then
		if is_yes ${opt_force}; then
			warn "You do not own $1. (proceeding anyway)"
		else
			warn "You do not own $1. (specify -f to force)"
			return 1
		fi
	fi
}

create_working_dir() {
	WORKDIR=$(mktemp -dt .pkgdb) || return 1
	trap 'pkgdb_fix_clean; exit' 1 2 3 15
}

parse_args() {
	do_fix=
	while [ $# -gt 0 ]; do
		case $1 in
		?*=?*)
			opt_replace="${opt_replace} ${1}"
			do_fix="${do_fix} ${1%=*}" ;;
		*)
			do_fix="${do_fix} $1" ;;
		esac
		shift
	done
}

main() {
	init_variables
	init_options

	parse_options ${1+"$@"}
	shift ${OPTC}

	is_yes ${opt_quiet} && exec >/dev/null

	if is_yes ${opt_all}; then
		opt_depends=NO
		opt_required_by=NO
		is_yes ${opt_fix} && opt_fix_duporigin=YES
		set -- '*'
	fi

	[ $# -gt 0 ] || usage

	check_permission "${PKG_DBDIR}" || exit 1

	for do_fix in ${opt_substitute}; do
		case ${do_fix} in
		*/*/*/*/*)
			warn "Illegal expression: ${do_fix}"
			usage 1 ;;
		/?*//)
			do_fix=${do_fix#/}
			echo "Deleting dependency: ${do_fix%%/*}" ;;
		/?*/*/)
			do_fix=${do_fix#/}
			do_fix=${do_fix%/}
			echo "Replacing dependency: ${do_fix%/*} -> ${do_fix#*/}" ;;
		*)
			warn "Illegal expression: ${do_fix}"
			usage 1 ;;
		esac
	done

	create_working_dir || exit 1

	if is_yes ${opt_fix_duporigin}; then
		run_pkgdb_fix_duporigin
	fi

	parse_args ${1+"$@"}
	set -- $(pkg_glob ${do_fix})

	while [ $# -gt 0 ]; do
		run_pkgdb_fix "$1"
		shift
	done

	pkgdb_fix_done
}

IFS=' 
'
main ${1+"$@"}
